/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.carousel.ui;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarousel;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselClamp;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselSocket;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByFilter;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.ui.commons.CanOpenDevicePanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.EPOSControllerPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.GatherPanel;

/**
 * This panel gathers all the panels needed for the carousel GUI. 
 * Except the subsystemPanel which is handled by the GeneralPanel.
 * 
 * It takes care of initialization and update of the panels.
 *
 * @author virieux
 */
public class CarouselGatherPanel extends GatherPanel {
    private static final long serialVersionUID = -975949078768889254L;
    
    private final String[] socketNames = {"socket1","socket2","socket3","socket4","socket5"};
    private final CarouselGeneralViewPanel caGeneralViewPanel = new CarouselGeneralViewPanel();
    protected transient final Map<String,CarouselSocketPanel> socketPanels = new TreeMap<>();

    protected CarouselGUI subs;

    //CANopen devices panels
    private EPOSControllerPanel clampXminusControllerPanel;
    private EPOSControllerPanel clampXplusControllerPanel;
    private EPOSControllerPanel carouselControllerPanel;
    private CanOpenDevicePanel hyTTCPanel;
    private CanOpenDevicePanel cancbxAI420_1Panel;
    private CanOpenDevicePanel cancbxAI420_2Panel;

    
    //Filters panels
    private FilterListPanel filterListPanel;

    /**
     * A Runnable class to initialize the GUI with data coming from the filter names list.
     */
    private class GuiInitializationFilterListPanel implements Runnable {
        private final ConfigurationInfo configInfo;
        private final List<String> filterNamesList;

        public GuiInitializationFilterListPanel(ConfigurationInfo configInfo, List<String> filterNamesList) {
            this.configInfo = configInfo;
            /*initialize this.filterNamesList from a copy of the List<String> argument.*/
            this.filterNamesList = new ArrayList<>();
            this.filterNamesList.addAll(filterNamesList);
        }

        @Override
        public void run() {
            /** Filter List Panel */         
            filterListPanel.initializeGui(configInfo,filterNamesList);
        }
        
    }

    /**
     * Initialization of the GUI for data stored in tcpConfiguration. *
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        /**
         * Create a new GuiInitialization from a CarouselConfig and a ConfigurationInfo.
         * The CarouselConfig should be suppressed and initialization of the GUI done from ConfigurationInfo only.
         * @param configInfo 
         */
        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo =configInfo;
        }

        @Override
        public void run() {
            /**
             * ClampXminus Controller Panel *
             */
            clampXminusControllerPanel.initializeGui(configInfo);
            
            /**
             * ClampXplus Controller Panel *
             */
            clampXplusControllerPanel.initializeGui(configInfo);
           
            /**
             * Carousel Controller Panel *
             */
            carouselControllerPanel.initializeGui(configInfo);
            
            /**
             * CANOpen HY-TTC Panel*
             */
            hyTTCPanel.initializeGui(configInfo);
            
            /**
             * CANOpen CAN-CBX-4I420_1 Panel*
             */
            cancbxAI420_1Panel.initializeGui(configInfo);
            
            /**
             * CANOpen CAN-CBX-4I420_2 Panel*
             */
            cancbxAI420_2Panel.initializeGui(configInfo);

            //** Carousel General View Panel **/
            caGeneralViewPanel.initializeGui(configInfo);
            
            /**
             * Carousel Sockets Panel *
             */
            socketPanels.entrySet().stream().forEach((entry) -> {
                entry.getValue().initializeGui(configInfo);
            });
            
        }
    }


    public CarouselGatherPanel(String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
    }

    public CarouselGatherPanel(CarouselGUI subs, String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
        this.subs = subs;
        this.clampXminusControllerPanel.setSubsystem(subs);
        this.clampXplusControllerPanel.setSubsystem(subs);
        this.carouselControllerPanel.setSubsystem(subs);
        this.caGeneralViewPanel.setSubsystem(subs);
        socketPanels.entrySet().stream().forEach((entry) -> {
            entry.getValue().setSubsystem(subs);
        });
    }
    
    

    public EPOSControllerPanel getClampXminusControllerPanel() {
        return clampXminusControllerPanel;
    }

    public EPOSControllerPanel getClampXplusControllerPanel() {
        return clampXplusControllerPanel;
    }

    public EPOSControllerPanel getCarouselControllerPanel() {
        return carouselControllerPanel;
    }

    public CanOpenDevicePanel getHyTTCPanel() {
        return hyTTCPanel;
    }    

    public CanOpenDevicePanel getCancbxAI420_1Panel() {
        return cancbxAI420_1Panel;
    }

    public CanOpenDevicePanel getCancbxAI420_2Panel() {
        return cancbxAI420_2Panel;
    }

    public CarouselGeneralViewPanel getCaGeneralViewPanel() {
        return caGeneralViewPanel;
    }

    public FilterListPanel getFilterListPanel() {
        return filterListPanel;
    }
    

    @Override
    public void onStatusMessage(StatusMessage msg) {
         // Safe cast : filtering ensures we only receive StatusSubsystemData messages
        StatusSubsystemData d = (StatusSubsystemData)msg;
        KeyValueData data = (KeyValueData)d.getObject();
        
        if (data.getValue() instanceof StatusDataPublishedByHardware) {
            updateCanOpenHardwarePanel((StatusDataPublishedByHardware) data.getValue());
            
        } else if(data.getValue() instanceof StatusDataPublishedByEPOSController) {
            updateControllerPanel((String)data.getKey(),(StatusDataPublishedByEPOSController) data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByFilter) {
            updateFilterListPanel((String)data.getKey(),(StatusDataPublishedByFilter) data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByCarouselClamp) {
            updateSocketPanels((StatusDataPublishedByCarouselClamp) data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByCarouselSocket) {
            updateSocketPanels((String)data.getKey(),(StatusDataPublishedByCarouselSocket) data.getValue());    
            
        } else if (data.getValue() instanceof StatusDataPublishedByCarousel) {
            FCSLOG.debug("STATUS data coming from carousel.");
            caGeneralViewPanel.update((StatusDataPublishedByCarousel) data.getValue());
            
        } else {
            FCSLOG.warning(getName() + " An unexpected dataType was read on the status bus:"
                                + data.getKey());
        }
        
//                switch (data.getKey()) {
////                    case "tcpProxy":
////                        updateCanOpenHardwarePanel((StatusDataPublishedByHardware) data.getValue());
////                        break;
    }

    @Override
    public void initializeGUIWithConfigInfo(ConfigurationInfo configInfo, List<String> hardwareNames) 
            throws InterruptedException, InvocationTargetException {
        super.initializeGUIWithConfigInfo(configInfo, hardwareNames);
        SwingUtilities.invokeAndWait(new GuiInitialization(configInfo));
    }
    
    /**
     * 
     * @param configInfo 
     * @param filterNamesList 
     * @throws java.lang.InterruptedException 
     * @throws java.lang.reflect.InvocationTargetException 
     */
    @Override
    public void initializeGUIWithFiltersInfo(ConfigurationInfo configInfo, List<String> filterNamesList) 
            throws InterruptedException, InvocationTargetException {
        SwingUtilities.invokeAndWait(new GuiInitializationFilterListPanel(configInfo,filterNamesList));
    }

    

    /**
     * Update the controller Panels from data coming from STATUS bus.
     * @param controllerName
     * @param s 
     */
    protected void updateControllerPanel (String controllerName, StatusDataPublishedByEPOSController s) {
        switch (controllerName) {
            case "clampXminusController":
                clampXminusControllerPanel.updateController(s);
                break;
                
            case "clampXplusController":
                clampXplusControllerPanel.updateController(s);
                break;

            case "carouselController":
                carouselControllerPanel.updateController(s);
                break; 
            
            default:
                FCSLOG.error(this.getName() + " :received an unknown controller name on STATUS bus:" 
                        + controllerName);
                assert false;
        }
    }

    

    /**
     * Update the filterListPanel from a StatusDataPublishedByFilter.
     * Update the panel where all the filters are listed from the data published on the Status bus
     * by a particular filter.
     * @param s 
     */
    private void updateFilterListPanel(String filterName, StatusDataPublishedByFilter s) {
        filterListPanel.updateFilterListPanel(filterName, s);
    }
    
    
    /**
     * Update the Carousel Socket Panels from the data published on the Status bus
     * by a clamp. 
     * @param s 
     */
    private void updateSocketPanels(StatusDataPublishedByCarouselClamp s) {
        /**
         * Carousel Sockets Panel *
         */
        String clampName = s.getName();
        
        /*To find the name of the socket needed to find the socket panel to be updated.*/
        char num = clampName.charAt(clampName.length()-1);
        String socketName = "socket" + num;
        
        socketPanels.get(socketName).update(s);
    }
    
    /**
     * Update the Carousel Socket Panels from the data published on the Status bus
     * by a Carousel Socket. 
     * @param s 
     */
    private void updateSocketPanels(String socketName, StatusDataPublishedByCarouselSocket s) {      
        socketPanels.get(socketName).updateSocket(s);
    }



    private void initPanels() {
        //controllers Panels
        clampXminusControllerPanel = new EPOSControllerPanel("clampXminusController");
        clampXplusControllerPanel = new EPOSControllerPanel("clampXplusController");
        carouselControllerPanel = new EPOSControllerPanel("carouselController");
        
        //Clamps Sensors Reader Device
        hyTTCPanel = new CanOpenDevicePanel("clampsSensorsReaderDevice");

        //CANOpen CAN-CBX-AI420
        cancbxAI420_1Panel = new CanOpenDevicePanel("ai420_1");
        cancbxAI420_2Panel = new CanOpenDevicePanel("ai420_2");
        
        //Sockets Panels        
        for (String socketName: socketNames) {
            socketPanels.put(socketName, new CarouselSocketPanel(socketName));
        }

        /** Filter List Panel*/
        filterListPanel = new FilterListPanel();
    }

}
