
package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.HeadlessException;
import java.io.PrintWriter;
import java.util.concurrent.ExecutionException;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.SwingWorker;
import org.freehep.graphicsbase.swing.ErrorDialog;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.Agent;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.BusMessage;
import org.lsst.ccs.bus.DataStatusListener;
import org.lsst.ccs.bus.ModuleInvokerCommand;
import org.lsst.ccs.bus.StatusListener;
import org.lsst.ccs.bus.StateChangeNotification;
import org.lsst.ccs.bus.utils.SynchronousCommandAgent;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This class gathers the common stuff for all FCS GUIs.
 * @author virieux
 */
public abstract class GeneralGUISubsystem extends Subsystem implements DataStatusListener, StatusListener {
    
    protected static final Logger fcslog = Logger.getLogger("org.lsst.ccs.subsystems.fcs");
    protected final String destination;
    protected PrintWriter out = new PrintWriter(System.out, true);
    protected TopPanel topPanel;
    protected boolean initialized = false;
    public JScrollPane pane;
    public long abortTimeout;
    public long updateStateWithSensorsTimeout;
    public long initializationTimeout;
    
     /**
     * To send command which takes time from a JButton.
    */
    public class CommandSwingWorker extends SwingWorker<Object, Object> {
       String cmdDestination;
       String cmdName;
       long cmdTimeout;
       
       public CommandSwingWorker(String cmdN, long timeout) {
           super();
           if (cmdN == null) throw new IllegalArgumentException("Command should not be null in CommandSwingWorker");
           if (timeout == 0) throw new IllegalArgumentException("timeout should not be equal to 0 in CommandSwingWorker");
           this.cmdDestination = destination;
           this.cmdName = cmdN;
           this.cmdTimeout = timeout;
       }
       
       public CommandSwingWorker(String cmdN, long timeout, String moduleName) {
           super();
           if (cmdN == null) throw new IllegalArgumentException("Command should not be null in CommandSwingWorker");
           if (timeout == 0) throw new IllegalArgumentException("timeout should not be equal to 0 in CommandSwingWorker");      
           if (moduleName == null) throw new IllegalArgumentException("moduleName should not be null in this constructor of CommandSwingWorker");          
           this.cmdDestination = destination + "/" + moduleName;
           this.cmdName = cmdN;
           this.cmdTimeout = timeout;
       }
       
       @Override
       public Object doInBackground() {
           log.info("Executing " + cmdName + " on " + cmdDestination); 
           log.info("/timeout=" + cmdTimeout);
           String key = "";
           int level = 0;
           //should be : new ModuleInvokerCommand(key,level,cmdDestination,cmdName) but ambiguous constructor
           return newSendCommand(new ModuleInvokerCommand(cmdName),cmdDestination,cmdTimeout);
       }

       @Override
       protected void done() {
           try {
               Object response = get();
               String strRes;
               if (response instanceof Throwable) {
                    log.error(String.valueOf(response));
//                    JOptionPane.showMessageDialog(null, response.toString(), "ALERT", JOptionPane.ERROR_MESSAGE);
                      log.error(" command returned error",(Throwable) response); 
                        //ErrorDialog.showErrorDialog(null,"Command returned error",(Throwable) response);
                      ErrorDialog.showErrorDialog(null,response.toString(),(Throwable) response);
                      if (!(response instanceof BadCommandException)) raiseAlarm(response.toString());
               
               } else if (response == null) {
                    strRes = "ok : DONE" ;
                    JOptionPane.showMessageDialog(null, strRes, cmdName,  JOptionPane.INFORMATION_MESSAGE);
                
                } else {
                    strRes = String.valueOf(response);
                    JOptionPane.showMessageDialog(null, strRes, cmdName, JOptionPane.INFORMATION_MESSAGE);
                }
           } catch (InterruptedException | ExecutionException | HeadlessException ex) {
                log.error(ex.toString());
                //JOptionPane.showMessageDialog(null, ex.toString(), "ALERT", JOptionPane.ERROR_MESSAGE);
                ErrorDialog.showErrorDialog(null,ex.toString(),(Throwable) ex);
                raiseAlarm(ex.toString());
           }
       }
    }

    public GeneralGUISubsystem(String name, Agent.AgentType type,
            String destination) {
        super(name, type);
        this.destination = destination;
        this.abortTimeout = 1000;
        this.updateStateWithSensorsTimeout = 1000;
        this.initializationTimeout = 2000;
    }



    public void initGui() {
        setListenToStatus(true);
        //setStatusBroadcastPeriod(0);
        start();
    }

    public JComponent getGuiLayout() {
        return pane;
    }

    public void resetGui() {
    }

    public void shutdownMyDestinationSubsystem() {
        System.out.println("Shutting down " + destination);
        sendAsynchronousCommand(new ModuleInvokerCommand("shutdown"));
        //This does not work properly because the abort method return code is void so it
        //always opens a JOPtionPane with ok : DONE
        //new CommandSwingWorker("abort",abortTimeout).execute();
    }

    public void abortMyDestinationSubsystem() {
        System.out.println("Aborting " + destination);
        sendAsynchronousCommand(new ModuleInvokerCommand("abort"));
        //This does not work properly because the abort method return code is void so it
        //always opens a JOPtionPane with ok : DONE
        //new CommandSwingWorker("abort",abortTimeout).execute();
    }

    public void stopMyDestinationSubsystem() {
        System.out.println("Stopping " + destination);
        sendAsynchronousCommand(new ModuleInvokerCommand(destination, "stop", 1000L));
    }

    public void updateStateWithSensors() {
        log.info("Reading sensors");
        log.info("/timeout=" + updateStateWithSensorsTimeout);
        new CommandSwingWorker("updateStateWithSensors", updateStateWithSensorsTimeout).execute();
    }
    
    public void completeInitialization() {
        log.info("Reading sensors");
        log.info("/timeout=" + initializationTimeout);
        new CommandSwingWorker("completeInitialisation", initializationTimeout).execute();
    }

    public void sendCommandSwingWorker(String commandName, long timeout, String moduleName) {
        log.info(getName() + ": sending command= " + commandName + " to Module=" + moduleName);
        new CommandSwingWorker(commandName, timeout, moduleName).execute();
    }

    public void sendCommandSwingWorker(String commandName, long timeout) {
        log.info(getName() + ": sending command= " + commandName);
        new CommandSwingWorker(commandName, timeout).execute();
    }

    public void setPrintWriter(PrintWriter out) {
        this.out = out;
    }



    public void sendAsynchronousCommand(ModuleInvokerCommand cmd) {
        cmd.setDestination(destination);
        fac.sendCommand(cmd);
    }

    public Object newSendCommand(ModuleInvokerCommand cmd, String commandDestination, long timeout) {
        try {
            cmd.setDestination(commandDestination);
            log.debug("fac=" + String.valueOf(fac));
            SynchronousCommandAgent agent = new SynchronousCommandAgent(fac);
            log.debug("agent=" + String.valueOf(agent));
            log.debug("command=" + String.valueOf(cmd));
            Object response = agent.invoke(cmd, timeout);
            //ask Bernard
            String strRes = String.valueOf(response);
            log.debug("strRes=" + strRes);
            if (response == null) {
                return strRes = "ok : DONE";
                //JOptionPane.showMessageDialog(null, strRes, cmd.getCommand(),  JOptionPane.INFORMATION_MESSAGE);
            } else {
                return strRes = String.valueOf(response);
                //JOptionPane.showMessageDialog(null, strRes, cmd.getCommand(), JOptionPane.INFORMATION_MESSAGE);
            }
        } catch (Exception ex) {
            return ex;
        }
    }
    
    //To update the Subsystem state panel.
    @Override
    public void onStatus(BusMessage s) {
        if (!(s instanceof StateChangeNotification)) return;
        if (((StateChangeNotification)s).getOrigin().equals(this.destination)) {
            this.topPanel.getSubsystemPanel().update(((StateChangeNotification)s).getState());
        }
    }
   
    
}
