
package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.HeadlessException;
import java.io.PrintWriter;
import java.util.concurrent.ExecutionException;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.SwingWorker;
import org.freehep.graphicsbase.swing.ErrorDialog;
import org.lsst.ccs.bus.messages.AgentInfo;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandError;
import org.lsst.ccs.bus.messages.CommandNack;

import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.messaging.DataStatusListener;

import org.lsst.ccs.bus.messages.StateChangeNotification;
import org.lsst.ccs.bus.messages.StatusMessage;


import org.lsst.ccs.bus.utils.SynchronousCommandAgent;

import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.messaging.StatusMessageListener;

import org.lsst.ccs.utilities.logging.Logger;


/**
 * This class gathers the common stuff for all FCS GUIs.
 * @author virieux
 */
public abstract class GeneralGUI implements DataStatusListener, StatusMessageListener,
                        AgentPresenceListener {
    
    
    protected static final Logger fcslog = Tools.fcslog;
    protected final String destination;
    protected PrintWriter out = new PrintWriter(System.out, true);
    protected TopPanel topPanel;
    protected boolean initialized = false;
    protected boolean panelStateInitialized = false;
    public JScrollPane pane;
    public long abortTimeout;
    public long updateStateWithSensorsTimeout;

    private String name;
    protected AgentMessagingLayer aml;

    public long initializationTimeout;

    
     /**
     * To send command which takes time from a JButton.
    */
    public class CommandSwingWorker extends SwingWorker<Object, Object> {
       String cmdDestination;
       String cmdName;
       long cmdTimeout;
       
       public CommandSwingWorker(String cmdN, long timeout) {
           super();
           if (cmdN == null) throw new IllegalArgumentException("Command should not be null in CommandSwingWorker");
           if (timeout == 0) throw new IllegalArgumentException("timeout should not be equal to 0 in CommandSwingWorker");
           this.cmdDestination = destination;
           this.cmdName = cmdN;
           this.cmdTimeout = timeout;
       }
       
       public CommandSwingWorker(String cmdN, long timeout, String moduleName) {
           super();
           if (cmdN == null) throw new IllegalArgumentException("Command should not be null in CommandSwingWorker");
           if (timeout == 0) throw new IllegalArgumentException("timeout should not be equal to 0 in CommandSwingWorker");      
           if (moduleName == null) throw new IllegalArgumentException("moduleName should not be null in this constructor of CommandSwingWorker");          
           this.cmdDestination = destination + "/" + moduleName;
           this.cmdName = cmdN;
           this.cmdTimeout = timeout;
       }
       
       @Override
       public Object doInBackground() {
           fcslog.info("Executing " + cmdName + " on " + cmdDestination); 
           fcslog.info("/timeout=" + cmdTimeout);
           return sendCommand(new CommandRequest(cmdDestination,cmdName), cmdTimeout);
       }

       @Override
       protected void done() {
           try {
               Object response = get();
               String strRes;
               if (response instanceof Throwable) {
                    fcslog.error(String.valueOf(response));
//                    JOptionPane.showMessageDialog(null, response.toString(), "ALERT", JOptionPane.ERROR_MESSAGE);
                      fcslog.error(" command returned error",(Throwable) response); 
                        //ErrorDialog.showErrorDialog(null,"Command returned error",(Throwable) response);
                      ErrorDialog.showErrorDialog(null,response.toString(),(Throwable) response);
                      // if (!(response instanceof BadCommandException)) raiseAlarm(response.toString());
               
               } else if (response == null) {
                    strRes = "ok : DONE" ;
                    JOptionPane.showMessageDialog(null, strRes, cmdName,  JOptionPane.INFORMATION_MESSAGE);
                
                } else {
                    strRes = String.valueOf(response);
                    JOptionPane.showMessageDialog(null, strRes, cmdName, JOptionPane.INFORMATION_MESSAGE);
                }
           } catch (InterruptedException | ExecutionException | HeadlessException ex) {
                fcslog.error(ex.toString());
                //JOptionPane.showMessageDialog(null, ex.toString(), "ALERT", JOptionPane.ERROR_MESSAGE);
                ErrorDialog.showErrorDialog(null,ex.toString(),(Throwable) ex);
                // raiseAlarm(ex.toString());
           }
       }
    }

    public GeneralGUI(String name,
            String destination, 
            AgentMessagingLayer aml) {
        this.aml = aml;
        this.name = name;
        this.destination = destination;
        this.abortTimeout = 1000;
        this.updateStateWithSensorsTimeout = 1000;
        this.initializationTimeout = 2000;

    }


    public String getName(){
        return name;
    }
    
    public void initGui() {
        aml.addStatusMessageListener(this);
        fcslog.info(name + ":initGUI");
        
        postInitGui();
    }
    
    /**
     * This method can be called in the initialization methods of the FCS GUI.
     * When the GUI starts after the subsystem, it's not initialized with the 
     * data coming from the sensors.
     * This methods send the updateStateWithSensors command to the subsystem.
     */
    public void postInitGui() {
        if (aml.getAgentPresenceManager().agentExists(destination)) {
            fcslog.warning(destination + " is already connected.");
            sendAsynchronousCommand(new CommandRequest(destination,"updateStateWithSensors"));
        };
        aml.getAgentPresenceManager().addAgentPresenceListener(this);
    }

    public JComponent getGuiLayout() {
        return pane;
    }

    public void resetGui() {
        topPanel.resetPanel();
    }

    public void shutdownMyDestinationSubsystem() {
        System.out.println("Shutting down " + destination);
        //the following doesn't work with Toolkit-V2.0
        //sendAsynchronousCommand(new CommandRequest("shutdown", destination));
        //This does not work properly because the abort method return code is void so it
        //always opens a JOPtionPane with ok : DONE
        new CommandSwingWorker("shutdown",abortTimeout).execute();
    }

    public void abortAction() {
        System.out.println("Aborting " + destination);
        //the following doesn't work with Toolkit-V2.0
        //sendAsynchronousCommand(new CommandRequest("abort", destination));
        //This does not work properly because the abort method return code is void so it
        //always opens a JOPtionPane with ok : DONE
        new CommandSwingWorker("abort",abortTimeout).execute();
    }

    public void stopMyDestinationSubsystem() {
        System.out.println("Stopping " + destination);
        sendAsynchronousCommand(new CommandRequest(destination, "stop", 1000L));
    }

    public void updateStateWithSensors() {
        fcslog.info("Reading sensors");
        fcslog.info("/timeout=" + updateStateWithSensorsTimeout);
        new CommandSwingWorker("updateStateWithSensors", updateStateWithSensorsTimeout).execute();
    }
    
    public void completeInitialization() {
        fcslog.info("Reading sensors");
        fcslog.info("/timeout=" + initializationTimeout);
        new CommandSwingWorker("completeInitialization", initializationTimeout).execute();
    }

    public void sendCommandSwingWorker(String commandName, long timeout, String moduleName) {
        fcslog.info(getName() + ": sending command= " + commandName + " to Module=" + moduleName);
        new CommandSwingWorker(commandName, timeout, moduleName).execute();
    }

    public void sendCommandSwingWorker(String commandName, long timeout) {
        fcslog.info(getName() + ": sending command= " + commandName);
        new CommandSwingWorker(commandName, timeout).execute();
    }

    public void setPrintWriter(PrintWriter out) {
        this.out = out;
    }



    public void sendAsynchronousCommand(CommandRequest cmd) {
        aml.sendCommandRequest(cmd, new CommandOriginator() {

                    @Override
                    public void processAck(CommandAck ack) {
                    }

                    @Override
                    public void processResult(CommandResult result) {
                    }

                    @Override
                    public void processNack(CommandNack nack) {
                    }

                    @Override
                    public void processError(CommandError nack) {
                    }
                });
    }

    public Object sendCommand(CommandRequest cmd, long timeout) {
        try {
            fcslog.debug("fac=" + String.valueOf(aml));
            SynchronousCommandAgent agent = new SynchronousCommandAgent(aml);
            fcslog.debug("agent=" + String.valueOf(agent));
            fcslog.debug("command=" + String.valueOf(cmd));
            Object response = agent.invoke(cmd, timeout);
            //ask Bernard
            String strRes = String.valueOf(response);
            fcslog.debug("strRes=" + strRes);
            if (response == null) {
                return strRes = "ok : DONE";
                //JOptionPane.showMessageDialog(null, strRes, cmd.getCommand(),  JOptionPane.INFORMATION_MESSAGE);
            } else {
                return strRes = String.valueOf(response);
                //JOptionPane.showMessageDialog(null, strRes, cmd.getCommand(), JOptionPane.INFORMATION_MESSAGE);
            }
        } catch (Exception ex) {
            return ex;
        }
    }
    
    //To update the Subsystem state panel.
    @Override
    public void onStatusMessage(StatusMessage s) {
        
        if (!(s.getOriginAgentInfo().getName().equals(this.destination))) return;
        
        if (!panelStateInitialized) {
            this.topPanel.getSubsystemPanel().updateState(s.getState());
            panelStateInitialized = true;
        } else {
            if (!(s instanceof StateChangeNotification)) return;
            this.topPanel.getSubsystemPanel().updateState(s.getState());
        }
    }
    
    @Override
    public void connecting(AgentInfo agent) {
        //my subsystem is connecting on the bus
        if (agent.getName().equals(this.destination)) {
               fcslog.warning(destination + " is connecting NOW");
               sendAsynchronousCommand(new CommandRequest(destination,"updateStateWithSensors"));
        }
    };

    @Override
    public void disconnecting(AgentInfo agent) {
        //my subsystem is disconnecting from the bus.
        //we have to reset all the panels
        //TODO
        resetGui();
    };
   
    
}
