package org.lsst.ccs.subsystems.fcs;

import java.awt.Color;
import org.lsst.ccs.bus.annotations.SkipEncoding;

/**
 * This class groups different enum used in all the project. (for main and gui)
 *
 * @author virieux
 */
public class FcsEnumerations {

    /**
     * An enum to designate better the discrete mode as argument to the MCM wakeFilterChanger command
     *
     * 0: Go to sleep immediately
     * 1: Start the wake up process, go back to sleep at the end of filter change
     * 2: Wake up and do not go to sleep afterwards
     *
     */
    public enum CarouselPowerMode {

        GO_TO_SLEEP, WAKE_UP, STAY_UP;
    }

    /**
     * An enum for the carousel brakes state
     */
    public enum BrakeState {

        CLOSED, NO_BRAKE, NO_SENSOR
    }

    /**
     * An enum for carousel holding brakes state.
     */
    public enum CarouselHoldingBrakesState {

        TRUE, FALSE;
    }

    /**
     * An enum for the location of the loader carrier.
     */
    public enum LoaderCarrierLocation {

        STORAGE, HANDOFF, UNKNOWN, ERROR
    }

    /**
     * An enum for filters family.
     */
    public enum FilterFamily {

        S("Science Filter"),
        D("Dummy Filter"),
        F("French Dummy Filter"),
        T("Test Filter LNL"),
        O("Open / Non Attributed Filter"),
        U("Unknown Filter"),
        N("No Filter");

        @SkipEncoding
        private final String familyName;

        private FilterFamily(String familyName) {
            this.familyName = familyName;
        }

        public String getFamilyName() {
            return familyName;
        }
    }

    /**
     * The different possibilities for a carousel clamp within the presence of a
     * filter : - no filter - filter engaged and lockable - unknown state - sensor
     * in error The different possibilities for a latch in the autochanger within
     * the presence of a filter : - no filter - filter - engaged - unknown state
     *
     */
    public enum FilterPresenceStatus implements FcsColors.ColoredObject {

        NOFILTER(Color.DARK_GRAY),
        ENGAGED(FcsColors.chocolate),
        LOCKABLE(FcsColors.green),
        NOT_LOCKABLE(Color.BLUE),
        ERROR(Color.RED),
        UNKNOWN(Color.YELLOW);

        /* For GUI */
        @SkipEncoding
        private final Color color;

        /**
         * Build a FilterPresenceStatus with a color for the GUI.
         *
         * @param color
         */
        private FilterPresenceStatus(Color color) {
            this.color = color;
        }

        @Override
        public Color getColor() {
            return this.color;
        }
    }

    /**
     * An enum used for the loader clamps and autochanger latches and online clamps.
     * It describe lock status
     */
    public enum LockStatus implements FcsColors.ColoredObject {

        UNLOCKED(Color.BLUE, "UNLOCKED"),
        LOCKED(FcsColors.green, "LOCKED"),
        UNKNOWN(Color.ORANGE, "UNKNOWN"),
        ERROR(Color.RED, "IN ERROR"),
        RELAXED(FcsColors.lightblue, "RELAXED"),
        /* used for the autochanger onlineClamps and latches */
        OPENED(Color.BLUE, "OPENED"),
        CLOSED(FcsColors.green, "CLOSED"),
        /* used for the loader clamp */
        CLAMPED(FcsColors.darkGreen, "CLAMPED"),
        UNCLAMPED(Color.BLUE, "UNCLAMPED"),
        /* used for the loader clamp when there is not enough force to be clamped but too much to be taken by autochanger.*/
        UNDER_CLAMPED(Color.YELLOW, "UNDER LOAD"),
        /* used by loader clamp when force sensor value is higher that max force. */
        OVER_CLAMPED(Color.RED, " OVER LOAD"),
        /* when neither locked, nor locked */
        INTRAVEL(Color.YELLOW, "IN TRAVEL"),
        /* for the GUI's digital switch "NO ERROR" */
        NOERROR(Color.GREEN, "NO ERROR");

        @SkipEncoding
        private final Color color;

        @SkipEncoding
        private final String text;

        /**
         * Build a LockStatus with a color and a text to display on the GUI.
         *
         * @param color
         * @param text
         */
        private LockStatus(Color color, String text) {
            this.color = color;
            this.text = text;
        }

        @Override
        public Color getColor() {
            return color;
        }

        public String getText() {
            return text;
        }
    }

    /**
     * An enum for the carousel clamp state.
     */
    public enum FilterClampState implements FcsColors.ColoredObject {

        READY_TO_LOCK(Color.BLUE, "READY TO LOCK"),
        UNLOCKED_ON_FILTER(FcsColors.chocolate, "UNLOCKED ON FILTER"),
        UNLOCKED_EMPTY(FcsColors.royalblue, "UNLOCKED AND EMPTY"),
        LOCKED_ON_FILTER(FcsColors.green, "LOCKED ON FILTER"),
        UNDEFINED(Color.ORANGE, "UNDEFINED"),
        ERROR(Color.RED, "IN ERROR"),
        UNLOCKABLE(FcsColors.magenta2, "FILTER UNLOCKABLE");

        @SkipEncoding
        private final String shortDescription;

        @SkipEncoding
        private final Color color;

        FilterClampState(Color color, String aString) {
            this.shortDescription = aString;
            this.color = color;
        }

        @Override
        public String toString() {
            return shortDescription;
        }

        @Override
        public Color getColor() {
            return color;
        }
    }

    /**
     * Status of IO module (ttc-30) in carousel subsystem. There is 5
     * IO modules, one for each socket. IO status module is a device where
     * carousel clamp sensors are plugged in. 1 = is socket in standby position 2 =
     * is ready, not in position 3 = error reading position 4 = safe state 5 =
     * booting 6 = not powered on 7 = not working for other reason
     */
    public enum IOModuleStatus implements FcsColors.ColoredObject {

        UNKNOWN_STATUS(0, Color.RED),
        IS_SOCKET_AT_STANDBY(1, FcsColors.green),
        IS_READY_NOT_IN_POSITION(2, Color.BLUE),
        ERROR_READING_POSITION(3, Color.RED),
        SAFE_STATE(4, Color.RED),
        BOOTING(5, FcsColors.orange1),
        NOT_POWERED_ON(6, Color.RED),
        NOT_WORKING_FOR_OTHER_REASON(7, FcsColors.chocolate),;

        @SkipEncoding
        private final int status;

        @SkipEncoding
        private final Color color;

        private IOModuleStatus(int status, Color color) {
            this.status = status;
            this.color = color;
        }

        public int getStatus() {
            return status;
        }

        /**
         * To be displayed on GUI if needed.
         *
         * @return
         */
        @Override
        public Color getColor() {
            return color;
        }

        /**
         * Retrieve and returns a IOModuleStatus which IO status is given as an
         * argument.
         *
         * @param s numeric value of ioStatus
         * @return  IOModuleStatus
         */
        public static IOModuleStatus getStatusByCode(int s) {
            IOModuleStatus[] codes = IOModuleStatus.values();
            IOModuleStatus status = null;
            for (IOModuleStatus ioStatus : codes) {
                if (ioStatus.getStatus() == s) {
                    status = ioStatus;
                }
            }
            if (status == null) {
                throw new IllegalArgumentException(s + ": this status doesn't exist in IOModuleStatus ENUM");
            } else {
                return status;
            }
        }
    }

}
