package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.services.DataProviderDictionaryService;

public class FcsActions {

    /**
     * Interface that homogenizes the way the counters and duration of FCS actions are
     * registered and published
     */
    public interface ActionCounterPath {
        default public String name() {
            return "This is a placeholder. enum's name() function will override it.";
        };

        // Paths for the trending database, the default version will put counter at the top-level (ie. right under fcs>duration or fcs>counters)
        default public String getCounterPath() {
            return getCounterPath(null);
        }
        default public String getCounterPath(String subpath) {
            return getPath("counters", subpath);
        }

        default public String getDurationPath() {
            return getDurationPath(null);
        }

        default public String getDurationPath(String subpath) {
            return getPath("duration", subpath);
        }

        default public String getPath(String pathType, String subpath) {
            if (subpath == null) {
                return String.format("%s/%s", pathType, name());
            } else {
                return String.format("%s/%s/%s", pathType, subpath, name());
            }
        }

        /**
         * Register action duration for trending DB
         *
         * @param this the action to be recorded
         * @param dpds the data dictionary service
         * @param subpath the path where is stored the duration
         */
        default public void registerDurationPerElement(DataProviderDictionaryService dpds, String subpath) {
            registerDuration(dpds, subpath);
        }

        default public void registerDurationTopLevel(DataProviderDictionaryService dpds) {
            registerDuration(dpds, null);
        }

        default public void registerDuration(DataProviderDictionaryService dpds, String subpath) {
            String path = this.getDurationPath(subpath);
            dpds.registerData(new KeyValueData(path, 0L));
            DataProviderInfo info = dpds.getDataProviderDictionary().getDataProviderInfoForPath(path);
            info.addAttribute(DataProviderInfo.Attribute.UNITS, "millisecond");
            info.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "Duration of the action " + this.name());
            info.addAttribute(DataProviderInfo.Attribute.TYPE, "long");
        }

        /**
         * Publish action duration for trending DB
         *
         * @param this the action to be recorded
         * @param subs the subsystem
         * @param duration duration of the action
         * @param subpath the path where is stored the duration
         */
        default public void publishDurationPerElement(Subsystem subs, long duration, String subpath) {
            publishDuration(subs, duration, subpath);
        }
        default public void publishDurationTopLevel(Subsystem subs, long duration) {
            publishDuration(subs, duration, null);
        }
        default public void publishDuration(Subsystem subs, long duration, String subpath) {
            subs.publishSubsystemDataOnStatusBus(new KeyValueData(this.getDurationPath(subpath), duration));
        }
    }

    /**
     * High level actions used for recording durations and counters in trending database
     * These actions are called on the GUI and defined in FCSMain and can be made of multiple MobileItemAction.
     */
    public enum GeneralAction implements ActionCounterPath {
        SET_FILTER,
        SET_FILTER_AT_HANDOFF_FOR_LOADER,
        STORE_FILTER_ON_CAROUSEL,
        SET_NO_FILTER,
        DISENGAGE_FILTER_FROM_CAROUSEL,
        LOAD_FILTER,
        UNLOAD_FILTER,
        CONNECT_LOADER,
        DISCONNECT_LOADER,
        MOVE_AND_CLAMP_FILTER_ONLINE,
        GRAB_FILTER_AT_STANDBY,
        ROTATE_SOCKET_TO_STANDBY,
        RECOVERY_ROTATION_STRAIGHT,
        RECOVERY_ROTATION_FORWARD,
        RECOVERY_ROTATION_BACKWARD,
        RECOVERY_CLAMP_XMINUS,
        RECOVERY_UNCLAMP;
    }

    /**
     * Actions that can be made by a MobileItemModule.
     *
     * Each of these actions are associated with appropriate messages
     * that are displayed in the console or the logs
     *
     *  1-when doing the action,
     *  2-when the action is completed with success,
     *  3-when the action couldn't be completed.
     */
    public enum MobileItemAction implements ActionCounterPath {
        /********************
         * CAROUSEL ACTIONS *
         ********************/
        /* Carousel clamps */
        UNLOCK("UNLOCKING CLAMP", "IS UNLOCKED", "COULD NOT UNLOCK CLAMP"),
        RELEASE("RELEASING CLAMP", "IS RELEASED", "COULD NOT RELEASE CLAMP"),
        UNLOCK_CLAMPS("UNLOCKING CLAMPS", "CLAMPS ARE UNLOCKED", "COULD NOT UNLOCK CLAMPS"),
        RELEASE_CLAMPS("RELEASING CLAMPS", "CLAMPS ARE RELEASED", "COULD NOT RELEASE CLAMPS"),
        /* Carousel rotation */
        ROTATE_CAROUSEL_TO_RELATIVE_POSITION("ROTATING CAROUSEL", "CAROUSEL ROTATION IS COMPLETED", "COULD NOT ROTATE CAROUSEL"),
        ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION("ROTATING CAROUSEL", "CAROUSEL ROTATION IS COMPLETED", "COULD NOT ROTATE CAROUSEL"),

        /***********************
         * AUTOCHANGER ACTIONS *
         ***********************/
        /* Autochanger latches */
        OPEN("OPENING LATCH", "IS OPENED", "COULD NOT OPEN LATCH"),
        CLOSE("CLOSING LATCH", "IS CLOSED", "COULD NOT CLOSE LATCH"),
        /* Autochanger online clamps */
        CLOSE_ONLINECLAMP_MODE_CURRENT("CLOSING ONLINECLAMP", "IS CLOSED", "COULD NOT CLOSE ONLINECLAMP"),
        CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION("CLOSING ONLINECLAMP", "IS CLOSED", "COULD NOT CLOSE ONLINECLAMP"),
        LOCK_ONLINECLAMP("LOCKING ONLINECLAMP", "IS LOCKED", "COULD NOT LOCK ONLINECLAMP"),
        OPEN_ONLINECLAMP_MODE_CURRENT("OPENING ONLINECLAMP CURRENT MODE", "IS OPENED", "COULD NOT OPEN ONLINECLAMP"),
        OPEN_HOMING_ONLINECLAMP("OPEN HOMING ONLINE CLAMPS", "IS OPENED AND HOMING DONE", "COULD NOT OPEN AND DO HOMING"),
        OPEN_HALFWAY_RELATIVE_POSITION("OPEN HALFWAY RELATIVE POSITION", "IS HALFWAY BETWEEN OPEN AND CLOSE", "COULD NOT MOVE IN RELATIVE POSITION"),
        REOPEN_ONLINECLAMP_MODE_CURRENT("OPENING ONLINECLAMP CURRENT MODE", "IS OPENED", "COULD NOT OPEN ONLINECLAMP"),
        OPEN_ONLINECLAMP_MODE_PROFILE_POSITION("OPENING ONLINECLAMP", "IS OPENED", "COULD NOT OPEN ONLINECLAMP"),
        OPEN_ONLINECLAMP_MODE_PROFILE_NOBRAKE("OPENING ONLINE CLAMP WITHOUT BRAKE", "IS OPENED", "COULD NOT OPEN ONLINE CLAMP"),
        UNLOCK_ONLINECLAMP("UNLOCKING ONLINECLAMP", "IS UNLOCKED", "COULD NOT UNLOCK ONLINECLAMP"),
        /* Autochanger trucks */
        MOVE_TO_ABSOLUTE_POSITION("MOVING TO ABSOLUTE POSITION", "ABSOLUTE POSITION REACHED", "COULD NOT MOVE TO ABSOLUTE POSITION"),
        ALIGN_FOLLOWER("ALIGNING FOLLOWER TRUCK", "FOLLOWER TRUCK ALIGNED", "COULD NOT ALIGN FOLLOWER TRUCK"),
        MOVE_DRIVER_TRUCK_ALONE("MOVING DRIVER TRUCK ALONE", "DRIVER TRUCK MOVED ALONE", "COULD NOT MOVE DRIVER TRUCK ALONE"),
        MOVE_FOLLOWER_TRUCK_ALONE("MOVING FOLLOWER TRUCK ALONE", "FOLLOWER TRUCK MOVED ALONE", "COULD NOT MOVE FOLLOWER TRUCK ALONE"),

        /******************
         * LOADER ACTIONS *
         ******************/
        /* Loader clamps */
        UNCLAMP_LOADER_HOOKS("LOADER CLAMP UNCLAMPING", "HOOKS ARE UNCLAMPED", "COULD NOT UNCLAMP"),
        OPEN_HOMING_LOADER_HOOKS("OPENING LOADER HOOKS", "LOADER HOOKS ARE OPEN", "COULD NOT OPEN LOADER HOOKS"),
        CLOSE_LOADER_HOOKS("CLOSING LOADER HOOKS", "LOADER HOOKS ARE CLOSED", "COULD NOT CLOSE LOADER HOOKS"),
        CLAMP_LOADER_HOOKS("CLAMPING LOADER HOOKS", "LOADER HOOKS ARE CLAMPED", "COULD NOT CLAMP LOADER HOOKS"),
        /* Loader carrier */
        MOVE_LOADERCARRIER_TO_STORAGE("MOVING LOADER CARRIER TO STORAGE", "LOADER CARRIER IS AT STORAGE POSITION", "COULD NOT MOVE LOADER CARRIER TO STORAGE POSITION"),
        MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION("MOVING LOADER CARRIER TO ABSOLUTE POSITION", "LOADER CARRIER IS AT REQUIRED POSITION", "COULD NOT MOVE LOADER CARRIER TO REQUIRED POSITION"),
        MOVE_LOADERCARRIER_TO_HANDOFF("MOVING LOADER CARRIER TO HANDOFF", "LOADER CARRIER IS AT HANDOFF POSITION", "COULD NOT MOVE LOADER CARRIER TO HANDOFF POSITION"),
        MOVE_LOADERCARRIER_TO_ENGAGED("MOVING LOADER CARRIER TO ENGAGED", "LOADER CARRIER IS AT ENGAGED POSITION", "COULD NOT MOVE LOADER CARRIER TO ENGAGED POSITION");


        private final String shortDescription;
        private final String doneActionMsg;
        private final String failureMsg;

        /**
         * Build a MobileItemAction with 3 messages to be displayed on FCSLOG.
         *
         * @param aString
         * @param aDoneString
         * @param aFailureMsg
         */
        MobileItemAction(String aString, String aDoneString, String aFailureMsg) {
            this.shortDescription = aString;
            this.doneActionMsg = aDoneString;
            this.failureMsg = aFailureMsg;
        }

        @Override
        public String toString() {
            return shortDescription;
        }

        public String doneString() {
            return doneActionMsg;
        }

        public String getFailureMsg() {
            return failureMsg;
        }
    }

}
