package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.bus.annotations.DataAttributes;

/**
 * Data published by Autochanger seneca hardware: 8 temperatures
 */
public class StatusDataPublishedByAutochangerSeneca extends StatusDataPublishedByHardware {

    @DataAttributes(description = "Temperature of AC Cell Xminus", units = "Celsius")
    private double cellXMinus;

    @DataAttributes(description = "Temperature of AC Clamp Motor Xminus", units = "Celsius")
    private double clampMotorXMinus;

    @DataAttributes(description = "Temperature of AC Clamp Motor Xplus", units = "Celsius")
    private double clampMotorXPlus;

    @DataAttributes(description = "Temperature of AC Clamp Motor Yminus", units = "Celsius")
    private double clampMotorYMinus;

    @DataAttributes(description = "Temperature of AC Front Box", units = "Celsius")
    private double frontBox;

    @DataAttributes(description = "Temperature of AC Linear rail Motor Xminus", units = "Celsius")
    private double linearRailMotorXMinus;

    @DataAttributes(description = "Temperature of AC Linear rail Motor Xplus", units = "Celsius")
    private double linearRailMotorXPlus;

    @DataAttributes(description = "Temperature of AC Rear Box", units = "Celsius")
    private double rearBox;

    /**
     * enum definition following the order seneca1.channel 1 to 4, seneca2.channel 1 to 4
     */
    public enum TempId {
        LINEAR_RAIL_MOTOR_X_PLUS("Linear Rail Motor Xplus"),
        LINEAR_RAIL_MOTOR_X_MINUS("Linear Rail Motor Xminus"),
        CLAMP_MOTOR_X_PLUS("Clamp Motor Xplus"),
        CLAMP_MOTOR_Y_MINUS("Clamp Motor Yminus"),
        CLAMP_MOTOR_X_MINUS("Clamp Motor Xminus"),
        FRONT_BOX("Front box"),
        REAR_BOX("Rear box"),
        CELL_X_MINUS("Cell Xminus");

        private final String description;

        TempId(String description) {
            this.description = description;
        }

        public String getDescription() {
            return description;
        }
    }

    public StatusDataPublishedByAutochangerSeneca(double[] temperatures) {
        super();
        for (TempId id : TempId.values()) {
            setTemperature(id, temperatures[id.ordinal()]);
        }
    }

    private void setTemperature(TempId id, double value) {
        switch (id) {
            case LINEAR_RAIL_MOTOR_X_PLUS:
                linearRailMotorXPlus = value;
                break;
            case LINEAR_RAIL_MOTOR_X_MINUS:
                linearRailMotorXMinus = value;
                break;
            case CLAMP_MOTOR_X_PLUS:
                clampMotorXPlus = value;
                break;
            case CLAMP_MOTOR_Y_MINUS:
                clampMotorYMinus = value;
                break;
            case CLAMP_MOTOR_X_MINUS:
                clampMotorXMinus = value;
                break;
            case FRONT_BOX:
                frontBox = value;
                break;
            case REAR_BOX:
                rearBox = value;
                break;
            case CELL_X_MINUS:
                cellXMinus = value;
                break;
            default:
                throw new IllegalArgumentException("Unexpected TempId: " + id);
        }
    }

    public double getTemperature(int i) {
        if (i < 0 || i >= TempId.values().length) {
            throw new IllegalArgumentException("Temperature index must be between 0 and " + (TempId.values().length - 1) + ", inclusive. Requesting index = " + i);
        }

        TempId id = TempId.values()[i];
        switch (id) {
            case LINEAR_RAIL_MOTOR_X_PLUS:
                return linearRailMotorXPlus;
            case LINEAR_RAIL_MOTOR_X_MINUS:
                return linearRailMotorXMinus;
            case CLAMP_MOTOR_X_PLUS:
                return clampMotorXPlus;
            case CLAMP_MOTOR_Y_MINUS:
                return clampMotorYMinus;
            case CLAMP_MOTOR_X_MINUS:
                return clampMotorXMinus;
            case FRONT_BOX:
                return frontBox;
            case REAR_BOX:
                return rearBox;
            case CELL_X_MINUS:
                return cellXMinus;
            default:
                throw new IllegalArgumentException("Unexpected TempId: " + id);
        }
    }


}
