package org.lsst.ccs.subsystems.fcs;

import java.awt.Color;
import static java.awt.Color.red;
import java.io.Serializable;

/**
 * This class groups different enum used in all the project. (for main and gui)
 *
 * @author virieux
 */
public class FcsEnumerations {

    /*
     * For GUI: a green color for digital switch associated with clamps sensors when
     * clamp is LOCKED
     */
    public static final Color greenColor = new Color(0, 165, 0);

    /*
     * For GUI: a dark green color for digital switch associated with loader clamps
     * sensors used clamp is CLAMPED
     */
    public static final Color darkGreen = new Color(0, 100, 0);

    public static final Color chocolate = new Color(210, 105, 30);

    public static final Color royalblue = new Color(65, 105, 225);

    public static final Color lightblue = new Color(173, 216, 230);

    public static final Color magenta2 = new Color(238, 0, 238);

    public static final Color orange1 = new Color(255, 165, 0);

    /* The name of FCS subsystem for mcm. */
    public static final String FCS_NAME_FOR_MCM = "FILTER";

    /**
     * Name for autochanger gateway where sensors can be read. Used in simulation
     * and tests
     */
    public static final String AC_PLUTOGATEWAY_NAME = "acSensorsGateway";

    /**
     * An interface for an object to be displayed on the GUI with the appropriate
     * color.
     */
    @FunctionalInterface
    public interface ColoredObject {

        Color getColor();
    }

    /**
     * The different possibilities for a carousel clamp within the presence of a
     * filter : - no filter - filter engaged and lockable - unknown state - sensor
     * in error The different possibilities for a latch in the autochanger within
     * the presence of a filter : - no filter - filter - engaged - unknown state
     *
     */
    public enum FilterPresenceStatus implements ColoredObject {

        NOFILTER(Color.DARK_GRAY), ENGAGED(chocolate), LOCKABLE(greenColor), NOT_LOCKABLE(Color.blue), ERROR(Color.red),
        UNKNOWN(Color.YELLOW);

        /* For GUI */
        private final Color color;

        /**
         * Build a FilterPresenceStatus with a color for the GUI.
         *
         * @param color
         */
        private FilterPresenceStatus(Color color) {
            this.color = color;
        }

        @Override
        public Color getColor() {
            return this.color;
        }
    }

    /**
     * An enum used for the loader clamps and autochanger latches and online clamps.
     * It describe lock status
     */
    public enum LockStatus implements ColoredObject {

        UNLOCKED(Color.blue, "UNLOCKED"), LOCKED(greenColor, "LOCKED"), UNKNOWN(Color.orange, "UNKNOWN"),
        ERROR(Color.red, "IN ERROR"), RELAXED(lightblue, "RELAXED"),
        /* used for the autochanger onlineClamps and latches */
        OPENED(Color.blue, "OPENED"), CLOSED(greenColor, "CLOSED"),
        /* used for the loader clamp */
        CLAMPED(darkGreen, "CLAMPED"), UNCLAMPED(Color.blue, "UNCLAMPED"),
        /*
         * used for the loader clamp when there is not enough force to be clamped but
         * too much to be taken by autochanger.
         */
        UNDER_CLAMPED(Color.YELLOW, "UNDER LOAD"),
        /* used by loader clamp when force sensor value is higher that max force. */
        OVER_CLAMPED(Color.RED, " OVER LOAD"),
        /* when neither locked, nor locked */
        INTRAVEL(Color.YELLOW, "IN TRAVEL"),
        /* for the GUI's digital switch "NO ERROR" */
        NOERROR(Color.GREEN, "NO ERROR");

        private final Color color;
        private final String text;

        /**
         * Build a LockStatus with a color and a text to display on the GUI.
         *
         * @param color
         * @param text
         */
        private LockStatus(Color color, String text) {
            this.color = color;
            this.text = text;
        }

        @Override
        public Color getColor() {
            return color;
        }

        public String getText() {
            return text;
        }
    }

    /**
     * An enum for the location of the autochanger trucks.
     */
    public enum AutoChangerTrucksLocation {

        STANDBY, STANDBACK, HANDOFF, ONLINE, UNKNOWN, ERROR,
    }

    /**
     * An enum for the location of the loader carrier.
     */
    public enum LoaderCarrierLocation {

        STORAGE, HANDOFF, UNKNOWN, ERROR
    }

    /**
     * An enum for the carousel clamp state.
     */
    public enum FilterClampState implements ColoredObject {

        READYTOCLAMP(Color.blue, "READY TO CLAMP"), UNCLAMPEDONFILTER(chocolate, "UNCLAMPED ON FILTER"),
        UNCLAMPEDEMPTY(royalblue, "UNCLAMPED AND EMPTY"), CLAMPEDONFILTER(greenColor, "CLAMPED ON FILTER"),
        UNDEFINED(Color.orange, "UNDEFINED"), ERROR(Color.red, "IN ERROR"), UNLOCKABLE(magenta2, "FILTER UNLOCKABLE"); // in
                                                                                                                       // the
                                                                                                                       // case
                                                                                                                       // of
                                                                                                                       // filter
                                                                                                                       // detected.

        private final String shortDescription;
        private final Color color;

        FilterClampState(Color color, String aString) {
            this.shortDescription = aString;
            this.color = color;
        }

        @Override
        public String toString() {
            return shortDescription;
        }

        @Override
        public Color getColor() {
            return color;
        }
    }

    /**
     * Status of slave status module (ttc-30) in carousel subsystem. There is 5
     * slave modules, one for each socket. Slave status module is a device where
     * carousel clamp sensors are plugged in. 1 = is socket in standby position 2 =
     * is ready, not in position 3 = error reading position 4 = safe state 5 =
     * booting 6 = not powered on 7 = not working for other reason
     */
    public enum SlaveModuleStatus implements ColoredObject {

        UNKNOWN_STATUS(0, Color.RED), IS_SOCKET_AT_STANDBY(1, greenColor), IS_READY_NOT_IN_POSITION(2, Color.BLUE),
        ERROR_READING_POSITION(3, red), SAFE_STATE(4, Color.RED), BOOTING(5, orange1), NOT_POWERED_ON(6, Color.RED),
        NOT_WORKING_FOR_OTHER_REASON(7, chocolate),;

        private final int status;
        private final Color color;

        private SlaveModuleStatus(int status, Color color) {
            this.status = status;
            this.color = color;
        }

        public int getStatus() {
            return status;
        }

        /**
         * to be displayed on GUI if needed.
         *
         * @return
         */
        @Override
        public Color getColor() {
            return color;
        }

        /**
         * Retrieve and returns a slaveStatus which s slaveStatus is given as an
         * argument.
         *
         * @param s numeric value of slaveStatus
         * @return slaveStatus
         */
        public static SlaveModuleStatus getStatusByCode(int s) {
            SlaveModuleStatus[] codes = SlaveModuleStatus.values();
            SlaveModuleStatus status = null;
            for (SlaveModuleStatus slaveStatus : codes) {
                if (slaveStatus.getStatus() == s) {
                    status = slaveStatus;
                }
            }
            if (status == null) {
                throw new IllegalArgumentException(s + ": this status doesn't exist in SlaveModuleStatus ENUM");
            } else {
                return status;
            }
        }
    }

    /**
     * This is the different actions that can be made by a MobileItemModule. For
     * each of these actions, there is 3 String fields to define, they are messages
     * that are displayed at the console in Engineering slaveStatus or in the
     * Tracer, and log BUS: 1-when doing the action 2-when the action is completed
     * with success, 3-when the action couldn't be completed.
     */
    public enum MobileItemAction {
        /**
         * **CAROUSEL ACTIONS***
         */
        /* actions for the carousel clamps */
        UNLOCK("UNLOCKING CLAMP", "IS UNLOCKED", "COULD NOT UNLOCK CLAMP"),
        RELEASE("RELEASING CLAMP", "IS RELEASED", "COULD NOT RELEASE CLAMP"),
        UNLOCKCLAMPS("UNLOCKING CLAMPS", "CLAMPS ARE UNLOCKED", "COULD NOT UNLOCK CLAMPS"),
        RELEASECLAMPS("RELEASING CLAMPS", "CLAMPS ARE RELEASED", "COULD NOT RELEASE CLAMPS"),
        /* actions for the carousel rotation */
        ROTATE_CAROUSEL_TO_RELATIVE_POSITION("ROTATING CAROUSEL", "CAROUSEL ROTATION IS COMPLETED",
                "COULD NOT ROTATE CAROUSEL"),
        ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION("ROTATING CAROUSEL", "CAROUSEL ROTATION IS COMPLETED",
                "COULD NOT ROTATE CAROUSEL"),
        /**
         * **END OF CAROUSEL ACTIONS***
         */
        /**
         * **AUTOCHANGER ACTIONS***
         */
        /* actions for the autochanger latches */
        OPEN("OPENING LATCH", "IS OPENED", "COULD NOT OPEN LATCH"),
        CLOSE("CLOSING LATCH", "IS CLOSED", "COULD NOT CLOSE LATCH"),
        // ---- DEPRECATED
        // OPENLATCHES("OPENING LATCHES AT STANDBY", "LATCHES ARE OPENED", "COULD NOT
        // OPEN LATCHES"),
        // CLOSELATCHES("CLOSING LATCHES AT STANDBY", "LATCHES ARE CLOSED", "COULD NOT
        // CLOSE LATCHES"),
        /* actions for an autochanger online clamp */
        CLOSE_ONLINECLAMP_MODE_CURRENT("CLOSING ONLINECLAMP", "IS CLOSED", "COULD NOT CLOSE ONLINECLAMP"),
        CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION("CLOSING ONLINECLAMP", "IS CLOSED", "COULD NOT CLOSE ONLINECLAMP"),
        LOCK_ONLINECLAMP("LOCKING ONLINECLAMP", "IS LOCKED", "COULD NOT LOCK ONLINECLAMP"),
        OPEN_ONLINECLAMP_MODE_CURRENT("OPENING ONLINECLAMP CURRENT MODE", "IS OPENED", "COULD NOT OPEN ONLINECLAMP"),
        REOPEN_ONLINECLAMP_MODE_CURRENT("OPENING ONLINECLAMP CURRENT MODE", "IS OPENED", "COULD NOT OPEN ONLINECLAMP"),
        OPEN_ONLINECLAMP_MODE_PROFILE_POSITION("OPENING ONLINECLAMP", "IS OPENED", "COULD NOT OPEN ONLINECLAMP"),
        UNLOCK_ONLINECLAMP("UNLOCKING ONLINECLAMP", "IS UNLOCKED", "COULD NOT UNLOCK ONLINECLAMP"),
        /* actions for the 3 clamps all together ---- DEPRECATED */
        /* CLOSE_ONLINECLAMPS("CLOSING ONLINECLAMP", "IS CLOSED", "COULD NOT CLOSE ONLINECLAMP"),
        CLOSE_ONLINECLAMPS_MODE_CURRENT("CLOSING ONLINECLAMP", "IS CLOSED", "COULD NOT CLOSE ONLINECLAMP"),
        LOCK_ONLINECLAMPS("LOCKING ONLINECLAMPS", "ONLINECLAMPS ARE LOCKED", "COULD NOT LOCK ONLINECLAMPS"),
        OPEN_ONLINECLAMPS("OPENING ONLINECLAMP", "ONLINECLAMPS ARE OPENED", "COULD NOT OPEN ONLINECLAMP"),
        OPEN_ONLINECLAMPS_MODE_CURRENT("OPENING ONLINECLAMP", "IS OPENED", "COULD NOT OPEN ONLINECLAMP"),
        UNLOCK_ONLINECLAMPS("UNLOCKING ONLINECLAMPS", "ONLINECLAMPS ARE UNLOCKED", "COULD NOT UNLOCK ONLINECLAMPS"),
        OPEN_ONLINECLAMPSX("OPENING ONLINECLAMPS X- and X+", "ONLINECLAMPS X OPENED", "COULD NOT OPEN ONLINECLAMP X"),
        OPEN_ONLINECLAMPSX_MODE_CURRENT("OPENING ONLINECLAMPS X- and X+", "ONLINECLAMPS X OPENED",
                "COULD NOT OPEN ONLINECLAMP X"),
        CLOSE_ONLINECLAMPSX("CLOSING ONLINECLAMPS X- and X+", "ONLINECLAMPS X CLOSED", "COULD NOT CLOSE ONLINECLAMP X"), */
        /* actions for autochanger trucks */
        MOVE_TO_STANDBY_POSITION_WITH_FCS("MOVING TO STANDBY POSITION WITH FCS", "STANDBY POSITION WITH FCS REACHED",
                "COULD NOT MOVE TO STANDBY POSITION WITH FCS"),
        MOVE_TO_ABSOLUTE_POSITION("MOVING TO ABSOLUTE POSITION", "ABSOLUTE POSITION REACHED",
                "COULD NOT MOVE TO ABSOLUTE POSITION"),
        MOVE_TO_ABSOLUTE_POSITION_NO_BREAK("MOVING TO ABSOLUTE POSITION NO BREAK", "ABSOLUTE POSITION REACHED",
                "COULD NOT MOVE TO ABSOLUTE POSITION"),
        MOVE_TO_RELATIVE_POSITION("MOVING TO RELATIVE POSITION", "RELATIVE POSITION REACHED",
                "COULD NOT MOVE TO RELATIVE POSITION"),
        ALIGN_SLAVE("ALIGNING SLAVE", "SLAVE ALIGNED", "COULD NOT ALIGN SLAVE"),
        MOVE_SLAVE_TRUCK_ALONE("MOVING SLAVE_TRUCK", "SLAVE_TRUCK MOVED", "COULD NOT MOVE SLAVE_TRUCK"),
        MOVE_TRUCK_XPLUS("MOVING TRUCK_XPLUS", "TRUCK_XPLUS MOVED", "COULD NOT MOVE TRUCK_XPLUS"),
        /**
         * **END OF AUTOCHANGER ACTIONS***
         */
        /**
         * **LOADER ACTIONS***
         */
        // actions of the loader clamp
        UNCLAMPLOADERHOOKS("LOADER CLAMP UNCLAMPING", "HOOKS ARE UNCLAMPED", "COULD NOT UNCLAMP"),
        OPENLOADERHOOKS("OPENING LOADER HOOKS", "LOADER HOOKS ARE OPEN", "COULD NOT OPEN LOADER HOOKS"),
        OPENHOMINGLOADERHOOKS("OPENING LOADER HOOKS", "LOADER HOOKS ARE OPEN", "COULD NOT OPEN LOADER HOOKS"),
        CLOSELOADERHOOKS("CLOSING LOADER HOOKS", "LOADER HOOKS ARE CLOSED", "COULD NOT CLOSE LOADER HOOKS"),
        CLAMPLOADERHOOKS("CLAMPING LOADER HOOKS", "LOADER HOOKS ARE CLAMPED", "COULD NOT CLAMP LOADER HOOKS"),
        MOVE_HOOKS_TO_ABSOLUTE_POSITION("MOVING LOADER CLAMP TO POSITION", "ABSOLUTE POSITION REACHED",
                "COULD NOT MOVED" + "LOADER CLAMP TO ABSOLUTE POSITION"),
        // actions of the loader carrier
        MOVE_LOADERCARRIER_TO_STORAGE("MOVING LOADER CARRIER TO STORAGE", "LOADER CARRIER IS AT STORAGE POSITION",
                "COULD NOT MOVE LOADER CARRIER TO STORAGE POSITION"),
        MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION("MOVING LOADER CARRIER TO ABSOLUTE POSITION",
                "LOADER CARRIER IS AT REQUIRED POSITION", "COULD NOT MOVE LOADER CARRIER TO REQUIRED POSITION"),
        MOVE_LOADERCARRIER_TO_HANDOFF("MOVING LOADER CARRIER TO HANDOFF", "LOADER CARRIER IS AT HANDOFF POSITION",
                "COULD NOT MOVE LOADER CARRIER TO HANDOFF POSITION"),
        MOVE_LOADERCARRIER_TO_ENGAGED("MOVING LOADER CARRIER TO ENGAGED", "LOADER CARRIER IS AT ENGAGED POSITION",
                "COULD NOT MOVE LOADER CARRIER TO ENGAGED POSITION");

        /**
         * **END OF LOADER ACTIONS***
         *//* END OF LOADER ACTIONS */

        private final String shortDescription;
        private final String doneActionMsg;
        private final String failureMsg;

        /**
         * Build a MobileItemAction with 3 messages to be displayed on FCSLOG.
         *
         * @param aString
         * @param aDoneString
         * @param aFailureMsg
         */
        MobileItemAction(String aString, String aDoneString, String aFailureMsg) {
            this.shortDescription = aString;
            this.doneActionMsg = aDoneString;
            this.failureMsg = aFailureMsg;
        }

        @Override
        public String toString() {
            return shortDescription;
        }

        public String doneString() {
            return doneActionMsg;
        }

        public String getFailureMsg() {
            return failureMsg;
        }
    }

    /**
     * For mcm.
     */
    public enum FilterState implements Serializable {

        MOVING_TRUCKS_TO_STANDBY, MOVING_TRUCKS_TO_ONLINE, MOVING_TRUCKS_TO_HANDOFF, TRUCKS_STOPPED, MOVING_TRUCKS,
        ROTATING, CAROUSEL_STOPPED, WAITING_FOR_HARWARE_BOOTING, CAN_DEVICES_BOOTING, READY, OFF_LINE, ONLINE_U,
        ONLINE_G, ONLINE_R, ONLINE_I, ONLINE_Z, ONLINE_Y, ONLINE_NONE, ONLINE_ID_46, ONLINE_ID_40, ONLINE_ID_30,
        ONLINE_ID_27, ONLINE_ID_5, ONLINE_ID_54,
    }

    /**
     * For mcm.
     */
    public enum FilterReadinessState implements Serializable {

        NOT_READY, READY;
    }

    /**
     * An ENUM to describe all the Alerts than can be raised by FCS.
     */
    public enum FcsAlert implements Serializable {

        UNSYNC_BOOT("A boot message has been received for a device."),
        EMCY("Received an EMERGENCY message from a device."), IN_FAULT("Controller in fault."),
        CAN_BUS_TIMEOUT("Timeout expired while waiting to response from CANbus"),
        CAN_BUS_READING_ERROR("Error in reading on CANbus"),
        HARDWARE_MISSING("HARDWARE NOT DETECTED - POWER FAILURE ?"), HARDWARE_ERROR("HARDWARE ERROR"),
        CA_LOCKING_ISSUE("CAROUSEL LOCKING ISSUE"), CA_LOCKING_RECOVERY_SUCCESS("CAROUSEL LOCKING RECOVERY SUCCESSFUL"),
        BAD_SERIAL_NB("Serial number read in CPU different from serial number in configuration"),
        PARAMETER_ERROR("ERROR in CPU controller for a parameter"), SDO_ERROR("Error in a request to a CANopen device"),
        SDO_TOO_SHORT("SDO request received a too short response"),
        READ_PDO_ERROR("Error in response to a sync command"), AC_SENSOR_ERROR("Autochanger sensors are in error"),
        CA_SENSOR_ERROR("Carousel sensors are in error"), LO_SENSOR_ERROR("Loader sensors are in error"),
        AC_TRUCKS_ERROR("Autochanger trucks ERROR"), UPDATE_ERROR("ERROR in updateStateWithSensors"),
        COMPLEMENTARY_SENSOR_ERROR("complementary sensors return same values");

        private final String longDescription;

        private FcsAlert(String longDescription) {
            this.longDescription = longDescription;
        }

        public String getLongDescription() {
            return longDescription;
        }
    }

}
