package org.lsst.ccs.subsystems.fcs;

import java.awt.Color;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.greenColor;

/**
 * Here are 2 Enum : Parameter and EposMode
 *
 * @author virieux
 */
public class EPOSEnumerations {

    /**
     * Here are the parameters that we can set in an Epos controller. cf EPOS
     * Application Notes Collection
     */
    public enum Parameter {

        ConfigurationOfDigitalInput1(0x2070, 1, 2, false),
        ConfigurationOfDigitalInput2(0x2070, 2, 2, false),
        ConfigurationOfDigitalInput3(0x2070, 03, 2, false),
        DigitalInputFonctionnalityState(0x2071, 0x01, 2, false),
        DigitalInputFonctionnalityMask(0x2071, 0x02, 2, false),
        DigitalInputFonctionnalityPolarity(0x2071, 0x03, 2, false),
        DigitalInputFonctionnalityExecutionMask(0x2071, 0x04, 2, false),
        DigitalOutputFonctionnalityState(0x2078, 0x01, 2, false),
        DigitalOutputFonctionnalityMask(0x2078, 0x02, 2, false),
        DigitalOutputFonctionnalityPolarity(0x2078, 0x03, 2, false),
        ConfigurationOfDigitalOutput1(0x2079, 0x01, 2, false),
        ConfigurationOfDigitalOutput2(0x2079, 0x02, 2, false),
        ConfigurationOfDigitalOutput3(0x2079, 0x03, 2, false),
        ConfigurationOfDigitalOutput4(0x2079, 0x04, 2, false),
        CurrentThresholdHomingMode(0x2080, 0x00, 2, false), //UNSIGNED16
        HomePosition(0x2081, 0x00, 4, false),
        BrakeReactionRiseTime(0x2100, 0x01, 2, false),
        BrakeReactionFallTime(0x2100, 0x02, 2, false),
        BrakeTimeout(0x2100, 0x03, 2, false),
        PositionSensorType(0x2210, 0x02, 2, false),
        SSIEncoderActualPosition(0x2211, 0x03, 4, true), //SingleSerialData encodeur absolu only for EPOS70/10
        ControllerStructure(0x2220, 0x00, 2, false), //UNSIGNED16
        PositionActualValue(0x6064, 0x00, 4, true),
        MaxFollowingError(0x6065, 0x00, 4, false),
        PositionWindow(0x6067, 0x00, 4, false), //UNSIGNED32
        PositionWindowTime(0x6068, 0x00, 2, false), //UNSIGNED16
        HomeOffset(0x607C, 0x00, 4, true),
        MinPositionLimit(0x607D, 0x01, 4, true),
        MaxPositionLimit(0x607D, 0x02, 4, true),
        MaxProfileVelocity(0x607F, 0x00, 4, false),
        ProfileVelocity(0x6081, 0x00, 4, false),
        ProfileAcceleration(0x6083, 0x00, 4, false),
        ProfileDeceleration(0x6084, 0x00, 4, false),
        QuickStopDeceleration(0x6085, 0x00, 4, false),
        MotionProfileType(0x6086, 0x00, 2, false),
        HomingMethod(0x6098, 0x00, 1, true),
        SpeedForSwitchSearch(0x6099, 0x01, 4, false),
        SpeedForZeroSearch(0x6099, 0x02, 2, false),
        HomingAcceleration(0x609A, 0x00, 4, false),
        MotorType(0x6402, 0x00, 2, false),
        ContinuousCurrentLimit(0x6410, 0x01, 2, false),
        OutputCurrentLimit(0x6410, 0x02, 2, false),
        PolePairNumber(0x6410, 0x03, 1, false),
        MaxSpeedInCurrentMode(0x6410, 0x04, 4, false),
        ThermalTimeConstantWinding(0x6410, 0x05, 2, false),
        StatusWord(0x6041, 0x00, 2, false),;

        private final int index;
        private final int subindex;
        /* Size of the parameter : numbers of bytes*/
        private final int size;
        /*false if UNSIGNED else true*/
        private final boolean signed;

        private Parameter(int index, int subindex, int size, boolean signed) {
            this.index = index;
            this.subindex = subindex;
            this.size = size;
            this.signed = signed;
        }

        public int getIndex() {
            return index;
        }

        public int getSubindex() {
            return subindex;
        }

        public int getSize() {
            return size;
        }

        public boolean isSigned() {
            return signed;
        }

        public String display() {
            StringBuilder sb = new StringBuilder(this.toString());
            sb.append("/index=").append(Integer.toHexString(this.index));
            sb.append("/subindex=").append(Integer.toHexString(this.subindex));
            sb.append("/size=").append(this.size).append(" bytes");
            if (signed) {
                sb.append("/SIGNED");
            } else {
                sb.append("/UNSIGNED");
            }
            return sb.toString();
        }

    }

    /**
     * Here are all the modes that can be configurated for an EPOS controller.
     * For each mode, some parameters have to be set : field parameters
     */
    public enum EposMode {

        HOMING(0x6, Parameter.HomeOffset, Parameter.QuickStopDeceleration,
                Parameter.SpeedForSwitchSearch,
                Parameter.SpeedForZeroSearch, Parameter.HomingAcceleration,
                Parameter.CurrentThresholdHomingMode, Parameter.HomePosition),
        PROFILE_POSITION(0x1, Parameter.MaxFollowingError, Parameter.MinPositionLimit,
                Parameter.MaxPositionLimit, Parameter.MaxProfileVelocity,
                Parameter.ProfileVelocity, Parameter.ProfileAcceleration,
                Parameter.ProfileDeceleration, Parameter.QuickStopDeceleration,
                Parameter.MotionProfileType, Parameter.PositionWindow, Parameter.PositionWindowTime),
        PROFILE_VELOCITY(0x3),
        POSITION(0xFF),
        MASTER_ENCODER(0xFB),
        VELOCITY(0xFE),
        CURRENT(0xFD, Parameter.ContinuousCurrentLimit,
                Parameter.OutputCurrentLimit, Parameter.MaxSpeedInCurrentMode,
                //the following parameters are specific to the motor that we control with this controller.
                Parameter.PolePairNumber, Parameter.ThermalTimeConstantWinding,
                Parameter.MotorType, Parameter.PositionSensorType);

        private final int mode;
        private final Parameter[] parameters;

        private EposMode(int mode, Parameter... params) {
            this.mode = mode;
            this.parameters = params.clone();
        }

        public int getMode() {
            return this.mode;
        }

        public Parameter[] getParameters() {
            return parameters.clone();
        }

        /**
         * Retrieve and returns a mode which m code is given as an argument.
         *
         * @param m numeric value of mode
         * @return mode
         */
        public static EposMode getMode(int m) {
            EposMode[] modes = EposMode.values();
            EposMode eposMode = null;
            for (EposMode mode : modes) {
                if (mode.getMode() == m) {
                    eposMode = mode;
                }
            }
            if (eposMode == null) {
                throw new IllegalArgumentException(m + ": this mode doesn't exist in CanOpenEpos.EposMode ENUM");
            } else return eposMode;
        }

        /**
         * For a given parameter, return the mode in which this parameter has to be configured.
         * @param parameter
         * @return
         */
        public static EposMode getModeForParameter(Parameter parameter) {
            EposMode eposMode = null;
            for (EposMode mode : EposMode.values()) {
                for (EPOSEnumerations.Parameter param: mode.getParameters()) {
                    if (param.equals(parameter)) {
                        eposMode = mode;
                    }
                }
            }
            if (eposMode == null) {
                throw new IllegalArgumentException(parameter + ": this EPOS parameter doesn't exist in CanOpenEpos.EposMode ENUM");
            } else return eposMode;
        }
    }

    /**
     * States of the EPOS controller state diagram.
     */
    public enum EposState implements FcsEnumerations.ColoredObject {
        SWITCH_ON_DISABLED(Color.ORANGE),
        READY_TO_SWITCH_ON(greenColor),
        SWITCHED_ON(Color.BLUE),
        OPERATION_ENABLE(greenColor),
        FAULT(Color.RED),
        QUICKSTOP(Color.MAGENTA),
        UNKNOWN_STATE(Color.YELLOW),;

        /*For GUI*/
        private final Color color;

        /**
         * Build a EposState with a color for the GUI.
         *
         * @param color
         */
        private EposState(Color color) {
            this.color = color;
        }

        @Override
        public Color getColor() {
            return color;
        }
    }

}
