
package org.lsst.ccs.subsystems.fcs;

import java.awt.Color;
import java.io.Serializable;

/**
 * This class groups different enum used in all the project.
 * (for main and gui)
 * @author virieux
 */
public class FcsEnumerations {

    /* For GUI: a green color for digital switch associated with clamps sensors when clamp is LOCKED */
    public static final Color greenColor = new Color(0, 165, 0);
    
    /* For GUI: a dark green color for digital switch associated with loader clamps sensors used clamp is CLAMPED */
    public static final Color darkGreen = new Color(0, 100, 0);
    
    /* The name of FCS subsystem for mcm.*/
    public static final String FCS_NAME_FOR_MCM = "FILTER";

    /**
     * Name for autochanger gateway where sensors can be read.
     * Used in simulation and tests
    */
    public static final String AC_PLUTOGATEWAY_NAME = "acSensorsGateway";
    

    /**
     * An enum to describe where a filter can be in the changer subsystem.
     */
    public enum FilterLocation {
        CAROUSEL, AUTOCHANGER, LOADER, OUT, UNKNOWN
    }
    
    /**
     * An interface for an object to be displayed on the GUI with the appropriate color.
     */
    @FunctionalInterface
    public interface ColoredObject {
        Color getColor();
    }

    /**
     * The different possibilities for a carousel clamp within the presence of a
     * filter : 
     * - no filter 
     * - filter engaged and lockable 
     * - unknown state 
     * - sensor in error 
     * The different possibilities for a latch in the
     * autochanger within the presence of a filter : 
     * - no filter 
     * - filter
     * - engaged 
     * - unknown state
     *
     */
    public enum FilterPresenceStatus implements ColoredObject {

        NOFILTER(Color.DARK_GRAY), 
        ENGAGED(greenColor), 
        LOCKABLE(greenColor), 
        NOT_LOCKABLE(Color.orange), 
        ERROR(Color.red), 
        UNKNOWN(Color.gray);
        
        /*For GUI*/
        private final Color color;

        /**
         * Build a FilterPresenceStatus with a color for the GUI.
         * @param color 
         */
        private FilterPresenceStatus(Color color) {
            this.color = color;
        }

        @Override
        public Color getColor() {
            return this.color;
        }
    }


    /**
     * An enum used for the loader clamps and autochanger latches and online clamps.
     * It describe lock status
     */
    public enum LockStatus implements ColoredObject {

        UNLOCKED(Color.blue, "UNLOCKED"),
        LOCKED(greenColor, "LOCKED"),
        UNKNOWN(Color.orange, "UNKNOWN"),
        ERROR(Color.red, "IN ERROR"),
        
        /*used for the onlineClamps*/
        OPENED(Color.blue, "OPENED"),
        CLOSED(greenColor, "CLOSED"),
        
        /*used for the loader clamp*/
        CLAMPED(darkGreen, "CLAMPED"),
        UNCLAMPED(Color.blue, "UNCLAMPED"),
        
        /*used for the loader clamp when tehre is not enough force to be clamped but 
        too much to be taken by autochanger.*/
        UNDER_LOAD(Color.YELLOW, "UNDER LOAD"),        

        /*when neither locked, nor locked*/
        INTRAVEL(Color.YELLOW, "IN TRAVEL"),

        /*for the GUI's digital switch "NO ERROR"*/
        NOERROR(Color.GREEN, "NO ERROR");

        private final Color color;
        private final String text;

        /**
         * Build a LockStatus with a color and a text to display on the GUI.
         * @param color
         * @param text 
         */
        private LockStatus(Color color, String text) {
            this.color = color;
            this.text = text;
        }

        @Override
        public Color getColor() {
            return color;
        }

        public String getText() {
            return text;
        }
    }
    

    /**
     * An enum for the location of the autochanger trucks.
     */
    public enum AutoChangerTrucksLocation {

        STANDBY, STANDBACK, HANDOFF, ONLINE, UNKNOWN, ERROR,
    }

    /**
     * An enum for the location of the loader carrier.
     */
    public enum LoaderCarrierLocation {

        STORAGE, HANDOFF, UNKNOWN, ERROR
    }

    /**
     * An enum for the carousel clamp state.
     */
    public enum FilterClampState implements ColoredObject {

        READYTOCLAMP(Color.blue, "READY TO CLAMP"),
        UNCLAMPEDONFILTER(Color.orange,"UNCLAMPED ON FILTER"),
        UNCLAMPEDEMPTY(Color.DARK_GRAY,"UNCLAMPED AND EMPTY"),
        CLAMPEDONFILTER(greenColor,"CLAMPED ON FILTER"),
        UNDEFINED(Color.YELLOW,"UNDEFINED"),
        ERROR(Color.red,"IN ERROR");

        private final String shortDescription;
        private final Color color;

        FilterClampState(Color color, String aString) {
            this.shortDescription = aString;
            this.color = color;
        }

        @Override
        public String toString() {
            return shortDescription;
        }

        @Override
        public Color getColor() {
            return color;
        }
    }


    /**
     * This is the different actions that can be made by a MobileItemModule. For
     * each of these actions, there is 3 String fields to define, they are
     * messages that are displayed at the console in Engineering mode or in the
     * Tracer, and log BUS: 1-when doing the action 2-when the action is
     * completed with success, 3-when the action couldn't be completed.
     */
    public enum MobileItemAction {
        /****CAROUSEL ACTIONS****/
        /*actions for the carousel clamps*//****CAROUSEL ACTIONS****/
        /*actions for the carousel clamps*/
        UNLOCK("UNLOCKING CLAMP", "IS UNLOCKED", "COULD NOT UNLOCK CLAMP"),
        RELEASE("RELEASING CLAMP", "IS RELEASED", "COULD NOT RELEASE CLAMP"),
        UNLOCKCLAMPS("UNLOCKING CLAMPS", "CLAMPS ARE UNLOCKED", "COULD NOT UNLOCK CLAMPS"),
        RELEASECLAMPS("RELEASING CLAMPS", "CLAMPS ARE RELEASED", "COULD NOT RELEASE CLAMPS"),
        
        /*actions for the carousel rotation*/
        ROTATE_CAROUSEL_TO_RELATIVE_POSITION("ROTATING CAROUSEL", "CAROUSEL ROTATION IS COMPLETED", "COULD NOT ROTATE CAROUSEL"),
        ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION("ROTATING CAROUSEL", "CAROUSEL ROTATION IS COMPLETED", "COULD NOT ROTATE CAROUSEL"),
        /****END OF CAROUSEL ACTIONS****/
        
        /****AUTOCHANGER ACTIONS****/
        /*actions for the autochanger latches*/
        OPEN("OPENING LATCH", "IS OPEN", "COULD NOT OPEN LATCH"),
        CLOSE("CLOSING LATCH", "IS CLOSED", "COULD NOT CLOSE LATCH"),
        OPENLATCHES("OPENING LATCHES AT STANDBY", "LATCHES ARE OPEN", "COULD NOT OPEN LATCHES"),
        CLOSELATCHES("CLOSING LATCHES AT STANDBY", "LATCHES ARE CLOSED", "COULD NOT CLOSE LATCHES"),
        
        /*actions for an autochanger online clamp*/
        CLOSE_ONLINECLAMP("CLOSING ONLINECLAMP", "IS CLOSED", "COULD NOT CLOSE ONLINECLAMP"),
        LOCK_ONLINECLAMP("LOCKING ONLINECLAMP", "IS LOCKED", "COULD NOT LOCK ONLINECLAMP"), 
        OPEN_ONLINECLAMP("OPENING ONLINECLAMP", "IS OPENED", "COULD NOT OPEN ONLINECLAMP"),
        UNLOCK_ONLINECLAMP("UNLOCKING ONLINECLAMP", "IS UNLOCKED", "COULD NOT UNLOCK ONLINECLAMP"),
        /*actions for the 3 clamps all together */
        CLOSE_ONLINECLAMPS("CLOSING ONLINECLAMP", "IS CLOSED", "COULD NOT CLOSE ONLINECLAMP"),
        LOCK_ONLINECLAMPS("LOCKING ONLINECLAMPS", "ARE LOCKED", "COULD NOT LOCK ONLINECLAMPS"),
        OPEN_ONLINECLAMPS("OPENING ONLINECLAMP", "IS OPENED", "COULD NOT OPEN ONLINECLAMP"),
        UNLOCK_ONLINECLAMPS("UNLOCKING ONLINECLAMPS", "ARE CLOSED", "COULD NOT UNLOCK ONLINECLAMPS"),
        CLOSE_AND_LOCK_ONLINECLAMPS("CLOSING AND LOCKING ONLINECLAMPS", "ARE LOCKED", "COULD NOT LOCK ONLINECLAMPS"),
        UNLOCK_AND_OPEN_ONLINECLAMPS("UNLOCKING AND OPENING ONLINECLAMPS", "ARE OPENED", "COULD NOT UNLOCK ONLINECLAMPS"),
        /*actions for autochanger trucks*/
        MOVE_TO_ABSOLUTE_POSITION("MOVING TO ABSOLUTE POSITION", "ABSOLUTE POSITION REACHED",
                "COULD NOT MOVE TO ABSOLUTE POSITION"),
        MOVE_TO_RELATIVE_POSITION("MOVING TO RELATIVE POSITION", "RELATIVE POSITION REACHED",
                "COULD NOT MOVE TO RELATIVE POSITION"),
        /****END OF AUTOCHANGER ACTIONS****/
        
        /****LOADER ACTIONS****/
        //actions of the loader clamp
        UNCLAMPLOADERHOOKS("LOADER CLAMP UNCLAMPING", "HOOKS ARE UNCLAMPED", "COULD NOT UNCLAMP"),
        GOTOHOMEPOSITION("LOADER CLAMP GOING TO HOME POSITION", "HOOKS ARE AT HOME POSITION", "COULD NOT GO TO HOME POSITION"),
        OPENLOADERHOOKS("OPENING LOADER HOOKS", "LOADER HOOKS ARE OPEN", "COULD NOT OPEN LOADER HOOKS"),
        CLOSELOADERHOOKS("CLOSING LOADER HOOKS", "LOADER HOOKS ARE CLOSED", "COULD NOT CLOSE LOADER HOOKS"),
        CLAMPLOADERHOOKS("CLAMPING LOADER HOOKS", "LOADER HOOKS ARE CLAMPED", "COULD NOT CLAMP LOADER HOOKS"),
        //actions of the loader carrier
        MOVE_LOADERCARRIER_TO_STORAGE("MOVING LOADER CARRIER TO STORAGE", "LOADER CARRIER IS AT STORAGE POSITION", "COULD NOT MOVE LOADER CARRIER TO STORAGE POSITION"),
        MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION("MOVING LOADER CARRIER TO ABSOLUTE POSITION", "LOADER CARRIER IS AT REQUIRED POSITION", "COULD NOT MOVE LOADER CARRIER TO REQUIRED POSITION"),
        MOVE_LOADERCARRIER_TO_HANDOFF("MOVING LOADER CARRIER TO HANDOFF", "LOADER CARRIER IS AT HANDOFF POSITION", "COULD NOT MOVE LOADER CARRIER TO HANDOFF POSITION");
        /****END OF LOADER ACTIONS****//*END OF LOADER ACTIONS*/

        private final String shortDescription;
        private final String doneActionMsg;
        private final String failureMsg;

        /**
         * Build a MobileItemAction with 3 messages to be displayed on FCSLOG.
         * @param aString
         * @param aDoneString
         * @param aFailureMsg 
         */
        MobileItemAction(String aString, String aDoneString, String aFailureMsg) {
            this.shortDescription = aString;
            this.doneActionMsg = aDoneString;
            this.failureMsg = aFailureMsg;
        }

        @Override
        public String toString() {
            return shortDescription;
        }

        public String doneString() {
            return doneActionMsg;
        }

        public String getFailureMsg() {
            return failureMsg;
        }
    }
    
    /**
     * For mcm.
     * //TODO add an other enum for the name of filter at ONLINE
     */
    public enum FilterState implements Serializable {
        MOVING_TRUCKS_TO_STANDBY, MOVING_TRUCKS_TO_ONLINE, TRUCKS_STOPPED,
        ONLINE_U, ONLINE_G, ONLINE_R, ONLINE_I, ONLINE_Z, ONLINE_Y, ONLINE_NONE, 
        ROTATING, CAROUSEL_STOPPED, 
        WAITING_FOR_CWRAPPER, CAN_DEVICES_BOOTING, HOMING_TO_BE_DONE, HARDWARE_READY, OFF_LINE
    }

    /**
     * For mcm.
     */
    public enum FilterReadinessState implements Serializable {
        NOT_READY, READY;
    }
    
    /**
     * An ENUM to describe all the Alerts than can be raised by FCS.
     */
    public enum FcsAlert implements Serializable {
        
        UNSYNC_BOOT("An unsynchronous boot message is arrived for a booted device."),
        EMCY("Received an EMERGENCY message from a device."),
        CAN_BUS_TIMEOUT("Timeout expired while waiting to response from CANbus"),
        HARDWARE_MISSING("HARDWARE NOT DETECTED - POWER FAILURE ?"),
        HARDWARE_ERROR("HARDWARE ERROR during INITIALIZATION PHASE"),
        PARAMETER_ERROR("ERROR in CPU controller for a parameter"),
        SDO_ERROR("Error in a request to a CANopen device"),
        SDO_TOO_SHORT("SDO request received a too short response"),
        AC_SENSOR_ERROR ("Autochanger sensors are in error"),
        CA_SENSOR_ERROR ("Carousel sensors are in error"),
        LO_SENSOR_ERROR ("Loader sensors are in error"),
        AC_TRUCKS_ERROR("Autochanger trucks ERROR"),
        COMPLEMENTARY_SENSOR_ERROR("complementary sensors return same values");
        
        private final String longDescription;

        private FcsAlert(String longDescription) {
            this.longDescription = longDescription;
        }

        public String getLongDescription() {
            return longDescription;
        } 
    }

}
