package org.lsst.ccs.subsystem.doorman.main;

import java.util.Optional;

import java.util.function.Function;

import java.util.stream.Stream;

/**
 * The different kinds of instruments recognized by the subsystem.
 * @author tether
 */
public enum InstrumentType {
    
    /**
     * Motivair ChilledDoor Rack Cooling system.
     */
    CHILLED_DOOR("chilledDoor", ChilledDoor::new),
    
    /**
     * A simulated instrument for testing purposes.
     */
    DUMMY("dummy", DummyInstrument::new);
    
    private final String configName;
    
    private final Function<InstrumentConfig, Instrument> maker;
    
    InstrumentType(String configName, Function<InstrumentConfig, Instrument> maker) {
        this.configName = configName;
        this.maker = maker;
    }
    
    /**
     * Gets the instrument type name as it should appear in configuration data.
     * @return The configuration name.
     */
    public String getConfigName() {return configName;}
    
    /**
     * Makes the right kind of {@code Instrument} objects using {@code InstrumentConfigs}.
     * @param config The configuration information needed to make the new instrument object.
     * @return The new instrument object.
     */
    public Instrument make(InstrumentConfig config) {return maker.apply(config);}
    
    /**
     * Converts a string to one of the instances of this enum. The string, converted to
     * lower case, should match the configuration name of one of the instances.
     * @param name The string to parse.
     * @return A full Optional if success, otherwise an empty one.
     */
    public static Optional<InstrumentType> parse(final String name) {
        return Stream.of(InstrumentType.values())
            .filter(x -> name.equalsIgnoreCase(x.getConfigName()))
            .findAny();
    }
}
