package org.lsst.ccs.subsystem.doorman.main;

import java.time.Instant;

/**
 * A stand-alone application to pull data out of the IR2 ChilledDoor and
 * write the results to the standard output. The first line contains the location
 * of the device, such as "ir2/chilledDoor", followed by the readout time in UTC,
 * all preceded by "@@ " to make the line sort to the top.
 * The next lines, if any, each contain a channel name and channel value.
 * If any exceptions were thrown during the reading they'll be printed to
 * the standard error after the channel values have been output.
 * @author tether
 */
public class DoorPuller {

    public static void main(final String[] args) {
        if (args.length != 1) {
            usage("Wrong number of arguments.");
        }
        else {
            final String doorhost = args[0];
            // Create a new Instrument object. The index number and last readout time
            // don't matter for this application so we use zero and the present
            // instant.
            final InstrumentConfig config =
                new InstrumentConfig(0, InstrumentType.CHILLED_DOOR, doorhost);
            Instrument door = new ChilledDoor(config);
            // Read the data and create a new Instrument with the results.
            door = door.read();
            // Print any channel/value pairs that were collected, preceded by the line
            door
                .getTrendables()
                .forEach(trend -> {
                    System.out.format
                        ("%n@@ %s %s%n", trend.getMasterKey(), trend.getMasterTimestamp());
                    trend
                        .getItems()
                        .forEach((key, value)->System.out.format("%s %s%n", key, value));
            });
            // Log to stderr any exceptions that occurred.
            door
                .getLastException()
                .ifPresent(exc -> System.err.println(Instant.now() + "\n" +exc));
        }
    }

    private static void usage(final String msg) {
        System.err.println(msg);
        System.err.println("Usage: <path-to-distribution>/bin/CCSbootstrap.sh -app DoorPuller <door hostname or IP>");
    }
}
