package org.lsst.ccs.subsystem.doorman.main;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;

import org.lsst.ccs.ComponentConfigurationEnvironment;

import org.lsst.ccs.framework.HasLifecycle;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;

import java.util.List;
import java.util.Set;

import java.util.Collections;

import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;

/**
 * Implements a local configuration service based on the CCS subsystem configuration API.
 * An instance will be the receptacle of all the subsystem configuration data which is declared
 * here using the {@link ConfigurationParameter} annotation.
 * @author tether
 */
public class CCSConfiguration implements HasLifecycle, LocalConfigurationService {
    ////////// References to subsystem services and components //////////
    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile ConfigurationService ccsConfig;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    ////////// Implementation of LocalConfigurationService //////////
    private volatile Duration readoutInterval;
    private volatile List<InstrumentConfig> instConfigs;

    /**
     * Checks the raw configuration data and makes the configuration objects.
     * Called at the start of the initialization phase.
     */
    @Override
    public void makeConfigurationObjects() {
        final ComponentConfigurationEnvironment configEnv = subsys.getComponentConfigurationEnvironment(this);
        configEnv.printConfigurableParameters(new String[]{})
            .forEach((key, value)->System.out.println(key + " :: " + value));
        try {
            readoutInterval = Duration.parse(readoutIntervalString);
        }
        catch (java.time.DateTimeException exc) {
            throw new IllegalArgumentException("Bad configuration. Invalid readout interval.");
        }
        makeInstrumentConfigs();
    }

    /** {@inheritDoc } */
    @Override
    public List<InstrumentConfig> getInstrumentConfigs() {
        return instConfigs;
    }

    /** {@inheritDoc } */
    @Override
    public Duration getReadoutInterval() {return readoutInterval;}
    
    /** {@iinheritDoc} */
    @Override
    public String getEmailSender() {return emailSender;}
    
    /** {@inheritDoc } */
    @Override
    public String getEmailBounceAddress() {return emailBounceTo;}
    
    /** {@inheritDoc } */
    @Override
    public String getSMTPServer() {return SMTPServer;}
    
    /** {@inheritDoc } */
    @Override
    public List<String> getEmailRecipients() {
        return Collections.unmodifiableList(Arrays.asList(emailRecipients));
    }


    private void makeInstrumentConfigs() {
        final List<String> types = instrumentTypes;
        final List<String> conns = instrumentConnections;
        // Validity checking.
        if (types.size() != conns.size()) {
            throw new IllegalArgumentException(
                "Bad configuration. Not all the instrumentXxx[] arrays are the same length."
            );
        }
        if (types.isEmpty()) {
            throw new IllegalArgumentException("Bad configuration. No instruments defined!");
        }
        checkStringSet(
            types.stream(),
            Stream.of(InstrumentType.values()).map(InstrumentType::getConfigName),
            "Unknown instrument types");
        // Construction of configuration objects.
        instConfigs = Collections.unmodifiableList(
            IntStream
            .range(0, types.size())
            .mapToObj(i -> new InstrumentConfig(
                    i,
                    InstrumentType.parse(types.get(i)).get(),
                    conns.get(i)))
            .collect(Collectors.toList()));
    }

    private static void checkStringSet(
        Stream<String> mentioned,
        Stream<String> legal,
        String errorMsg)
    {
        Set<String> unknown = mentioned.collect(Collectors.toSet());
        unknown.removeAll(legal.collect(Collectors.toSet()));
        if (unknown.size() > 0) {
            throw new IllegalArgumentException(
                "Bad configuration. " + errorMsg + ": " +
                unknown.stream().collect(Collectors.joining(", ")));
        }
    }

    ////////// Lifecycle methods //////////

    /** Makes an instance. */
    public CCSConfiguration() {}

    /**
     * Check that all the subsystem services we need have been found.
     * @throws RuntimeException if any are missing.
     */
    @Override
    public void init() {
        final List<String> missing = new ArrayList<>();
        if (ccsConfig == null) {missing.add("CCS configuration service");}
        if (subsys == null) {missing.add("CCS Subsystem object");}
        if (!missing.isEmpty()) {
            throw new RuntimeException("Can't find " + String.join(", ", missing));
        }
    }

    ////////// Configuration data //////////

    /**
     * The time between readouts of all the instruments.
     */
    @ConfigurationParameter(name="readoutInterval", description="The time between readouts.")
    private volatile String readoutIntervalString = "PT1H";

    /**
     * For each instrument, its make and/or model, for example "chilledDoor".
     */
    @ConfigurationParameter(description="Instrument make and/or model.")
    private volatile List<String> instrumentTypes = Collections.synchronizedList(new ArrayList<>());

    /**
     * For each instrument, a string containing connection information to be parsed.
     * For example "lsst-ir2ChilledDoor.slac.stanford.edu".
     */
    @ConfigurationParameter(description="Connection info for the instrument.")
    private volatile List<String> instrumentConnections = Collections.synchronizedList(new ArrayList<>());
    
    /**
     * The name or IPv4 address of the SMTP server to use to send emails.
     */
    @ConfigurationParameter(description="The name or IPv4 address of the SMTP server.")
    private volatile String SMTPServer = "";
    
    /**
     * What to put in the From: fields of emails headers.
     */
    @ConfigurationParameter(description="Email address, possibly fake, used as email sender.")
    private volatile String emailSender = "";
    
    /**
     * What to put in the Bounce-to: fields for alarm emails. Must be a the address of
     * a real mailbox.
     */
    @ConfigurationParameter(description="Real email address which receives bounced messages.")
    private volatile String emailBounceTo = "";
    
    /**
     * The list of email recipients. A set of real email addresses.
     */
    @ConfigurationParameter(description="List of recipients of alarm emails.")
    private volatile String[] emailRecipients = new String[] {};
}
