package org.lsst.ccs.subsystem.doorman.main;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;

import org.lsst.ccs.ComponentConfigurationEnvironment;
import org.lsst.ccs.UsesSubsystem;

import org.lsst.ccs.framework.HasLifecycle;

import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;

import java.util.List;
import java.util.Set;

import java.util.Collections;

import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

/**
 * Implements a configuration service based on the CCS subsystem configuration API.
 * An instance will be the receptacle of all the subsystem configuration data which is declared
 * here using the {@link ConfigurationParameter} annotation.
 * @author tether
 */
public class CCSConfiguration implements HasLifecycle, UsesSubsystem, ConfigurationService {

    ////////// Implementation of ConfigurationService //////////
    private volatile Duration readoutInterval;
    private volatile List<InstrumentConfig> instConfigs;

    /** {@inheritDoc } */
    @Override
    public List<InstrumentConfig> getInstrumentConfigs() {
        return instConfigs;
    }

    /** {@inheritDoc } */
    @Override
    public void updateInstrument(InstrumentStatus stat) {
        // Safely publish a new last-data-times array using defensive copy. Needed
        // since elements of a voltile array are not volatile.
        String[] newTimes = Arrays.copyOf(instrumentLastDataTimes, instrumentLastDataTimes.length);
        newTimes[stat.index] = stat.lastDataTime.toString();
        instrumentLastDataTimes = newTimes;
        // Make change requests.
        configEnv.change("instrumentLastDataTimes", instrumentLastDataTimes);
        // Atomically update the central copy of the configuration so that other
        // users will see a consistent new configuration.
        configEnv.saveAllChanges();
    }

    /** {@inheritDoc } */
    @Override
    public Duration getReadoutInterval() {return readoutInterval;}
    
    /** {@iinheritDoc} */
    @Override
    public String getEmailSender() {return emailSender;}
    
    /** {@inheritDoc } */
    @Override
    public String getEmailBounceAddress() {return emailBounceTo;}
    
    /** {@inheritDoc } */
    @Override
    public String getSMTPServer() {return SMTPServer;}
    
    /** {@inheritDoc } */
    @Override
    public List<String> getEmailRecipients() {
        return Collections.unmodifiableList(Arrays.asList(emailRecipients));
    }


    ////////// Lifecycle methods //////////

    /** Makes an instance. */
    public CCSConfiguration() {}

    /**
     * Checks the raw configuration data and makes the configuration objects.
     * Called at the start of the initialization phase.
     */
    @Override
    public void start() {
        configEnv = getComponentConfigurationEnvironment();
        configEnv.printConfigurableParameters(new String[]{})
            .forEach((key, value)->System.out.println(key + " :: " + value));
        try {
            readoutInterval = Duration.parse(readoutIntervalString);
        }
        catch (java.time.DateTimeException exc) {
            throw new IllegalArgumentException("Bad configuration. Invalid readout interval.");
        }
        System.out.println(readoutInterval);
        makeInstrumentConfigs();
    }

    private void makeInstrumentConfigs() {
        final String[] types = instrumentTypes;
        final String[] conns = instrumentConnections;
        final String[] times = instrumentLastDataTimes;
        // Validity checking.
        if ((types.length != conns.length)
            || (types.length != times.length)) {
            throw new IllegalArgumentException(
                "Bad configuration. Not all the instrumentXxx[] arrays are the same length."
            );
        }
        if (types.length == 0) {
            throw new IllegalArgumentException("Bad configuration. No instruments defined!");
        }
        checkStringSet(
            Stream.of(types),
            Stream.of(InstrumentType.values()).map(InstrumentType::getConfigName),
            "Unknown instrument types");
        checkStringSet(
            Stream.of(times),
            Stream.of(times).filter(t -> {
                try {
                    Instant.parse(t);
                    return true;
                } catch (java.time.format.DateTimeParseException exc) {
                    return false;
                }
            }),
            "Last data-times rejected by Instant.parse()"
        );
        // Construction of configuration objects.
        instConfigs = Collections.unmodifiableList(
            IntStream
            .range(0, types.length)
            .mapToObj(i -> new InstrumentConfig(
                    i,
                    InstrumentType.parse(types[i]).get(),
                    conns[i],
                    Instant.parse(times[i])))
            .collect(Collectors.toList()));
    }

    private static void checkStringSet(
        Stream<String> mentioned,
        Stream<String> legal,
        String errorMsg)
    {
        Set<String> unknown = mentioned.collect(Collectors.toSet());
        unknown.removeAll(legal.collect(Collectors.toSet()));
        if (unknown.size() > 0) {
            throw new IllegalArgumentException(
                "Bad configuration. " + errorMsg + ": " +
                unknown.stream().collect(Collectors.joining(", ")));
        }
    }

    ////////// Local non-configuration data //////////

    private ComponentConfigurationEnvironment configEnv = null;

    ////////// Configuration data //////////

    /**
     * The time between readouts of all the instruments.
     */
    @ConfigurationParameter(name="readoutInterval", description="The time between readouts.")
    private volatile String readoutIntervalString = "PT1H";

    /**
     * For each instrument, its make and/or model, for example "chilledDoor".
     */
    @ConfigurationParameter(description="Instrument make and/or model.")
    private volatile String[] instrumentTypes = new String[] {};

    /**
     * For each instrument, a string containing connection information to be parsed.
     * For example "lsst-ir2ChilledDoor.slac.stanford.edu".
     */
    @ConfigurationParameter(description="Connection info for the instrument.")
    private volatile String[] instrumentConnections = new String[] {};

    /**
     * For each instrument, the time of the most recent data read from it. Subsequent
     * reads will obtain data later than this. The string must be acceptable to
     * Instant.parse().
     */
    @ConfigurationParameter(description="The Instant of the lastest data read from the instrument.")
    private volatile String[] instrumentLastDataTimes = new String[] {};
    
    /**
     * The name or IPv4 address of the SMTP server to use to send emails.
     */
    @ConfigurationParameter(description="The name or IPv4 address of the SMTP server.")
    private volatile String SMTPServer = "";
    
    /**
     * What to put in the From: fields of emails headers.
     */
    @ConfigurationParameter(description="Email address, possibly fake, used as email sender.")
    private volatile String emailSender = "";
    
    /**
     * What to put in the Bounce-to: fields for alarm emails. Must be a the address of
     * a real mailbox.
     */
    @ConfigurationParameter(description="Real email address which receives bounced messages.")
    private volatile String emailBounceTo = "";
    
    /**
     * The list of email recipients. A set of real email addresses.
     */
    @ConfigurationParameter(description="List of recipients of alarm emails.")
    private volatile String[] emailRecipients = new String[] {};
}
