package org.lsst.ccs.subsystem.doorman.main;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import org.apache.commons.mail.Email;
import org.apache.commons.mail.SimpleEmail;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.framework.SignalLevel;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

/** The main module for the Doorman subsystem. This module is responsible for:
 * <ul>
 * <li>Establishing initial contact with the configured instruments.</li>
 * <li>Providing and managing the periodic readout task.</li>
 * <li>Alert management.</li>
 * <li>Instrument management and status reporting.</li>
 * </ul>
 *  * @author tether
 */
public final class DoormanMain
extends Subsystem
implements HasLifecycle, SignalHandler {
/* The module maintains a list of immutable Instrument objects which describe the state
 * of each instrument and specify the allowed operations. Mutating operations therefore follow
 * the read-modify-write pattern. A CopyOnWriteArrayList is enough to synchronize query
 * commands with mutating action commands and with the readout task but it isn't enough to
 * synchronize the action commands with the periodic readout.
 *
 * Take for example the case of disabling readout for a previously enabled instrument with
 * no prior readout data. The disable() command copies the instrument state and
 * produces a new state with the enabled flag set to false. While that's happening
 * the readout task might copy the same old state and produce a new state with new readout data.
 * Depending on who writes back first either the enable flag will remain true or the
 * new readout data will become inaccessible.
 *
 * For this reason there's an extra lock used to synchronize the readout task with mutating
 * action commands. The readout task will hold the lock while it reads instruments and modifies
 * the state list. The mutating action commands must take the lock before doing anything else,
 * delaying the readout task until they finish their execution, which should be brief.
 * They in turn release the lock after completing their changes.
 *
 * The periodic readout task normally works its way down the instrument list, reading any data
 * accumulated since the last-data-time for each instrument and publishing that data in the
 * trending database. As it goes the task modifies the list of Instrument objects to
 * reflect the new state of data collection for each instrument.
 *
 * The readout task checks for interruption after performing read-and-publish for each
 * instrument. This is needed to ensure that the instrument's last-data-time is consistent with
 * what has been posted for trending. An interruption will cause the Instrument
 * objects that have not been processed to be left unchanged in the instrument list.
 *
 */

    ////////// Commands //////////

    /**
     * Gives the status of each of the instruments. Includes instrument number,
     * enable flag and the timestamp of the last record read.
     * @return A string with a heading line followed by one line per instrument.
     */
    @Command(type=QUERY, description="Displays a short report on instrument status.")
    public String instruments() {return new InstrumentReport(instruments).toString();}

    /**
     * Prevents an instrument from being read until further notice.
     * @param index An instrument index number.
     * @return "OK", or "BUSY" if a readout is in progress.
     * @throws Exception if {@code ident} is invalid or ambiguous.
     * @see #enable(int)
     */
    @Command(type=ACTION, description="Prevents further readout of an instrument")
    public String disable(
        @Argument(description="Instrument index number.")
        final int index
    ) throws Exception
    {
        return callIfNotBusy(() -> {
            final Instrument ins = findInstrument(index);
            if (ins == null) {
                throw new IllegalArgumentException(("No such instrument."));
            }
            instruments.set(ins.getIndex(), ins.disable());
            return "OK";
        });
    }

    /**
     * Enables periodic readout for an instrument.
     * @param index An instrument index number.
     * @return "OK", or "BUSY" if a readout is in progress.
     * @throws IllegalArgumentException if {@code ident} is invalid or ambiguous.
     * @throws IllegalArgumentException if {@code newLocation} is invalid.
     * @throws HardwareException if device operations fail.
     * @see #disable(int)
     */
    @Command(type = ACTION, description="Enables an instrument for readout.")
    public String enable(
        @Argument(description="Instrument index number.")
        final int index
    ) throws Exception
    {
        return callIfNotBusy( () -> {
            final Instrument ins = findInstrument(index);
            if (ins == null) {
                throw new IllegalArgumentException(("No such instrument."));
            }
            final Instrument newIns = ins.enable();
            instruments.set(ins.getIndex(), newIns);
            if (newIns.getLastException().isPresent()) {throw newIns.getLastException().get();}
            return "OK";
        });
    }

    /**
     * Signal handler for stop and abort.
     * @param sig The signal.
     * @return TreeWalkerDiag.GO.
     */
    @Override
    public TreeWalkerDiag signal(Signal sig) {
        if (sig.getLevel() == SignalLevel.STOP) {
            readoutTask.cancel(false); // Don't interrupt any current readout.
        }
        else if (sig.getLevel() == SignalLevel.HALT) {
            readoutTask.cancel(true); // Interrupt any current readout.
        }
        return TreeWalkerDiag.GO;
    }


    ////////// References to other subsystem components //////////

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private LocalConfigurationService configService;


    ////////// Subsystem state data //////////

    /**
     * The descriptions of all instruments being managed. Updated by ACTION
     * commands and read by the readout task and by QUERY commands.
     */
    private final CopyOnWriteArrayList<Instrument> instruments;

    /**
     * Synchronizes action commands that mutate {@link #instruments} with
     * the periodic readout task.
     */
    private final Lock readoutLock;

    /**
     * Regularly scheduled to read the instruments.
     */
    private PeriodicTask readoutTask;

    /**
     * The subsystem's logger.
     */
    private final Logger log;
    
    //A ClearAlertHandler that aways clears Alerts
    private final ClearAlertHandler alwaysClear = new ClearAlertHandler() {
        /**
         * Allows all alerts to be cleared unconditionally.
         *
         * @param alert The particular alert to be cleared.
         * @param state The current state (severity) of the alert.
         * @return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT
         */
        @Override
        public ClearAlertHandler.ClearAlertCode canClearAlert(final Alert alert, final AlertState state) {
            return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
        }
    };

    ////////// Lifecycle methods //////////
    // Arranged in chronological order of calling.

    /**
     * Constructor.
     */
    public DoormanMain() {
        // This is a worker subsystem.
        super("doorman", AgentInfo.AgentType.WORKER);

        instruments = new CopyOnWriteArrayList<>();
        log = Logger.getLogger("org.lsst.ccs.subsystem.doorman");
        readoutTask = null;
        readoutLock = new ReentrantLock();
    }

    /**
     * Checks whether all the subsystem components have been constructed
     * and registers the ClearAlertHandler.
     * Called just after the subsystem component tree has been built.
     * @throws RuntimeException if anything is missing.
     */
    @Override
    public void init() {
        final List<String> missing = new ArrayList<>();
        if (alertService == null) {missing.add("CCS alert service");}
        if (configService == null) {missing.add("local configuration service");}
        if (!missing.isEmpty()) {
            throw new RuntimeException("Can't find " + String.join(", ", missing));
        }
        alertService.registerAlert(getInstrumentStartupProblemAlert(), alwaysClear);
        alertService.registerAlert(getEmailProblemsAlert(), alwaysClear);
    }
    
    private Alert getInstrumentStartupProblemAlert() {
        return new Alert("InstrumentStartupProblem", "Raised when instruments fail to start.");
    }
    
    /**
     * Attempts to start all configured instruments.
     * All instruments are stopped and re-started
     * with correct configurations for their assigned locations.
     */
    @Override
    public void start() {
        try {
            configService.makeConfigurationObjects();
            getConfiguration();
            //Register ioProblems Alerts
            instruments.forEach(inst -> {
                alertService.registerAlert(getIOProblemsAlert(inst.getStatus().location),alwaysClear);
            });            
            enableAllInstruments();
            scheduleFirstReadout(Duration.ZERO);
        } catch (HardwareException ex) {
            alertService.raiseAlert(getInstrumentStartupProblemAlert(),AlertState.ALARM, ex.getMessage(),
                ON_SEVERITY_CHANGE);
        }
    }


    private void getConfiguration() {
        final List<InstrumentConfig> configs = configService.getInstrumentConfigs();
        final List<Instrument> insts
            = configs
            .stream()
            .map(cfg -> cfg.type.make(cfg))
            .collect(Collectors.toList());
        instruments.addAll(insts);
    }

    private void enableAllInstruments() throws HardwareException {
        final List<HardwareException> exceptions = new ArrayList<>();
        instruments.forEach(inst -> {
            final Instrument newIns = inst.enable();
                instruments.set(inst.getIndex(), newIns);
                newIns.getLastException().ifPresent(exc -> exceptions.add(exc));
        });
        // Make a chain of all the hardware exceptions and throw the top of the chain.
        if (exceptions.size() > 0) {
            HardwareException top = exceptions.remove(0);
            for (HardwareException exc : exceptions) {top = new HardwareException(exc);}
            throw top;
        }
    }

    private void scheduleFirstReadout(Duration initialDelay) {
        this.readoutTask =
            this
                .getScheduler()
                .scheduleAtFixedRate(this::readoutTaskBody,
                    initialDelay.getSeconds(),
                    configService.getReadoutInterval().getSeconds(),
                    TimeUnit.SECONDS);
    }

    private void readoutTaskBody() {
        //  Synchronize with mutating action commands.
        try {
            this.readoutLock.lockInterruptibly();
        }
        catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
            return;
        }

        // Read, update status, post trendables.
        try {
            instruments
                .stream()
                .filter(ins -> ins.getStatus().enabled)
                .forEach(ins -> {
                    if (Thread.currentThread().isInterrupted()) {
                        throw new TerminationException();
                    }
                    readAndUpdate(ins);
                });
        } catch (TerminationException exc) {
            log.warning("Instrument readout was interrupted!");
        }
        finally {
            readoutLock.unlock();
        }
    }

    private static class TerminationException extends RuntimeException {}

    private void readAndUpdate(final Instrument ins) {
        final Instrument newIns = ins.read();
        instruments.set(ins.getIndex(), newIns);
        checkForExceptionDuringReadout(newIns);
        // Send trendables, if any, to trending.
        newIns
            .getTrendables()
            .forEach(msg -> msg.post(this));
    }

    private static final int EXC_COUNT_ALERT_THRESHOLD = 15;

    private void checkForExceptionDuringReadout(final Instrument newIns) {
        // If too many consecutive reads had exceptions then raise an alert and send an e-mail message.
        final String badloc = newIns.getStatus().location;
        if (newIns.getExceptionCount() > 0) {
            log.error(badloc + ": Readout error", newIns.getLastException().get());
        }
        final boolean tooManyExceptions = newIns.getExceptionCount() > EXC_COUNT_ALERT_THRESHOLD;
        if (tooManyExceptions) {
            final HardwareException exc = newIns.getLastException().get();
            Alert ioAlert = getIOProblemsAlert(badloc);
            alertService.raiseAlert(
                ioAlert,
                AlertState.WARNING,
                exc.getMessage(),
                ON_SEVERITY_CHANGE);
            sendEmail(ioAlert.getDescription(), exc.toString());
        }
    }
    
    private Alert getIOProblemsAlert(String location) {
        return new Alert("IO-problems-"+location, "Too long a run of readout problems");
    }

    /**
     * Does nothing.  Called at the end of the initialization phase
     * after {@code start()} has been called for all configurable components and after
     * {@code checkHardware()} and {@code checkStarted()} have been called for all
     * hardware controllers.
     */
    @Override
    public void postStart() {}

    /**
     * Cancels the readout task if that hasn't already been done.
     * The instruments themselves are not stopped and continue to collect data.
     * Called at the beginning of the shutdown phase.
     * @see #signal(org.lsst.ccs.framework.Signal)
     */
    @Override
    public void shutdown() {
        readoutTask.cancel(false);
        if (!readoutTask.isCancelled()) {
            throw new RuntimeException("The readout task couldn't be cancelled.");
        }
    }

    ////////// Utilities //////////

    /**
     * Gets the Instrument with the given index.
     * @param index The index number.
     * @return The instrument, null if none.
     */
    private Instrument findInstrument(final int index) {
        Instrument ins = null;
        try {
            ins = instruments.get(index);
        }
        catch (IndexOutOfBoundsException exc) {}
        return ins;
    }

    /**
     * Calls a {@code Callable<String>} on behalf of a command that needs to synchronize
     * with the periodic readout. Upon exit the current thread will
     * not hold the readout lock.
     * @param cmd The code to run.
     * @return The result from {@code cmd}.
     * @throws Throwable if {@code cmd.call()} throws.
     */
    private String callIfNotBusy(Callable<String> cmd) throws Exception {
        if (!readoutLock.tryLock()) {
            // Send rejection of the command.
            return "BUSY";
        }
        try {
            // Execute the command.
            return cmd.call();
        } finally {
            readoutLock.unlock();
        }
    }

    private void sendEmail(final String subject, final String message) {
        try {
            final String[] recipients = configService.getEmailRecipients().toArray(new String[]{});
            if (recipients.length > 0) {
                final Email emsg = new SimpleEmail();
                emsg.setHostName(configService.getSMTPServer());
                emsg.setFrom(configService.getEmailSender());
                emsg.setBounceAddress(configService.getEmailBounceAddress());
                emsg.addReplyTo(configService.getEmailBounceAddress());
                emsg.setSubject("[Doorman] " + subject);
                emsg.setMsg(message);
                emsg.addTo(recipients);
                emsg.send();
            }
        }
        catch (org.apache.commons.mail.EmailException exc) {
            Alert a = getEmailProblemsAlert();
            log.error(a.getDescription(), exc);
            alertService.raiseAlert(a, AlertState.WARNING, message, ON_SEVERITY_CHANGE);
        }
    }
    
    private Alert getEmailProblemsAlert() {
        return new Alert("email-problems", "Error composing or sending email.");
    }
}
