package org.lsst.ccs.subsystem.doorman.main;

import java.util.Optional;

import java.util.stream.Stream;

import static org.lsst.ccs.subsystem.doorman.main.DoorChannelType.*;

/**
 * The quantities monitored by each instrument. Modbus holding register numbers have
 * 40000 subtracted, while coil numbers have 10000 subtracted.
 * @author tether
 */
public enum DoorChannel {

    /** Air supply outlet temperature. */
    AIR_OUTLET_TEMP("airOutletTemp", ANALOG, 2),

    /** The return-water coil temperature. */
    RETURN_WATER_TEMP("returnWaterTemp", ANALOG, 6),

    /** Valve output signal expressed as a percentage. */
    VALVE_OUTPUT("valveOutput", ANALOG, 9),

    /** The room temperature. */
    ROOM_TEMP("roomTemp", ANALOG, 10),

    /** The server outlet air temperature. */
    SERVER_OUTLET_AIR_TEMP("serverOutletAirTemp", ANALOG, 17),

    /** Speed of fan 1 in RPM. */
    FAN1_SPEED("fan1Speed", INTEGER, 5016),

    /** Speed of fan 1 in RPM. */
    FAN2_SPEED("fan2Speed", INTEGER, 5017),

    /** Speed of fan 1 in RPM. */
    FAN3_SPEED("fan3Speed", INTEGER, 5018),

    /** Speed of fan 1 in RPM. */
    FAN4_SPEED("fan4Speed", INTEGER, 5019),

    /** Speed of fan 1 in RPM. */
    FAN5_SPEED("fan5Speed", INTEGER, 5020),

    /** Door clock day. */
    CLOCK_DAY("clockDay", INTEGER, 5032),

    /** Door clock month. */
    CLOCK_MONTH("clockMonth", INTEGER, 5033),

    /** Door clock year. */
    CLOCK_YEAR("clockYear", INTEGER, 5034),

    /** Door clock hour. */
    CLOCK_HOUR("clockHour", INTEGER, 5035),

    /** Door clock minute. */
    CLOCK_MINUTE("clockMinute", INTEGER, 5036),

    /** High air temperature alarm. */
    HIGH_AIR_TEMP("highAirTemp", DIGITAL, 14),

    /** High water temperature alarm. */
    HIGH_WATER_TEMP("highWaterTemp", DIGITAL, 15),

    /** Low water temperature alarm. */
    LOW_WATER_TEMP("lowWaterTemp", DIGITAL, 16),

    /** Unit state: 0 for off, 1 for on. */
    UNIT_STATE("unitState", DIGITAL, 17),

    /** Water detected (condensation?) alarm. */
    WATER_DETECTED("waterDetected", DIGITAL, 22),

    /** General alarm output. */
    GENERAL_ALARM("generalAlarm", DIGITAL, 23),

    /** Clock board alarm. */
    CLOCK_ALARM("clockAlarm", DIGITAL, 24),

    /** High room temperature. */
    HIGH_ROOM_TEMP("highRoomTemp", DIGITAL, 25),

    /** Low room temperature. */
    LOW_ROOM_TEMP("lowRoomTemp", DIGITAL, 26),

    /** Fan 1 speed alarm. */
    FAN1_ALARM("fan1Alarm", DIGITAL, 30),

    /** Fan 1 speed alarm. */
    FAN2_ALARM("fan2Alarm", DIGITAL, 31),

    /** Fan 1 speed alarm. */
    FAN3_ALARM("fan2Alarm", DIGITAL, 32),

    /** Fan 1 speed alarm. */
    FAN4_ALARM("fan3Alarm", DIGITAL, 33),

    /** Fan 1 speed alarm. */
    FAN5_ALARM("fan4Alarm", DIGITAL, 34),

    /** High server outlet temp. */
    HIGH_SERVER_OUTLET_TEMP("highServerOutletTemp", DIGITAL, 37),

    /** Fan high pressure alarm. */
    HIGH_FAN_PRESSURE("highFanPressure", DIGITAL, 40),

    /** Probe B4 alarm. */
    B4_ALARM("b4Alarm", DIGITAL, 58),

    /** Low temperature in the air supply. */
    LOW_AIR_SUPPLY_TEMP("lowAirSupplyTemp", DIGITAL, 59),

    /** The unit door is open. */
    DOOR_OPEN("doorOpen", DIGITAL, 60);


    private final String key;
    private final DoorChannelType type;
    private final int register;

    DoorChannel(final String key, final DoorChannelType type, final int register) {
        this.key = key;
        this.type = type;
        this.register = register;
    }

    /**
     * Gets the trending key used to identify the channel.
     * @return The key.
     */
    public String getKey() {return key;}

    /**
     * Gets the channel type.
     * @return The type enumerator.
     */
    public DoorChannelType getType() {return type;}

    /**
     * Gets the Modbus register number.
     * @return The register number.
     */
    public int getRegister() {
        return register;
    }

    /**
     * Converts a trending key string into the corresponding channel.
     * @param key The trending key.
     * @return An empty Optional if the key is invalid, otherwise an Optional holding
     * the enum value.
     */
    public static Optional<DoorChannel> parse(String key) {
        return Stream.of(DoorChannel.values())
            .filter(chan -> key.equalsIgnoreCase(chan.getKey()))
            .findAny();
    }
}
