package org.lsst.ccs.subsystem.doorman.main;

import org.lsst.ccs.HardwareException;

import java.util.stream.Stream;

import java.util.Optional;

/**
 * Represents ChilledDoor. For ease of sharing between threads, objects
 * implementing this interface must be immutable. In place of mutators there
 * are methods that produce new instances with altered states.
 * <p>
 * When first constructed from subsystem configuration data each {@code Instrument}
 * should be disabled, have no data and no stored exception.
 * @author tether
 */
public interface Instrument {

    /**
     * Gets the exception thrown (if any) the last time we tried to operate on this instrument.
     * @return An empty {@code Optional} if there have been no operations or if the last operation
     * was successful. Otherwise an {@code Optional} containing the exception that terminated
     * the operation. Getters are not considered operations.
     */
    Optional<HardwareException> getLastException();

    /**
     * Gets the number of consecutive operations that were terminated by an exception.
     * @return The consecutive exception count, which is zero if no exception occurred on
     * the last operation.
     */
    int getExceptionCount();

    /**
     * Gets the position of this instrument in the list of instruments.
     * @return The index.
     * @see LocalConfigurationService#getInstrumentConfigs() 
     */
    int getIndex();

    /**
     * Gets all the trendable data resulting from the last reading of the instrument.
     * @return A stream of {@code TrendableRecord} instances, empty if no data
     * is available. The master key of each record is the location for
     * which the data were taken. The master timestamp is the time at which the record was
     * made. Each key/value pair consists of a channel name and value.
     * @see #read()
     */
    Stream<TrendableRecord> getTrendables();

    /**
     * Gets the current status of the instrument. This method will be used
     * in QUERY-type commands, so implementations must rely
     * solely on data in memory and must not perform any hardware operations.
     * @return The status object.
     */
    InstrumentStatus getStatus();

    /**
     * Produces a new instance that has readout disabled.
     * @return The new instance.
     */
    Instrument disable();

    /**
     * Produces a new instance that has been enabled for readout.
     * @return The new instance.
     */
    Instrument enable();

    /**
     * Produces a new instance that contains a fresh set of data read from
     * the instrument. The {@link #getTrendables()} method relies on this
     * data to produce its result.
     * @return The new instance.
     */
    Instrument read();
}
