package org.lsst.ccs.subsystem.doorman.main;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.drivers.modbus.Modbus;
import org.lsst.ccs.drivers.commons.DriverException;
import static org.lsst.ccs.subsystem.doorman.main.DoorChannelType.*;


/**
 *
 * @author tether
 */
public class ChilledDoor implements Instrument {

    private final int index;
    private final boolean enabled;
    private final String hostname;
    private final Optional<HardwareException> lastExc;
    private final Optional<TrendableRecord> lastData;
    private final int excCount;

    private static final String LOCATION = "ir2/chilledDoor";

    public ChilledDoor(final InstrumentConfig config) {
        this.index = config.index;
        this.enabled = false;
        this.hostname = config.conn;
        this.lastExc = Optional.empty();
        this.lastData = Optional.empty();
        this.excCount = 0;
    }

    private ChilledDoor(
            final int index,
            final boolean enabled,
            final String hostname,
            final Optional<HardwareException> exc,
            final Optional<TrendableRecord> data,
            final int excCount)
    {
        this.index = index;
        this.enabled = enabled;
        this.hostname = hostname;
        this.lastExc = exc;
        this.lastData = data;
        this.excCount = excCount;
    }


    @Override
    public Optional<HardwareException> getLastException() {
        return lastExc;
    }

    @Override
    public int getExceptionCount() {
        return excCount;
    }

    @Override
    public int getIndex() {
        return index;
    }

    @Override
    public Stream<TrendableRecord> getTrendables() {
        return lastData.map(Stream::of).orElseGet(Stream::empty);
    }

    @Override
    public InstrumentStatus getStatus() {
        return new InstrumentStatus(index, enabled, LOCATION);
    }

    @Override
    public Instrument disable() {
        return new ChilledDoor(index, false, hostname, lastExc, lastData, excCount);
    }

    @Override
    public Instrument enable() {
        return new ChilledDoor(index, true, hostname, lastExc, lastData, 0);
    }

    @Override
    public Instrument read() {
        final Instant now = Instant.now();
        final Modbus door = new Modbus();
        HardwareException thisExc = null;
        try {
            door.open(Modbus.ConnType.NET, hostname);
        }
        catch (DriverException exc) {
            thisExc = new HardwareException(exc, thisExc);
        }
        final Map<String, java.io.Serializable> channels = new HashMap<>();
        if (thisExc == null) {
            for (DoorChannel chan: DoorChannel.values()) {
                try {
                    double chanval = 0.0;
                    if (chan.getType() == ANALOG) {
                        final short sval =
                        door.readRegisters((short)0, (short)chan.getRegister(), (short)1)[0];
                        chanval = sval / 10.0;
                    }
                    else if (chan.getType() == INTEGER) {
                        final short sval =
                        door.readRegisters((short)0, (short)chan.getRegister(), (short)1)[0];
                        chanval = sval;
                    }
                    else {
                        final byte cval =
                        door.readCoils((short)0, (short)chan.getRegister(), (short)1)[0];
                        chanval = (cval != 0) ? 1.0 : 0.0;
                    }
                    channels.put(chan.getKey(), chanval);
                }
                catch (DriverException exc) {
                    thisExc = new HardwareException(exc, thisExc);
                }
            }
        }
        final TrendableRecord msg =
            new TrendableRecord(LOCATION, now, channels);
        return new ChilledDoor(index, enabled, hostname,
            Optional.ofNullable(thisExc), Optional.of(msg),
            (thisExc == null) ? 0 : excCount + 1);
    }

}
