package org.lsst.ccs.subsystem.doorman.main;

import org.lsst.ccs.HardwareException;

import org.lsst.ccs.bus.data.KeyValueDataList;

import java.time.Instant;

import java.util.Optional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.HashMap;

import java.util.stream.Stream;

/**
 * A dummy instrument used to test the subsystem. Produces a trendable cosine wave of unit
 * amplitude on the room temperature channel. Every fifth simulated operation fails.
 * Read operations take ten seconds unless interrupted or unless they experience
 * a simulated exception.
 *
 * @author tether
 */
public class DummyInstrument implements Instrument {

    private final int index;
    private final boolean enabled;
    private final Instant lastDataTime;
    private final long nops;
    private final Optional<HardwareException> lastExc;
    private final Optional<TrendableRecord> lastData;

    private static final String LOCATION = "doormanTest/dummyDoor";

    public DummyInstrument(InstrumentConfig config) {
        this.index = config.index;
        this.enabled = false;
        this.lastDataTime = config.lastDataTime;
        this.nops = 0;
        this.lastExc = Optional.empty();
        this.lastData = Optional.empty();
    }

    private DummyInstrument(
            int index,
            boolean enabled,
            Instant lastDataTime,
            long nops,
            Optional<HardwareException> exc,
            Optional<TrendableRecord> data)
    {
        this.index = index;
        this.enabled = enabled;
        this.lastDataTime = lastDataTime;
        this.nops = nops;
        this.lastExc = exc;
        this.lastData = data;
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public Optional<HardwareException> getLastException() {
        return lastExc;
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public int getIndex() {
        return index;
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public InstrumentStatus getStatus() {
        return new InstrumentStatus(index, enabled, LOCATION, lastDataTime);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Stream<TrendableRecord> getTrendables() {
        return lastData.map(Stream::of).orElseGet(Stream::empty);
    }

    /** {@inheritDoc } */
    @Override
    public Instrument disable() {
        final long n = nops + 1;
        final Optional<HardwareException> exc =
                (n % 5 == 0)
                ? Optional.of(new HardwareException("Simulated exception during disable.", null))
                : Optional.empty();
        return new DummyInstrument(index, false, lastDataTime, n, exc, lastData);
    }

    /** {@inheritDoc } */
    @Override
    public Instrument enable() {
        final long n = nops + 1;
        final Optional<HardwareException> exc =
                (n % 5 == 0)
                ? Optional.of(new HardwareException("Simulated exception during enable.", null))
                : Optional.empty();
        return new DummyInstrument(index, true, lastDataTime, n, exc, lastData);
    }

    /** {@inheritDoc } */
    @Override
    public Instrument read() {
        // Simulate a hardware exception if it's time for one.
        final long n = nops + 1;
        if (n % 5 == 0) {
            return new DummyInstrument(index, enabled, lastDataTime, n,
            Optional.of(new HardwareException(false, "Simulated exception during read.")),
            Optional.empty());
        }

        // Simulate a time-consuming operation, which if interrupted becomes
        // essentially a no-op.
        try {
            Thread.sleep(10_000);
        }
        catch (InterruptedException exc) {
            Thread.currentThread().interrupt(); // Caller must know about interruptions.
            return new DummyInstrument(index, enabled, lastDataTime, n,
            Optional.empty(), Optional.empty());
        }

        // Generate data.
        final Instant now = Instant.now();
        final Map<String, java.io.Serializable> channels = new HashMap<>();
        final double chanval = Math.cos(now.getEpochSecond());
        channels.put(DoorChannel.ROOM_TEMP.getKey(), chanval);
        final boolean viol = (chanval > 0.8);
        final TrendableRecord msg =
            new TrendableRecord("", now, channels);
        return new DummyInstrument(index, enabled, now, n,
            Optional.empty(), Optional.of(msg));
    }
}
