package org.lsst.ccs.subsystem.demo.main;

import java.util.concurrent.TimeUnit;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.demo.bus.DemoData;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;
import hep.aida.*;
import java.io.Serializable;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.Options;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command.CommandCategory;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.command.annotations.Option;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.config.ConfigurationParameterDescription;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.jgroups.ClusterSplitUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.demo.bus.DemoState;
import org.lsst.ccs.subsystem.demo.main.DemoSubsystem.OtherState;
import org.lsst.ccs.utilities.scheduler.Scheduler;

public class DemoSubsystem extends Subsystem implements HasLifecycle {
    
    public static enum OtherState {
        LOW, HIGH;
    }
    
    public static enum AState {
        A,B;
    }

    /**
     * Scheduler task that publishes heartbeat status messages.
     */
    protected PeriodicTask dataPublisher;

    //Create analysis factory
    private final IAnalysisFactory af = IAnalysisFactory.create();
    // Create datapoint set factory
    private final IDataPointSetFactory dataPointSetFactory = af.createDataPointSetFactory(null);
    // Create histogram factory
    private final IHistogramFactory histogramFactory = af.createHistogramFactory(null);

    private final Random r = new Random();

    private final ArrayList<PeriodicTask> tracerTestTasks = new ArrayList<>();

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    protected Subsystem subsys;
    
    @Persist
    private volatile String somePersistStr = "abc";
    
    @LookupPath
    private String nodePath;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private List<DemoConfigurable> configurables = new ArrayList<>() ;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private DataProviderDictionaryService dataProviderDictionaryService;

    public final static Alert MONITOR_ALERT = new Alert("MonitorAlert","Raised when monitoring quantities are ouside boundaries.");
    
    private int dataPublishRate = 5;
    private int dataSize = 2;
    
    private volatile long taskSleep = 0L;
    
    private volatile int numberOfExceptions = 0;
    
    private final List<byte[]> junk = Collections.synchronizedList(new ArrayList<>());

    @ConfigurationParameter(category = "build")
    private volatile String buildPar = "someValue";

    private ScheduledExecutorService junkPublisher; // publishes test messages; null if not currently publishing
    
    public DemoSubsystem() {
        super("demo-subsystem", AgentInfo.AgentType.WORKER);
    }
    
    /**
     ***************************************************************************
     **
     ** Initializes the subsystem. *
     * **************************************************************************
     */
    @Override
    public void postStart() {
        // This must not be invoked in init because the subsystem is not yet 
        // connected on the buses.
        updateLimits("1.0", "2.0");
        somePersistStr = String.valueOf(System.currentTimeMillis());        
    }

    @Override
    public void postInit() {
        // Add AgentInfo properties
        subsys.getAgentService(AgentPropertiesService.class).setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY,"demo");
        
        //set default Persistency strategy
        subsys.getAgentPersistenceService().setAutomatic(true,true);
    }
    @Override
    public void build() {
                
        ConfigurationParameterDescription parDesc = new ConfigurationParameterDescription().withCategory("customCat")
                .withDescription("Period for data publishing").withName("period");
        
        //Schedule a periodic task to publish demo data
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask("demoData-publish",
                        () -> {
                            if ( numberOfExceptions > 0 ) {
                                numberOfExceptions--;
                                throw new RuntimeException("An Exception");
                            }
                            if ( taskSleep <=0 ) {
                                subsys.publishSubsystemDataOnStatusBus(getDemoDataToPublish("fixedRate"));
                            } else {
                                try {
                                    Thread.sleep(taskSleep);
                                } catch (InterruptedException e) {
                                    throw new RuntimeException(e);
                                }
                            }
                        }).withIsFixedRate(true).withPeriod(Duration.ofSeconds(dataPublishRate)).withPeriodParameterDescription(parDesc));
        
        //Schedule a periodic task to publish demo data
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask("demoData-publish-fixedDelay",
                        () -> {
                            if ( taskSleep <=0 ) {
                                subsys.publishSubsystemDataOnStatusBus(getDemoDataToPublish("fixedDelay"));
                            } else {
                                try {
                                    Thread.sleep(taskSleep);
                                } catch (InterruptedException e) {
                                    throw new RuntimeException(e);
                                }
                            }
                        }).withIsFixedRate(false).withPeriod(Duration.ofSeconds(dataPublishRate)));
    }

    @Command(type = CommandType.ACTION) 
    public void setNumberOfExceptionInMonitoringThread(int nExeptions) {
        numberOfExceptions = nExeptions;
    }

    @Command(type = CommandType.ACTION) 
    public void setPeriodicTaskSleepTime(long sleep) {
        taskSleep = sleep;
    }
    
    @Override
    public void init() {
        subsys.getAgentService(DataProviderDictionaryService.class).registerData(getDemoDataToPublish("fixedRate"));
        subsys.getAgentService(DataProviderDictionaryService.class).registerData(getDemoDataToPublish("fixedDelay"));

        for (DemoConfigurable c : configurables) {
            subsys.getAgentService(AgentStateService.class).registerState(OtherState.class, "some state", c);
            subsys.getAgentService(AgentStateService.class).updateAgentComponentState(c, OtherState.HIGH);
            subsys.getAgentService(AgentStateService.class).registerState(DemoState.class, "some state", c);
            subsys.getAgentService(AgentStateService.class).updateAgentComponentState(c, DemoState.NOMINAL);
        }

        subsys.getAgentService(AlertService.class).registerAlert(getCustomAlert());
    }
    
    
    private KeyValueData getDemoDataToPublish(String prefix) {
        DemoData dd = new DemoData(dataSize);
        KeyValueData d = new KeyValueData(prefix+"_demo_Data", dd);
        KeyValueDataList kvdl = new KeyValueDataList();
        KeyValueDataList limits = new KeyValueDataList(prefix+"_demo_Data/temp");
        limits.addData(prefix+"_test", 1.2,KeyValueData.KeyValueDataType.KeyValueTrendingData);
        kvdl.addData(limits);
        kvdl.addData(d);
        return kvdl;
    }
    
    
    @Command(type = CommandType.ACTION, description = "update limits")
    public void updateLimits(String low, String high) {
        KeyValueDataList limits = new KeyValueDataList("demo_Data/temp");
        dataProviderDictionaryService.addMetadataForObject(limits, "alarmLow", low, this);
        dataProviderDictionaryService.addMetadataForObject(limits, "alarmHigh", high, this);
        subsys.publishSubsystemDataOnStatusBus(limits);
    }

    private Alert getCustomAlert() {
        return new Alert("CustomAlert", "Custom Alert");
    }
    
    @Command(type = CommandType.QUERY, description = "raise custom alert")
    public void raiseCustomAlert(AlertState severity, String cause) {
        subsys.getAgentService(AlertService.class).raiseAlert(getCustomAlert(), severity, cause);
    }
    
    @Command(type = CommandType.QUERY, description = "raise custom alert")
    public void raiseAlert(AlertState severity, String id, String cause) {
        if (id == null || id.isBlank()) {
            id = "CustomAlert";
        }
        if (cause == null || cause.isBlank()) {
            cause = "Unspecified cause";
        }
        subsys.getAgentService(AlertService.class).raiseAlert(new Alert(id, "Description of "+ id +" alert."), severity, cause);
    }
    
    @Command(type = CommandType.QUERY, description = "Get Subsystem Data Sample")
    public DemoData getSampleData() {
        return new DemoData(dataSize);
    }

    @Command(type = CommandType.QUERY, description = "Set Subsystem Data Sample")
    public void setSampleDataSize(int size) {
        this.dataSize = size;
    }
    
    
    @Command(type = CommandType.QUERY, description = "Get Subsystem Array Sample")
    public Object[] getSampleArrayData() {
        Object[] data = new Object[]{"test", 2.3, new int[]{3, 2, 2, 2}};
        return data;
    }
    
    @Command(type = CommandType.QUERY, description = "Throw an Exception")
    public void throwAnException() {
        throw new RuntimeException("An Exception");
    }

    @Command(type = CommandType.ACTION, description = "Enable/Disable Publishing Tracer Test Messages")
    public void runTracerTest(boolean enable) {
        if (enable) {
            Scheduler s = subsys.getScheduler();
            tracerTestTasks.add(s.scheduleAtFixedRate(() -> subsys.getLogger().info("Self test on module " + r.nextInt(1000) + (r.nextInt(5) > 0 ? ": OK" : ": FAILED")), 0, 7, TimeUnit.SECONDS));
            tracerTestTasks.add(s.scheduleAtFixedRate(() -> subsys.getLogger().warn("London Bridge is in trouble, please repair."), 2, 15, TimeUnit.SECONDS));
            tracerTestTasks.add(s.scheduleAtFixedRate(() -> subsys.getLogger().error("Sun has gone supernova."), 3, 13, TimeUnit.SECONDS));
        } else {
            tracerTestTasks.forEach(task -> task.cancel(false));
            tracerTestTasks.clear();
        }
    }

    @Command(type = CommandType.ACTION, description = "Time out (action command)")
    public String testTimeout(int sleepSeconds) {
        if (sleepSeconds > 0) {
            try {
                Thread.sleep(sleepSeconds * 1000L);
            } catch (InterruptedException x) {
                return "Interrupted";
            }
        }
        return "Done.";
    }

    @Command(type = CommandType.QUERY, level = 0, category = CommandCategory.USER, description = "Empty level 0 category USER type QUERY command")
    public void testUserQuery0() {
    }

    @Command(type = CommandType.ACTION, level = 0, category = CommandCategory.USER, description = "Empty level 0 category USER type QUERY command")
    public void testUserAction0() {
    }

    @Command(type = CommandType.QUERY, level = 1, category = CommandCategory.USER, description = "Empty level 0 category USER type QUERY command")
    public void testUserQuery1() {
    }

    @Command(type = CommandType.QUERY, level = 0, category = CommandCategory.SYSTEM, description = "Empty level 0 category USER type QUERY command")
    public void testSystemQuery0() {
    }

    @Command(type = CommandType.QUERY, level = 1, category = CommandCategory.SYSTEM, description = "Empty level 0 category USER type QUERY command")
    public void testSystemQuery1() {
    }

    @Command(type = CommandType.QUERY, level = 0, category = CommandCategory.SYSTEM, description = "Fill heap with junk. 0 to clear.")
    public void fillMemory(int megabytes) {
        if (megabytes <= 0) {
            junk.clear();
        } else {
            byte[] b = new byte[megabytes * 1000000];
            r.nextBytes(b);
            junk.add(b);
        }
    }

    @Command(type = CommandType.QUERY, level = 0, description = "Test sending ACK/NACK. Delays are in seconds.", autoAck = false)
    public Object testCommand(
            @Argument(description = "If false, command is rejected with a NACK.")
            boolean accept,
            @Argument(defaultValue = "0")
            int delayBeforeAck,
            @Argument(description = "Delay between ACK and the result.", defaultValue = "0")
            int delayAfterAck,
            @Argument(defaultValue = "0")
            int timeout, 
            @Argument(defaultValue = "false")
            boolean returnException) {

        if (delayBeforeAck > 0) {
            try {
                Thread.sleep(delayBeforeAck * 1000L);
            } catch (InterruptedException x) {
                return "Interrupted while waiting to send ACK/NACK";
            }
        }

        if (accept) {
            if (timeout > 0) {
                subsys.sendAck(Duration.ofSeconds(timeout));
            } else {
                subsys.sendAck(null);
            }
        } else {
            subsys.sendNack("Sending NACK as requested.");
            return "NACK";
        }

        if (delayAfterAck > 0) {
            try {
                Thread.sleep(delayAfterAck * 1000L);
            } catch (InterruptedException x) {
                return "Interrupted while waiting to send the result";
            }
        }
        
        if (returnException) {
            return new RuntimeException("Requsted exception result.");
        } else {
            return "Done.";
        }
    }

    @Command(type = CommandType.QUERY, level = 0, description = "Test various types of command arguments and return types.")
    public Object testCommandArguments(
            @Argument(name = "namedBool", description = "boolean argument with name but no default")
            boolean b1,
            @Argument(defaultValue = "0", description = "unnamed int argument with default 0")
            int varName,
            @Argument(defaultValue = "Che", description = "String argument: nickname", name = "nickname")
            String nick,
            @Argument(description = "List of integers")
            List<Integer> intList,
            @Argument(description = "List of strings")
            List<String> stringList,
            @Argument(description = "Kind of object to return", allowedValueProvider = "testCommandArgumentsReturnTypeValues")
            String returnType,
            @Argument(name = "vararg", description = "String...")
            String... var) {
        
        switch (returnType) {
            case "unknown":
                return new Unknown();
            case "exception":
                return new RuntimeException("Exception returned.");
            case "string":
                return "Done";
            case "short_list":
                return Arrays.asList(new String[] {"11", "22", "33"});
            case "long list":
                return Arrays.asList(new String[] {"first moderately long line",
                                                   "shorty",
                                                   "second (kind of) long line with ()",
                                                   "The third, extremely long line that almost never ends. Contains a comma, too."});
            case "int array":
                return new int[] {0, 1, 2, 4};
            case "list of args":
                List<String> out = new ArrayList<>();
                out.add("b1 = "+ Boolean.toString(b1));
                out.add("varName = "+ Integer.toString(varName));
                out.add("nickname = "+ nick);
                out.add("intList = "+ String.join(",", intList.stream().map(i -> Integer.toString(i)).collect(Collectors.toList())));
                out.add("stringList = "+ String.join(",", stringList));
                out.add("returnType = "+ returnType);
                out.add("strings = "+ String.join(",", var));
                return out;
            case "map_of_args":
                HashMap<String, Object> map = new HashMap<>();
                map.put("b1", b1);
                map.put("varName", varName);
                map.put("nickname", nick);
                map.put("intList", intList);
                map.put("stringList", stringList);
                map.put("returnType", returnType);
                map.put("vararg", var);
                return map;
            default:
                return "Unexpected value for returnType:"+ returnType;
        }
    }
    
    
    @Option(name = "invert", description = "Invert the provided input String")
    @Option(name = "uppercase", description = "Uppercate the provided input String")
    @Option(name = "space", description = "Space apart the letters")
    @Command(description = "A Command with Options")
    public String echoWithOptions(Options opts, String input) {
        String tmpRes = input;
        if (opts.hasOption("invert")) {
            StringBuilder sb = new StringBuilder(input); 
            sb.reverse();
            tmpRes = sb.toString();
        }
            
        if ( opts.hasOption("uppercase") ) {
            tmpRes = tmpRes.toUpperCase();
        }

        if ( opts.hasOption("space") ) {
            String tmpTmpRes = "";
            for (int i = 0; i < tmpRes.length(); i++) {
                tmpTmpRes += Character.toString(tmpRes.charAt(i))+" ";
            }
            tmpRes = tmpTmpRes.trim();
        }
        
        return tmpRes;
            
            
        
    }

    /**
    @Command(type = CommandType.QUERY, level = 0, category = CommandCategory.SYSTEM, description = "Show data")
    public void showData() {
        
        AgentStatusAggregatorService sa = getAgentService(AgentStatusAggregatorService.class);
        DataProviderDictionaryService ds = getAgentService(DataProviderDictionaryService.class);
        
        Map<String, Object> data = sa.getAllLast();
        
        for (String fullPath : data.keySet() ) {
            int firstSlash = fullPath.indexOf("/");
            String agentName = fullPath.substring(0,firstSlash);            
            DataProviderDictionary dict = ds.getDataProviderDictionaryForAgent(agentName);
            
            String path = fullPath.substring(firstSlash+1);
            DataProviderInfo di = dict.getDataProviderInfoForPath(path);

            //Monitoring state is published as metadata
            if ( di == null && path.endsWith("state") ) {
                String dataPath = path.replace("/state","");
                di = dict.getDataProviderInfoForPath(dataPath);
            } 

            System.out.println("Path: "+path+" "+di);
        }
        
    }
    */
    
    
    private static class Unknown implements Serializable {
        
    }
    
    public List<String> testCommandArgumentsReturnTypeValues() {
        return Arrays.asList(new String[] {"unknown", "exception", "string", "short_list", "long list", "int array", "list of args", "map_of_args"});
    }
    
    @Command(type = CommandType.ACTION)
    public void updateState(String componentName, DemoState state) {
        if("".equals(componentName)) {
            subsys.getAgentService(AgentStateService.class).updateAgentState(state);
        } else {
            Object componentToUpdate = subsys.getComponentLookup().getComponentByPath(componentName);
            if ( componentToUpdate == null ) {
                throw new IllegalArgumentException("No component corresponds to path "+componentName);
            }
            subsys.getAgentService(AgentStateService.class).updateAgentComponentState(componentToUpdate, state);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = 99, description = "Publish status messages for testing the messaging system. Any previously requested junk publishing is stopped.")
    public String testLoadStatusBus(
            @Argument(description = "Approximate size of each message, KB. 0 means stop publishing.", defaultValue = "1000") int size,
            @Argument(description = "Messages per second on each thread. 0 means stop publishing.", defaultValue = "1") int frequency,
            @Argument(description = "Keep publishing for the specified number of seconds. 0 means publish one message on each thread.", defaultValue = "0") int time,
            @Argument(description = "Number of threads.", defaultValue = "1") int threads) {

        if (junkPublisher != null) {
            junkPublisher.shutdownNow();
            junkPublisher = null;
        }

        if (size <= 0 || frequency <= 0 || threads <= 0) {
            return "Stopped publishing";
        }

        junkPublisher = Executors.newScheduledThreadPool(threads, r -> {
            Thread t = new Thread(r);
            t.setName("Junk Publisher");
            t.setDaemon(true);
            return t;
        });
        int period = (int) Math.round(1000. / frequency);
        Random random = new Random();
        int n = size * 1000;
        for (int i = 0; i < threads; i++) {
            byte[] payload = new byte[n];
            random.nextBytes(payload);
            Runnable run = new Runnable() {
                private final byte[] load = payload;
                private final long deadline = System.currentTimeMillis() + time * 1000;
                boolean first = true;
                private final AgentMessagingLayer ml = getMessagingAccess();

                @Override
                public void run() {
                    StatusData mess = new StatusData(new KeyValueData("junk", Arrays.copyOf(load, n)));
                    if (System.currentTimeMillis() >= deadline) {
                        if (first) {
                            ml.sendStatusMessage(mess);
                        }
                        junkPublisher.shutdownNow();
                    } else {
                        ml.sendStatusMessage(mess);
                    }
                    first = false;
                }
            };
            getLogger().info("Scheduling: "+ period);
            junkPublisher.scheduleAtFixedRate(run, (period / threads) * i, period, TimeUnit.MILLISECONDS);
        }
        return "OK";
    }

    @Command(type = Command.CommandType.ACTION, level = 99, description = "Split the JGroups cluster into N parts")
    public void testSplitCluster(int nSplits) {
        Set<String> allAgents = new HashSet<>();
        for ( AgentInfo ai : getMessagingAccess().getAgentPresenceManager().listConnectedAgents() ) {
            allAgents.add(ai.getName());
        }
        ClusterSplitUtils.splitCluster(nSplits, allAgents);
    }
    
}
