package org.lsst.ccs.subsystem.demo.main;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.messages.StatusAlert;
import org.lsst.ccs.bus.messages.StatusCommandDictionary;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlertSummary;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.StatusMessageListener;

public class SubsystemMemoryTest extends Subsystem implements HasLifecycle, StatusMessageListener  {
    
    private final Map<String,DataProviderDictionary> dataDictionary = new ConcurrentHashMap<>();
    private final Map<String,Map<String,Dictionary>> commandDictionary = new ConcurrentHashMap<>();
    private final Map<String,StateBundle> stateBundle = new ConcurrentHashMap<>();
    private final Map<String,ConfigurationInfo> configurationInfo = new ConcurrentHashMap<>();
    private final Map<String,Map<String,String>> configurationData = new ConcurrentHashMap<>();
    private final Map<String,RaisedAlertSummary> alertSummary = new ConcurrentHashMap<>();
    
    
    public SubsystemMemoryTest() {
        super("mem-test",AgentInfo.AgentType.SERVICE);
    }
    
    
    /**
     ***************************************************************************
     **
     ** Initializes the subsystem. *
     * **************************************************************************
     */
    @Override
    public void preStart() {
        getMessagingAccess().addStatusMessageListener(this);
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {
        String agentName = msg.getOriginAgentInfo().getName();
        stateBundle.put(agentName, msg.getState());
        if ( msg instanceof StatusDataProviderDictionary ) {
            dataDictionary.put(agentName, ((StatusDataProviderDictionary) msg).getDataProviderDictionary());
        } else if ( msg instanceof StatusCommandDictionary) {
            commandDictionary.put(agentName, ((StatusCommandDictionary) msg).getDictionary());
        } else if ( msg instanceof StatusConfigurationInfo ) {
            ConfigurationInfo ci = ((StatusConfigurationInfo) msg).getConfigurationInfo();
            configurationInfo.put(agentName,ci);
            Map<String,String> configData = new HashMap<>();
            for (ConfigurationParameterInfo parInfo : ci.getAllParameterInfo()) {
                configData.put(parInfo.getPathName(), parInfo.getConfiguredValue());
            }
            configurationData.put(agentName, configData);
        } else if ( msg instanceof StatusRaisedAlertSummary ) {
            alertSummary.put(agentName, ((StatusRaisedAlertSummary) msg).getRaisedAlertSummary());
        }
    }
    
    @Command(type = Command.CommandType.QUERY)
    public String printDataSummary() {
        StringBuilder sb = new StringBuilder();
        sb.append("Current sizes of stored data in Kb\n");
        sb.append("Data Dictionary: ").append(sizeOf((Serializable)dataDictionary)).append("\n");
        dataDictionary.entrySet().forEach((e) -> {
            sb.append("\t").append(e.getKey()).append(": ").append(sizeOf(e.getValue())).append("\n");
        });
        sb.append("Command Dictionary: ").append(sizeOf((Serializable)commandDictionary)).append("\n");
        commandDictionary.entrySet().forEach((e) -> {
            sb.append("\t").append(e.getKey()).append(": ").append(sizeOf((Serializable)e.getValue())).append("\n");
        });
        sb.append("StateBundle: ").append(sizeOf((Serializable)stateBundle)).append("\n");
        stateBundle.entrySet().forEach((e) -> {
            sb.append("\t").append(e.getKey()).append(": ").append(sizeOf((Serializable)e.getValue())).append("\n");
        });
        sb.append("ConfigurationInfo: ").append(sizeOf((Serializable)configurationInfo)).append("\n");
        configurationInfo.entrySet().forEach((e) -> {
            sb.append("\t").append(e.getKey()).append(": ").append(sizeOf((Serializable)e.getValue())).append("\n");
        });
        sb.append("Configuration Data: ").append(sizeOf((Serializable)configurationData)).append("\n");
        configurationData.entrySet().forEach((e) -> {
            sb.append("\t").append(e.getKey()).append(": ").append(sizeOf((Serializable)e.getValue())).append("\n");
        });
        sb.append("Alert Summary: ").append(sizeOf((Serializable)alertSummary)).append("\n");
        alertSummary.entrySet().forEach((e) -> {
            sb.append("\t").append(e.getKey()).append(": ").append(sizeOf((Serializable)e.getValue())).append("\n");
        });
        
        return sb.toString();
    }

    public static double sizeOf(Serializable object) {
        ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
        try (ObjectOutputStream objectOutputStream = new ObjectOutputStream(byteOutputStream)) {
            objectOutputStream.writeObject(object);
            objectOutputStream.flush();
        } catch (IOException ieo) {
            return 0;
        }

        return (double)byteOutputStream.toByteArray().length / 1024.;
    }


}
