package org.lsst.ccs.subsystem.demo.main;

import java.util.concurrent.TimeUnit;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.demo.bus.DemoData;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;
import hep.aida.*;
import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command.CommandCategory;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.scheduler.Scheduler;

public class DemoSubsystem implements HasLifecycle {
    
    public static enum DemoState {
        IDLE, BUSY;
    }
    
    public static enum OtherState {
        LOW, HIGH;
    }
    
    public static enum AState {
        A,B;
    }

    /**
     * Scheduler task that publishes heartbeat status messages.
     */
    protected PeriodicTask dataPublisher;

    //Create analysis factory
    private final IAnalysisFactory af = IAnalysisFactory.create();
    // Create datapoint set factory
    private final IDataPointSetFactory dataPointSetFactory = af.createDataPointSetFactory(null);
    // Create histogram factory
    private final IHistogramFactory histogramFactory = af.createHistogramFactory(null);

    private final Random r = new Random();

    private final ArrayList<PeriodicTask> tracerTestTasks = new ArrayList<>();

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    protected Subsystem subsys;
    
    private int dataPublishRate = 5;
    private int dataSize = 2;
    
    /**
     ***************************************************************************
     **
     ** Initializes the subsystem. *
     * **************************************************************************
     */
    @Override
    public void postStart() {
        // This must not be invoked in init because the subsystem is not yet 
        // connected on the buses.
        updateLimits("1.0", "2.0");
    }

    @Override
    public void postInit() {
        // Add AgentInfo properties
        subsys.getAgentService(AgentPropertiesService.class).setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY,"demo");
    }
    @Override
    public void build() {

        //Schedule a periodic task to publish demo data
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask("demoData-publish",
                        () -> {
                            subsys.publishSubsystemDataOnStatusBus(getDemoDataToPublish());
                        }).withIsFixedRate(false).withPeriod(Duration.ofSeconds(dataPublishRate)));
        
    }

    @Override
    public void init() {
        subsys.getAgentService(DataProviderDictionaryService.class).registerData(getDemoDataToPublish());
    }
    
    
    private KeyValueData getDemoDataToPublish() {
        DemoData dd = new DemoData(dataSize);
        KeyValueData d = new KeyValueData("demo_Data", dd);
        KeyValueDataList kvdl = new KeyValueDataList();
        kvdl.addData(d);
        return kvdl;
    }
    
    
    @Command(type = CommandType.ACTION, description = "update limits")
    public void updateLimits(String low, String high) {
        KeyValueDataList limits = new KeyValueDataList("demo_Data/temp");
        limits.addData("alarmLow", low, KeyValueData.KeyValueDataType.KeyValueMetaData);
        limits.addData("alarmHigh", high, KeyValueData.KeyValueDataType.KeyValueMetaData);
        subsys.publishSubsystemDataOnStatusBus(limits);
    }

    @Command(type = CommandType.QUERY, description = "raise custom alert")
    public void raiseCustomAlert(String alertId, AlertState severity, String cause) {
        Alert a = new Alert(alertId, "Custom Alert");
        subsys.getAgentService(AlertService.class).raiseAlert(a, severity, cause);
    }
    
    @Command(type = CommandType.QUERY, description = "Get Subsystem Data Sample")
    public DemoData getSampleData() {
        return new DemoData(dataSize);
    }

    @Command(type = CommandType.QUERY, description = "Set Subsystem Data Sample")
    public void setSampleDataSize(int size) {
        this.dataSize = size;
    }
    
    
    @Command(type = CommandType.QUERY, description = "Get Subsystem Array Sample")
    public Object[] getSampleArrayData() {
        Object[] data = new Object[]{"test", 2.3, new int[]{3, 2, 2, 2}};
        return data;
    }
    
    @Command(type = CommandType.QUERY, description = "Throw an Exception")
    public void throwAnException() {
        throw new RuntimeException("An Exception");
    }

    @Command(type = CommandType.ACTION, description = "Enable/Disable Publishing Tracer Test Messages")
    public void runTracerTest(boolean enable) {
        if (enable) {
            Scheduler s = subsys.getScheduler();
            tracerTestTasks.add(s.scheduleAtFixedRate(() -> subsys.getLogger().info("Self test on module " + r.nextInt(1000) + (r.nextInt(5) > 0 ? ": OK" : ": FAILED")), 0, 7, TimeUnit.SECONDS));
            tracerTestTasks.add(s.scheduleAtFixedRate(() -> subsys.getLogger().warn("London Bridge is in trouble, please repair."), 2, 15, TimeUnit.SECONDS));
            tracerTestTasks.add(s.scheduleAtFixedRate(() -> subsys.getLogger().error("Sun has gone supernova."), 3, 13, TimeUnit.SECONDS));
        } else {
            tracerTestTasks.forEach(task -> task.cancel(false));
            tracerTestTasks.clear();
        }
    }

    @Command(type = CommandType.ACTION, description = "Time out (action command)")
    public String testTimeout(int sleepSeconds) {
        if (sleepSeconds > 0) {
            try {
                Thread.sleep(sleepSeconds * 1000L);
            } catch (InterruptedException x) {
                return "Interrupted";
            }
        }
        return "Done.";
    }

    @Command(type = CommandType.QUERY, level = 0, category = CommandCategory.USER, description = "Empty level 0 category USER type QUERY command")
    public void testUserQuery0() {
    }

    @Command(type = CommandType.ACTION, level = 0, category = CommandCategory.USER, description = "Empty level 0 category USER type QUERY command")
    public void testUserAction0() {
    }

    @Command(type = CommandType.QUERY, level = 1, category = CommandCategory.USER, description = "Empty level 0 category USER type QUERY command")
    public void testUserQuery1() {
    }

    @Command(type = CommandType.QUERY, level = 0, category = CommandCategory.SYSTEM, description = "Empty level 0 category USER type QUERY command")
    public void testSystemQuery0() {
    }

    @Command(type = CommandType.QUERY, level = 1, category = CommandCategory.SYSTEM, description = "Empty level 0 category USER type QUERY command")
    public void testSystemQuery1() {
    }

    @Command(type = CommandType.QUERY, level = 0, description = "Test sending ACK/NACK. Delays are in seconds.", autoAck = false)
    public Object testCommand(
            @Argument(description = "If false, command is rejected with a NACK.")
            boolean accept,
            @Argument(defaultValue = "0")
            int delayBeforeAck,
            @Argument(description = "Delay between ACK and the result.", defaultValue = "0")
            int delayAfterAck,
            @Argument(defaultValue = "0")
            int timeout, 
            @Argument(defaultValue = "false")
            boolean returnException) {

        if (delayBeforeAck > 0) {
            try {
                Thread.sleep(delayBeforeAck * 1000L);
            } catch (InterruptedException x) {
                return "Interrupted while waiting to send ACK/NACK";
            }
        }

        if (accept) {
            if (timeout > 0) {
                subsys.sendAck(Duration.ofSeconds(timeout));
            } else {
                subsys.sendAck(null);
            }
        } else {
            subsys.sendNack("Sending NACK as requested.");
            return "NACK";
        }

        if (delayAfterAck > 0) {
            try {
                Thread.sleep(delayAfterAck * 1000L);
            } catch (InterruptedException x) {
                return "Interrupted while waiting to send the result";
            }
        }
        
        if (returnException) {
            return new RuntimeException("Requsted exception result.");
        } else {
            return "Done.";
        }
    }

    @Command(type = CommandType.QUERY, level = 0, description = "Test various types of command arguments and return types.")
    public Object testCommandArguments(
            @Argument(name = "namedBool", description = "boolean argument with name but no default")
            boolean b1,
            @Argument(defaultValue = "0", description = "unnamed int argument with default 0")
            int varName,
            @Argument(defaultValue = "Che", description = "String argument: nickname", name = "nickname")
            String nick,
            @Argument(description = "List of integers")
            List<Integer> intList,
            @Argument(description = "List of strings")
            List<String> stringList,
            @Argument(description = "Kind of object to return", allowedValueProvider = "testCommandArgumentsReturnTypeValues")
            String returnType) {
        
        switch (returnType) {
            case "unknown":
                return new Unknown();
            case "exception":
                return new RuntimeException("Exception returned.");
            case "string":
                return "Done";
            case "short_list":
                return Arrays.asList(new String[] {"11", "22", "33"});
            case "long list":
                return Arrays.asList(new String[] {"first moderately long line",
                                                   "shorty",
                                                   "second (kind of) long line with ()",
                                                   "The third, extremely long line that almost never ends. Contains a comma, too."});
            case "int array":
                return new int[] {0, 1, 2, 4};
            case "list of args":
                List<String> out = new ArrayList<>();
                out.add("b1 = "+ Boolean.toString(b1));
                out.add("varName = "+ Integer.toString(varName));
                out.add("nickname = "+ nick);
                out.add("intList = "+ String.join(",", intList.stream().map(i -> Integer.toString(i)).collect(Collectors.toList())));
                out.add("stringList = "+ String.join(",", stringList));
                out.add("returnType = "+ returnType);
                return out;
            case "map_of_args":
                HashMap<String, Object> map = new HashMap<>();
                map.put("b1", b1);
                map.put("varName", varName);
                map.put("nickname", nick);
                map.put("intList", intList);
                map.put("stringList", stringList);
                map.put("returnType", returnType);
                return map;
            default:
                return "Unexpected value for returnType:"+ returnType;
        }
    }
    
    private static class Unknown implements Serializable {
        
    }
    
    public List<String> testCommandArgumentsReturnTypeValues() {
        return Arrays.asList(new String[] {"unknown", "exception", "string", "short_list", "long list", "int array", "list of args", "map_of_args"});
    }
    
    @Command(type = CommandType.ACTION)
    public void updateState(String componentName, DemoState state) {
        if("".equals(componentName)) {
            subsys.getAgentService(AgentStateService.class).updateAgentState(state);
        } else {
            Object componentToUpdate = subsys.getComponentLookup().getComponentByPath(componentName);
            if ( componentToUpdate == null ) {
                throw new IllegalArgumentException("No component corresponds to path "+componentName);
            }
            subsys.getAgentService(AgentStateService.class).updateAgentComponentState(componentToUpdate, state);
        }
    }

}
