package org.lsst.ccs.subsystem.demo.main;

import java.util.concurrent.TimeUnit;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.demo.bus.DemoData;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;
import hep.aida.*;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command.CommandCategory;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.scheduler.Scheduler;

public class DemoSubsystem extends Subsystem implements HasLifecycle {
    
    public static enum DemoState {
        IDLE, BUSY;
    }
    
    public static enum OtherState {
        LOW, HIGH;
    }
    
    public static enum AState {
        A,B;
    }

    /**
     * Scheduler task that publishes heartbeat status messages.
     */
    protected PeriodicTask dataPublisher;

    //Create analysis factory
    private final IAnalysisFactory af = IAnalysisFactory.create();
    // Create datapoint set factory
    private final IDataPointSetFactory dataPointSetFactory = af.createDataPointSetFactory(null);
    // Create histogram factory
    private final IHistogramFactory histogramFactory = af.createHistogramFactory(null);

    private final Random r = new Random();

    private final ArrayList<PeriodicTask> tracerTestTasks = new ArrayList<>();

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    protected Subsystem subsys;
    
    @Persist
    private String somePersistStr = "abc";
    
    
    private int dataPublishRate = 5;
    private int dataSize = 2;
    
    private volatile long taskSleep = 0L;
    
    private volatile int numberOfExceptions = 0;

    @ConfigurationParameter(isBuild = true)
    private volatile String buildPar = "someValue";
    
    public DemoSubsystem() {
        super("demo-subsystem", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }
    
    /**
     ***************************************************************************
     **
     ** Initializes the subsystem. *
     * **************************************************************************
     */
    @Override
    public void postStart() {
        // This must not be invoked in init because the subsystem is not yet 
        // connected on the buses.
        updateLimits("1.0", "2.0");
    }

    @Override
    public void postInit() {
        // Add AgentInfo properties
        subsys.getAgentService(AgentPropertiesService.class).setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY,"demo");
        
        //set default Persistency strategy
        subsys.getAgentPersistenceService().setAutomatic(true,true);
    }
    @Override
    public void build() {

        //Schedule a periodic task to publish demo data
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask("demoData-publish",
                        () -> {
                            if ( numberOfExceptions > 0 ) {
                                numberOfExceptions--;
                                throw new RuntimeException("An Exception");
                            }
                            if ( taskSleep <=0 ) {
                                subsys.publishSubsystemDataOnStatusBus(getDemoDataToPublish("fixedRate"));
                            } else {
                                try {
                                    Thread.sleep(taskSleep);
                                } catch (InterruptedException e) {
                                    throw new RuntimeException(e);
                                }
                            }
                        }).withIsFixedRate(true).withPeriod(Duration.ofSeconds(dataPublishRate)));
        
        //Schedule a periodic task to publish demo data
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask("demoData-publish-fixedDelay",
                        () -> {
                            if ( taskSleep <=0 ) {
                                subsys.publishSubsystemDataOnStatusBus(getDemoDataToPublish("fixedDelay"));
                            } else {
                                try {
                                    Thread.sleep(taskSleep);
                                } catch (InterruptedException e) {
                                    throw new RuntimeException(e);
                                }
                            }
                        }).withIsFixedRate(false).withPeriod(Duration.ofSeconds(dataPublishRate)));
    }

    @Command(type = CommandType.ACTION) 
    public void setNumberOfExceptionInMonitoringThread(int nExeptions) {
        numberOfExceptions = nExeptions;
    }

    @Command(type = CommandType.ACTION) 
    public void setPeriodicTaskSleepTime(long sleep) {
        taskSleep = sleep;
    }
    
    @Override
    public void init() {
        subsys.getAgentService(DataProviderDictionaryService.class).registerData(getDemoDataToPublish("fixedRate"));
        subsys.getAgentService(DataProviderDictionaryService.class).registerData(getDemoDataToPublish("fixedDelay"));
    }
    
    
    private KeyValueData getDemoDataToPublish(String prefix) {
        DemoData dd = new DemoData(dataSize);
        KeyValueData d = new KeyValueData(prefix+"_demo_Data", dd);
        KeyValueDataList kvdl = new KeyValueDataList();
        kvdl.addData(d);
        return kvdl;
    }
    
    
    @Command(type = CommandType.ACTION, description = "update limits")
    public void updateLimits(String low, String high) {
        KeyValueDataList limits = new KeyValueDataList("demo_Data/temp");
        limits.addData("alarmLow", low, KeyValueData.KeyValueDataType.KeyValueMetaData);
        limits.addData("alarmHigh", high, KeyValueData.KeyValueDataType.KeyValueMetaData);
        subsys.publishSubsystemDataOnStatusBus(limits);
    }

    @Command(type = CommandType.QUERY, description = "raise custom alert")
    public void raiseCustomAlert(String alertId, AlertState severity, String cause) {
        Alert a = new Alert(alertId, "Custom Alert");
        subsys.getAgentService(AlertService.class).raiseAlert(a, severity, cause);
    }
    
    @Command(type = CommandType.QUERY, description = "Get Subsystem Data Sample")
    public DemoData getSampleData() {
        return new DemoData(dataSize);
    }

    @Command(type = CommandType.QUERY, description = "Set Subsystem Data Sample")
    public void setSampleDataSize(int size) {
        this.dataSize = size;
    }
    
    
    @Command(type = CommandType.QUERY, description = "Get Subsystem Array Sample")
    public Object[] getSampleArrayData() {
        Object[] data = new Object[]{"test", 2.3, new int[]{3, 2, 2, 2}};
        return data;
    }
    
    @Command(type = CommandType.QUERY, description = "Throw an Exception")
    public void throwAnException() {
        throw new RuntimeException("An Exception");
    }

    @Command(type = CommandType.ACTION, description = "Enable/Disable Publishing Tracer Test Messages")
    public void runTracerTest(boolean enable) {
        if (enable) {
            Scheduler s = subsys.getScheduler();
            tracerTestTasks.add(s.scheduleAtFixedRate(() -> subsys.getLogger().info("Self test on module " + r.nextInt(1000) + (r.nextInt(5) > 0 ? ": OK" : ": FAILED")), 0, 7, TimeUnit.SECONDS));
            tracerTestTasks.add(s.scheduleAtFixedRate(() -> subsys.getLogger().warn("London Bridge is in trouble, please repair."), 2, 15, TimeUnit.SECONDS));
            tracerTestTasks.add(s.scheduleAtFixedRate(() -> subsys.getLogger().error("Sun has gone supernova."), 3, 13, TimeUnit.SECONDS));
        } else {
            tracerTestTasks.forEach(task -> task.cancel(false));
            tracerTestTasks.clear();
        }
    }

    @Command(type = CommandType.ACTION, description = "Time out (action command)")
    public String testTimeout(int sleepSeconds) {
        if (sleepSeconds > 0) {
            try {
                Thread.sleep(sleepSeconds * 1000L);
            } catch (InterruptedException x) {
                return "Interrupted";
            }
        }
        return "Done.";
    }

    @Command(type = CommandType.QUERY, level = 0, category = CommandCategory.USER, description = "Empty level 0 category USER type QUERY command")
    public void testUserQuery0() {
    }

    @Command(type = CommandType.ACTION, level = 0, category = CommandCategory.USER, description = "Empty level 0 category USER type QUERY command")
    public void testUserAction0() {
    }

    @Command(type = CommandType.QUERY, level = 1, category = CommandCategory.USER, description = "Empty level 0 category USER type QUERY command")
    public void testUserQuery1() {
    }

    @Command(type = CommandType.QUERY, level = 0, category = CommandCategory.SYSTEM, description = "Empty level 0 category USER type QUERY command")
    public void testSystemQuery0() {
    }

    @Command(type = CommandType.QUERY, level = 1, category = CommandCategory.SYSTEM, description = "Empty level 0 category USER type QUERY command")
    public void testSystemQuery1() {
    }

    @Command(type = CommandType.QUERY, description = "Test sending ACK/NACK. Delays are in seconds.", autoAck = false)
    public Object testAck(boolean accept, int delayBeforeAck, int delayAfterAck, int timeout, boolean returnException) {

        if (delayBeforeAck > 0) {
            try {
                Thread.sleep(delayBeforeAck * 1000L);
            } catch (InterruptedException x) {
                return "Interrupted while waiting to send ACK/NACK";
            }
        }

        if (accept) {
            if (timeout > 0) {
                subsys.sendAck(Duration.ofSeconds(timeout));
            } else {
                subsys.sendAck(null);
            }
        } else {
            subsys.sendNack("Sending NACK as requested.");
            return "NACK";
        }

        if (delayAfterAck > 0) {
            try {
                Thread.sleep(delayAfterAck * 1000L);
            } catch (InterruptedException x) {
                return "Interrupted while waiting to send the result";
            }
        }
        return returnException ? new RuntimeException("Requsted exception result") : "Done.";
    }
    
    @Command(type = CommandType.ACTION)
    public void updateState(String componentName, DemoState state) {
        if("".equals(componentName)) {
            subsys.getAgentService(AgentStateService.class).updateAgentState(state);
        } else {
            Object componentToUpdate = subsys.getComponentLookup().getComponentByPath(componentName);
            if ( componentToUpdate == null ) {
                throw new IllegalArgumentException("No component corresponds to path "+componentName);
            }
            subsys.getAgentService(AgentStateService.class).updateAgentComponentState(componentToUpdate, state);
        }
    }

}
