package org.lsst.ccs.subsystem.demo.main;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;

/**
 * Example component to demonstrate how to manage configuration changes for a
 * component.
 *
 * @author emarin
 */
public class DemoConfigurable implements ConfigurationBulkChangeHandler {

    @LookupName
    protected String name;

    /**
     * Being final, this parameter can be set only via groovy or the safe
     * configuration. It is not part of any other configuration. The field
     * itself should NOT be set as final.
     */
    @ConfigurationParameter(isFinal = true)
    protected String info;

    @ConfigurationParameter
    protected double doubleValue;

    @ConfigurationParameter
    protected final Map<String, Integer> currentParms = new HashMap<>();

    /**
     * A simple range defined in the annotation. It will be checked at the
     * validation step.
     */
    @ConfigurationParameter(range = "0..10")
    protected int min;

    @ConfigurationParameter
    protected int max;

    public DemoConfigurable(String info, double doubleValue, Map currentParms, int min, int max) {
        this.info = info;
        this.doubleValue = doubleValue;
        this.currentParms.putAll(currentParms);
        this.min = min;
        this.max = max;
    }

    /**
     * When a configuration change occurs (single change, configuration load),
     * setting of the parameters happens by calling the following methods :
     */
    
    /**
     * FIRST STEP : VALIDATION (Optional). Where more advanced validation of
     * parameters can be performed. In the following, the constraint involves
     * two different parameters. If an exception is thrown, the configuration
     * change is interrupted and the configuration state is guaranteed to be
     * unchanged.
     *
     * @param parametersView a map of all parameters for that component, with
     * values corresponding to the proposed changes
     */
    @Override
    public void validateBulkChange(Map<String, Object> parametersView) {
        int mi = (int) parametersView.get("min");
        int ma = (int) parametersView.get("max");
        if (mi > ma) {
            throw new IllegalArgumentException("min must be less than max");
        }
    }

    /**
     * SECOND STEP : BULK SETTING (Optional). Invoked prior to any other setter
     * method. Parameters that need to have their value modified in order to
     * reach the requested configuration have a corresponding entry in the
     * passed map. It is the responsibility of the subsystem developer to set
     * the field correctly.
     *
     * @param parametersView a map of parameter name to their new value to
     * assign.
     */
    @Override
    public void setParameterBulk(Map<String, Object> parametersView) {
        Object obj = parametersView.get("min");
        if (obj != null) {
            this.min = (int) obj;
        }

        obj = parametersView.get("max");

        if (obj != null) {
            this.max = (int) obj;
        }
    }

    /**
     * THIRD STEP : Single Setting. using the
     * {@code @ConfigurationParameterChanger} annotation. It is called only if
     * the parameter wasn't already set in the setParameterBulk method.
     * Parameters that are still not set once all @ConfigurationParameterChanger
     * methods have been called are attempted to be set via reflection.
     *
     * @param currentParms
     */
    @ConfigurationParameterChanger
    public void setCurrentParms(Object currentParms) {
        Map<String, Integer> map = (Map<String, Integer>) currentParms;
        this.currentParms.putAll(map);
    }

    @ConfigurationParameterChanger
    public void setDoubleValue(double value) {
        this.doubleValue = value;
    }
}
