package org.lsst.ccs.subsystem.demo.gui.configurationview;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.Future;
import java.util.function.Predicate;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Generic view of the parameters set of DemoConfigurableSubsystem.
 * This Panel has no knowledge of the structure of the worker subsystem.
 * @author The LSST CCS Team
 */
public class SubsystemConfigurationViewPanel extends JPanel implements StatusMessageListener {
    
    private static final long serialVersionUID = 4483704670424943133L;
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.demo.gui.configuration");

    private final Map<String, ComponentPanel> componentPanelMap = new TreeMap<>();
    private final JPanel componentsPanel;

    private final JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
    private final JButton submitAllButton = new JButton("Submit All");
    private final JButton commitButton = new JButton("Commit All");
    private final AgentMessagingLayer aml;
    private final ConcurrentMessagingUtils cmu;
    private final String destination;
    private final Predicate<BusMessage<? extends Serializable, ?>> filterConfig;
    
    public SubsystemConfigurationViewPanel(AgentMessagingLayer aml, String dest){
        super(new BorderLayout());
        this.aml = aml;
        this.cmu = new ConcurrentMessagingUtils(aml);
        this.destination = dest;
        filterConfig = BusMessageFilterFactory.messageOrigin(dest).and(BusMessageFilterFactory.messageClass(StatusConfigurationInfo.class));
        // panel gathering the panels for each component
        componentsPanel = new JPanel();
        componentsPanel.setLayout(new BoxLayout(componentsPanel, BoxLayout.Y_AXIS));
        add(new JScrollPane(componentsPanel), BorderLayout.CENTER);

        buttonPanel.add(submitAllButton);
        add(buttonPanel, BorderLayout.SOUTH);
        
        submitAllButton.addActionListener(e -> {
            for (ComponentPanel cp : componentPanelMap.values()) {
                cp.submitChanges();
            }
        });
        
        commitButton.addActionListener(e -> new CommandWorker(this, new CommandRequest(destination, "commitBulkChange")).execute());
        buttonPanel.add(commitButton);
    }
    
    void initGui() {
        Future<Object> future = cmu.sendAsynchronousCommand(new CommandRequest(destination, "getConfigurationInfo"));
        try {
            ConfigurationInfo configInfo = (ConfigurationInfo)future.get();
            initializeWithConfigInfo(configInfo);
            aml.addStatusMessageListener(this, filterConfig);
        } catch (Exception ex){
            log.warn("unable to retrieve configuration information", ex);
        }
    }

    private void initializeWithConfigInfo(ConfigurationInfo configInfo) {
        // grouping components by the component they belong to
        Map<String, List<ConfigurationParameterInfo>> mapByComponentName = 
                new TreeMap<String, List<ConfigurationParameterInfo>>(
                        ConfigurationInfo.getParameterInfoGroupByComponent(configInfo.getAllParameterInfo()));
        
        for (Map.Entry<String, List<ConfigurationParameterInfo>> entry : mapByComponentName.entrySet()){
            ComponentPanel componentPanel = componentPanelMap.get(entry.getKey());
            if (componentPanel == null){
                componentPanel = new ComponentPanel(entry.getKey());
                componentPanelMap.put(entry.getKey(), componentPanel);
                componentsPanel.add(componentPanel);
            }
            componentPanel.initializeWithConfigInfo(entry.getValue());
        }
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {
        ConfigurationInfo configInfo = (ConfigurationInfo)msg.getObject();
        updateWithConfigInfo(configInfo);
    }
    
    void updateWithConfigInfo(ConfigurationInfo configInfo) {
         // grouping components by the component they belong to
        Map<String, List<ConfigurationParameterInfo>> mapByComponentName = 
                ConfigurationInfo.getParameterInfoGroupByComponent(configInfo.getAllParameterInfo());
        SwingUtilities.invokeLater(() -> {
            for(Map.Entry<String, List<ConfigurationParameterInfo>> e : mapByComponentName.entrySet()){
                componentPanelMap.get(e.getKey()).updateValues(e.getValue());
            }
        });
    }
    
    public void close() {
        aml.removeStatusMessageListener(this);
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                componentsPanel.removeAll();
                componentPanelMap.clear();
                revalidate();
                repaint();
            }
        });
    }
    
    /**
     * A panel gathering configuration parameters for a component.
     */
    private class ComponentPanel extends JPanel {
        private static final long serialVersionUID = 5653132723598295156L;
        private final String componentName;
        private final ConfigurationParameterTable table = new ConfigurationParameterTable();
        private final JButton submitChangesB = new JButton("submit");
        private final JButton resetSubmissionB = new JButton("reset");
        
        private ComponentPanel(String componentName){
            super(new BorderLayout());
            setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(Color.BLACK), componentName));
            this.componentName = componentName;
            add(new JScrollPane(table), BorderLayout.CENTER);
            JPanel buttonPanel = new JPanel();
            buttonPanel.setLayout(new GridBagLayout());
            GridBagConstraints gbc = new GridBagConstraints();
            //natural height, maximum width
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.gridx = 0; gbc.gridy = 0;
            buttonPanel.add(submitChangesB, gbc);
            gbc.gridx = 0; gbc.gridy = 1;
            buttonPanel.add(resetSubmissionB, gbc);
            add(buttonPanel, BorderLayout.EAST);
        }

        private void initializeWithConfigInfo(List<ConfigurationParameterInfo> parmList) {
            table.initialize(parmList);            
            // activate buttons
            submitChangesB.addActionListener(e -> submitChanges());
            
        }
        
        private void updateValues(List<ConfigurationParameterInfo> parmList) {
            table.updateValues(parmList);
        }
        
        private void submitChanges() {
            Map<String, Object> res = new HashMap<>();
//            for (ParameterWidgets parmWidgets : parametersMap.values()){
//                String text = parmWidgets.getTextField().getText();
//                if (!text.isEmpty()){
//                    res.put(parmWidgets.getParameterName(), text);
//                }
//            }
            new CommandWorker(this,
                    new CommandRequest(destination, "submitChanges", componentName, res)
            ).execute();
        }
        
    }
    
    /**
     * A SwingWorker that executes a command and displays the result in a 
     * simple dialog.
     */
    private class CommandWorker extends SwingWorker {

        private final Container parent;
        private final CommandRequest commandRequest;
        
        private CommandWorker(Container parent, CommandRequest commandRequest){
            this.parent = parent;
            this.commandRequest = commandRequest;
        }
        
        @Override
        protected Object doInBackground() throws Exception {
            Future<Object> future = cmu.sendAsynchronousCommand(commandRequest);
            return future.get();
        }
        
        protected void done(){
            try {
                Object res = get();
                String textRes = "";
                if (res == null) textRes = "OK : VOID";
                else if (res instanceof Exception) {
                    StringWriter stackTraceWriter = new StringWriter();
                    ((Throwable)res).printStackTrace(new PrintWriter(stackTraceWriter));
                    textRes = stackTraceWriter.toString();
                }
                else textRes = res.toString();
                JOptionPane.showMessageDialog(parent, textRes);
            } catch (Exception ex) {
                log.error(ex);
            }
        }
        
    }
}
