package org.lsst.ccs.subsystem.demo.gui.plugins;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.util.*;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.OptionalPage;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorView;
import org.lsst.ccs.gconsole.plugins.monitor.TreeView;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;

/**
 * Demo plugin that creates three sets of pages through the {@link OptionalPage} utility.
 * Each set can be activated through the "CCS Tools/Demo/Optional Page" menu.
 * See documentation for {@code OptPage1}, {@code OptPage2}, {@code OptPage3Monitor}, and {@code OptPage3Control}
 * classes implementing {@link OptionalPage} with different behaviors.
 *
 * @author onoprien
 */
@Plugin(name = "Optional Page Demo Plugin",
        id = "opt-page-demo",
        description = "Demo for adding optional pages.",
        loadAtStart = false)
public class OptionalPageDemoPlugin extends ConsolePlugin {

// -- Fields : -----------------------------------------------------------------
    
    private final OptionalPage opt1 = new OptPage1();
    
    private final OptionalPage opt2 = new OptPage2();
    private AbstractAction act2;
    
    private final OptionalPage opt3m = new OptPage3Monitor();
    private final OptionalPage opt3c = new OptPage3Control();

// -- Life cycle : -------------------------------------------------------------
    
    @Override
    public void initialize() {
        
        AbstractAction act = new AbstractAction("Page-per-subsystem") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ((Boolean)getValue(Action.SELECTED_KEY)) {
                    Console.getConsole().getOptionalPageService().add(opt1);
                } else {
                    Console.getConsole().getOptionalPageService().remove(opt1);
                }
            }
        };
        act.putValue(Action.SELECTED_KEY, false);
        getServices().addMenu(act, " CCS Tools ", "Demo", "Optional Page");
        
        act2 = new AbstractAction("Page-lists-all") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ((Boolean)getValue(Action.SELECTED_KEY)) {
                    Console.getConsole().getOptionalPageService().add(opt2);
                } else {
                    Console.getConsole().getOptionalPageService().remove(opt2);
                }
            }
        };
        act2.putValue(Action.SELECTED_KEY, false);
        getServices().addMenu(act2, " CCS Tools ", "Demo", "Optional Page");
        
        act = new AbstractAction("Two-for-each-subsystem") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ((Boolean)getValue(Action.SELECTED_KEY)) {
                    Console.getConsole().getOptionalPageService().add(opt3m);
                    Console.getConsole().getOptionalPageService().add(opt3c);
                } else {
                    Console.getConsole().getOptionalPageService().remove(opt3m);
                    Console.getConsole().getOptionalPageService().remove(opt3c);
                }
            }
        };
        act.putValue(Action.SELECTED_KEY, false);
        getServices().addMenu(act, " CCS Tools ", "Demo", "Optional Page");

    }
    
// -- Demo 1: Open a page for every sussystem on the buses : -------------------

    /**
     * Displays one page per subsystem present on the buses.
     * Automatically opened on subsystem discovery, closed and removed from menus on disconnect.
     * Pages state (open or closed) is automatically saved or restored when the console session is saved or restored.
     * <p>
     * Subsystem named "demo-subsystem" gets special treatment.
     * On subsystem disconnect, it stays open until closed by the user.
     * It is also placed last in the menu.
     */
    static class OptPage1 implements OptionalPage {

        @Override
        public boolean isAutoSave() {
            return true;
        }

        @Override
        public String getPage(AgentInfo agent) {
            StringBuilder sb = new StringBuilder("CCS Subsystems/Demo/Page-per-subsystem");
            if (agent.getName().equals("demo-subsystem")) {
                sb.append(":2");
            }
            return sb.append("/").append(agent.getName()).toString();
        }

        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
            Box box = Box.createHorizontalBox();
            box.add(Box.createHorizontalGlue());
            JLabel label = new JLabel(agent.getName());
            box.add(label);
            box.add(Box.createHorizontalGlue());
            return box;
        }

        @Override
        public boolean agentDisconnected(AgentInfo agent) {
            return ! agent.getName().equals("demo-subsystem");
        }
        
    }
    
// -- Demo 2: Page lists all sussystems on the buses except monitor-test : -----

    /**
     * Displays a single page that lists all connected subsystems.
     * The page remains available regardless of whether there are any subsystems present.
     * Saving and restoring the state of this page is handled by this plugin, not by the framework.
     */    
    static class OptPage2 implements OptionalPage {
        
        private final JLabel label = new JLabel();
        private final ArrayList<String> agents = new ArrayList<>();

        @Override
        public String getPage(AgentInfo agent) {
            return "CCS Subsystems/Demo/Page-lists-all";
        }

        @Override
        public JComponent open(AgentInfo agent, JComponent component) {
            if (component == null) {
                component = new JPanel(new BorderLayout());
                component.add(label, BorderLayout.CENTER);
            }
            if (agent != null) {
                agents.add(agent.getName());
            }
            updateLabel();
            return component;
        }

        @Override
        public void closed(String page) {
            agents.clear();
        }

        @Override
        public boolean agentDisconnected(AgentInfo agent) {
            agents.remove(agent.getName());
            updateLabel();
            return false;
        }
        
        private void updateLabel() {
            StringBuilder sb = new StringBuilder("<html><h4>Connected subsystems:</h4>");
            agents.forEach(a -> sb.append(a).append("<p>"));
            label.setText(sb.toString());
        }
        
    }

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        try {
            Descriptor d = (Descriptor) storageBean;
            boolean installed = d.isInstalled();
            if (installed != (boolean) act2.getValue(Action.SELECTED_KEY)) {
                act2.putValue(Action.SELECTED_KEY, installed);
                act2.actionPerformed(null);
            }
            if (installed) {
                opt2.restore(d.getOpen());
            }
        } catch (ClassCastException x) {
        }
        return true;
    }

    @Override
    public Descriptor save() {
        Descriptor d = new Descriptor();
        d.setInstalled((Boolean) act2.getValue(Action.SELECTED_KEY));
        d.setOpen(opt2.save());
        return d;
    }
    
    static public class Descriptor extends ComponentDescriptor {

        private boolean installed;
        private HashMap<String,DataPanelDescriptor> open;

        public boolean isInstalled() {
            return installed;
        }

        public void setInstalled(boolean installed) {
            this.installed = installed;
        }

        public HashMap<String,DataPanelDescriptor> getOpen() {
            return open;
        }

        public void setOpen(HashMap<String,DataPanelDescriptor> open) {
            this.open = open;
        }
        
    }
    
// -- Demo 3: two types of pages for each subsystem : --------------------------
    
    static class OptPage3Monitor implements OptionalPage {
        
        private HashMap<String,MonitorView> views = new HashMap<>();
//        private final HashMap<String,JLabel> labels = new HashMap<>();

        @Override
        public boolean isAutoOpen() {
            return false;
        }

        @Override
        public boolean isAutoSave() {
            return true;
        }

        @Override
        public String getPage(AgentInfo agent) {
            return "CCS Subsystems/Demo/Two-for-each-subsystem/"+ agent.getName() +"/Monitor";
        }

        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
            
            MonitorView view = new TreeView();
            AgentChannelsFilter filter = new AbstractChannelsFilter() {
                @Override
                public List<String> getAgents() {
                    return Collections.singletonList(agent.getName());
                }
            };
            view.setFilter(filter);
            view.setName(agent.getName() +" monitor");
            views.put(getPage(agent), view);
            return view.getPanel();
            
//            Box box = Box.createHorizontalBox();
//            box.add(Box.createHorizontalGlue());
//            JLabel label = new JLabel(agent.getName() +" monitor");
//            labels.put(agent.getName(), label);
//            box.add(label);
//            box.add(Box.createHorizontalGlue());
//            box.setName(agent.getName() +" monitor");
//            return box;
        }

        @Override
        public void opened(String page) {
            MonitorView view = views.get(page);
            if (view != null) {
                view.install();
            }
        }

        @Override
        public void closed(String page) {
            MonitorView view = views.remove(page);
            if (view != null) {
                view.uninstall();
            }
        }

        @Override
        public boolean agentDisconnected(AgentInfo agent) {
//            JLabel label = labels.get(agent.getName());
//            if (label != null) {
//                label.setText(agent.getName() +" monitor (disconnected)");
//            }
            return false;
        }
        
    }
    
    static class OptPage3Control implements OptionalPage {
        
        private final HashMap<String,JLabel> labels = new HashMap<>();

        @Override
        public boolean isAutoOpen() {
            return false;
        }

        @Override
        public boolean isAutoSave() {
            return true;
        }

        @Override
        public String getPage(AgentInfo agent) {
            return "CCS Subsystems/Demo/Two-for-each-subsystem/"+ agent.getName() +"/Control";
        }

        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
            JLabel label;
            if (existingComponent == null) {
                existingComponent = Box.createHorizontalBox();
                existingComponent.add(Box.createHorizontalGlue());
                label = new JLabel(agent.getName() + " control");
                existingComponent.add(label);
                existingComponent.add(Box.createHorizontalGlue());
                existingComponent.setName(agent.getName() + " control");
            } else {
                existingComponent.removeAll();
                existingComponent.add(Box.createHorizontalGlue());
                label = new JLabel(agent.getName() + " control (re-connected)");
                existingComponent.add(label);
                existingComponent.add(Box.createHorizontalGlue());
                existingComponent.setName(agent.getName() + " control");
            }
            labels.put(getPage(agent), label);
            return existingComponent;
        }

        @Override
        public void closed(String page) {
            labels.remove(page);
        }

        @Override
        public boolean agentDisconnected(AgentInfo agent) {
            JLabel label = labels.get(getPage(agent));
            if (label != null) {
                label.setText(agent.getName() +" control (disconnected)");
            }
            return false;
        }
        
    }

}
