package org.lsst.ccs.subsystem.demo.gui;

import java.awt.Component;
import java.awt.Point;
import java.time.Duration;
import java.util.Arrays;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.command.BasicCommand;
import org.lsst.ccs.command.RawCommand;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.ui.CommandSender;
import org.lsst.ccs.subsystem.monitor.ui.MonitorAssembly;

public class DemoSubsystemWithMonitoringGUI implements CommandSender, StatusMessageListener, AgentPresenceListener, HasPopupItems {

    private final String subsys = "demo-subsystem-with-monitoring";
    private final DemoSubsystemWithMonitoringAssembly assembly;
    private final MonitorAssembly mon;
    private volatile boolean enabled = false;
    private ConcurrentMessagingUtils messagingUtils;
    private final AgentMessagingLayer layer;

    public DemoSubsystemWithMonitoringGUI(AgentMessagingLayer layer ) {
        this.layer = layer;
        try {
            assembly = new DemoSubsystemWithMonitoringAssembly(subsys, this);
            mon = assembly.getMonitor();
        } catch (Exception x) {
            throw new RuntimeException(x);
        }
    }

    public void initGui() {
        mon.setSubsystems(subsys);
        messagingUtils = new ConcurrentMessagingUtils(layer);        
        layer.getAgentPresenceManager().addAgentPresenceListener(this);
    }

    public JComponent getGuiLayout() {
        return assembly;
    }

    public void resetGui() {
    }

    /**
     * @param args the command line arguments
     * @throws  Exception
     */
    public static void main(String[] args) throws Exception {
        Agent guiAgent = new Agent("aSomething", AgentInfo.AgentType.LISTENER); 
        guiAgent.startAgent();
        DemoSubsystemWithMonitoringGUI t = new DemoSubsystemWithMonitoringGUI(guiAgent.getMessagingAccess());
        JFrame frame = new JFrame("Main Gui");
        frame.setContentPane(t.getGuiLayout());
        frame.pack();
        frame.setVisible(true);
        t.initGui();
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

    @Override
    public void onStatusMessage(StatusMessage s) {
        StatusSubsystemData ssd = (StatusSubsystemData)s;
        mon.handleData(subsys, ssd.getDataKey(), ssd.getSubsystemData());
    }

    

    @Override
    public void connecting(AgentInfo agent) {
        if (agent.getName().equals(subsys)) {
            enable();
        }
    }

    @Override
    public void disconnecting(AgentInfo agent) {
        if (agent.getName().equals(subsys)) {
            disable();
        }
    }

    @Override
    public String sendCommand(String dest, String target, String cmnd,
                              Object... args) {
        sendCmnd(dest, target, cmnd, args);
        return "";
    }

    private void enable() {
        if (enabled) return;
        enabled = true;
        Object reply = sendCmnd(null, null, "getMonitorFullState");
        if ( reply instanceof MonitorFullState ) {
            mon.updateTableModel(subsys, (MonitorFullState)reply);
        }
        else {
            enabled = false;
        }
        layer.addStatusMessageListener(this, BusMessageFilterFactory.messageOrigin(subsys).and(BusMessageFilterFactory.messageClass(StatusSubsystemData.class)));
    }

    private void disable() {
        if (!enabled) return;
        enabled = false;
        mon.disableSystem(subsys);
        layer.removeStatusMessageListener(this);
    }

    private Object sendCmnd(String dest, String target, String cmnd,
                            Object... args) {
        String dst = (dest == null ? subsys : dest)
                       + (target == null ? "" : "/" + target);
        BasicCommand cmd = new RawCommand(cmnd,Arrays.asList(args));
        CommandRequest r = new CommandRequest(dst,cmd);
        try {
            return messagingUtils.sendSynchronousCommand(r, Duration.ofMillis(5000));
        }
        catch (Exception e) {
            return null;
        }
    }

    @Override
    public JPopupMenu modifyPopupMenu(JPopupMenu jpm, Component cmpnt, Point point) {
        System.out.println("popup");
        jpm.add(new JMenuItem("popup"));
        return jpm;
    }

}
