package org.lsst.ccs.subsystem.demo.gui.plugins.trending;

import java.time.Instant;
import java.time.ZoneId;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.util.*;
import org.lsst.ccs.gconsole.plugins.trending.Trend;
import org.lsst.ccs.gconsole.plugins.trending.TrendData;
import org.lsst.ccs.gconsole.plugins.trending.TrendingChannel;
import org.lsst.ccs.gconsole.plugins.trending.TrendingSource;

/**
 *
 * @author onoprien
 */
public class DemoSource extends TrendingSource {

// -- Fields : -----------------------------------------------------------------
    
    private final TrendingChannel channel1 = new TrendingChannel("Extra/Minutes/OfDay", "Minute of a day");
    private final TrendingChannel channel2 = new TrendingChannel("Extra/Minutes/OfHour", "Minute of an hour");
    private final TrendingChannel channel3 = new TrendingChannel("Extra/ScaleTest", "Scale Test");
    private final TrendingChannel channelDelay5 = new TrendingChannel("Delayed/5 seconds", "5 seconds delay");
    private final TrendingChannel channelDelay8 = new TrendingChannel("Delayed/8 seconds", "8 seconds delay");
    private final TrendingChannel channelMeta = new TrendingChannel("WithMetadata/sin");
    private final TrendingChannel channelMetaDelay = new TrendingChannel("WithMetadata/sinDelay5sec");

// -- Life cycle : -------------------------------------------------------------

// -- Implementing TrendingSource: ---------------------------------------------

    @Override
    public List<TrendingChannel> getChannels() {
        ArrayList<TrendingChannel> out = new ArrayList<>();
        Collections.addAll(out, channel1, channel2, channel3, channelDelay5, channelDelay8, channelMeta, channelMetaDelay);
        return out;
    }

    @Override
    public TrendData get(TrendingChannel channel, long begin, long end, EnumSet<Trend.Meta> metadata, TrendData history) {
        if (end <= begin) return null;
        if (channel1.equals(channel) || channel2.equals(channel)) {
            return get12test(channel, begin, end);
        } else if (channelDelay5.equals(channel)) {
            try {Thread.sleep(5000L);} catch (InterruptedException x) {}
            return get12(channel1, begin, end);
        } else if (channelDelay8.equals(channel)) {
            try {Thread.sleep(8000L);} catch (InterruptedException x) {}
            return get12(channel2, begin, end);
        } else if (channelMeta.equals(channel) || channelMetaDelay.equals(channel)) {
            return getMeta(begin, end, channelMeta.equals(channel) ? 0L : 5000L);
        } else {
            return get3(begin, end);
        }
    }
    
    private TrendData get12(TrendingChannel channel, long begin, long end) {
        long[] time = new long[10];
        double[] value = new double[10];
        long delta = (end-begin)/10L;
        ChronoField field = channel.equals(channel1) ? ChronoField.MINUTE_OF_DAY : ChronoField.MINUTE_OF_HOUR;
        for (int i=0; i<10; i++) {
            long t = begin + i*delta;
            time[i] = t;
            value[i] = Instant.ofEpochMilli(t).atZone(ZoneId.systemDefault()).get(field);
        }
        Map<String, long[]> times = new HashMap<>();
        times.put("value", time);
        Map<String, double[]> values = new HashMap<>();
        values.put("value", value);
        return new TrendData(times, values, new long[] {begin, end});
    }
    
    private TrendData get12test(TrendingChannel channel, long begin, long end) {
        
        ArrayList<Long> t = new ArrayList();
        ArrayList<Double> v = new ArrayList();
        long now = System.currentTimeMillis();
        long x = begin;
        while(x < now) {
            t.add(x);
            v.add(Math.sin(x/10.));
            x += ChronoUnit.MINUTES.getDuration().toMillis();
        }
        long[] time = new long[t.size()];
        double[] value = new double[t.size()];
        for (int i=0; i<t.size(); i++) {
            time[i] = t.get(i);
            value[i] = v.get(i);
        }
        Map<String, long[]> times = new HashMap<>();
        times.put("value", time);
        Map<String, double[]> values = new HashMap<>();
        values.put("value", value);
        return new TrendData(times, values, new long[] {begin, end});
    }
    
    private TrendData get3(long begin, long end) {
        long range = end - begin;
        int n;
        long t0;
        if (range < 10*60*1000) {
            t0 = begin;
            n = (int) (range/10L);
        } else {
            t0 = begin + range/3L;
            n = 3*60*100;
        }
        long[] time = new long[n];
        double[] value = new double[n];
        for (int i=0; i<n; i++) {
            long t = t0 + i*10L;
            time[i] = t;
            value[i] = 10.*Math.sin((t-t0)/60000.) + Math.sin((t-t0)/1000.);
        }
        Map<String, long[]> times = new HashMap<>();
        times.put("value", time);
        Map<String, double[]> values = new HashMap<>();
        values.put("value", value);
        return new TrendData(times, values, new long[] {begin, end});
    }
    
    private TrendData getMeta(long begin, long end, long delay) {
        if (delay > 0L) {
            try {Thread.sleep(delay);} catch (InterruptedException x) {}
        }
        int n = 100;
        long period = 60000L;
        long range = end - begin;
        long delta = range/n;
        
        long[] time = new long[n];
        double[] value = new double[n];
        double[] min = new double[n];
        double[] max = new double[n];
        double[] rms = new double[n];
        
        for (int i=0; i<n; i++) {
            long t = begin + i*delta;
            time[i] = t;
            value[i] = Math.sin(t*2*Math.PI/period) + 2.;
            min[i] = value[i] - 1;
            max[i] = value[i] + 1;
            rms[i] = value[i] * .2;
        }
        Map<String, long[]> times = new HashMap<>();
        times.put("value", time);
        Map<String, double[]> values = new HashMap<>();
        values.put("value", value);
        values.put("min", min);
        values.put("max", max);
        values.put("rms", rms);
        
        long[] timeMetaOff = new long[] {begin, begin + range/2, begin + range/2, end};
        double[] low = new double[] {1., 1., .5, .5};
        double[] high = new double[] {3., 3., 3.5, 3.5};
        times.put("warningLow", timeMetaOff);
        values.put("warningLow", low);
        times.put("warningHigh", timeMetaOff);
        values.put("warningHigh", high);
        
        timeMetaOff = new long[] {begin, end};
        low = new double[] {0., 0.};
        high = new double[] {4., 4.};
        times.put("alarmLow", timeMetaOff);
        values.put("alarmLow", low);
        times.put("alarmHigh", timeMetaOff);
        values.put("alarmHigh", high);
        
        return new TrendData(times, values, new long[] {begin, end});
    }
    
}
