package org.lsst.ccs.subsystem.demo.gui.plugins;

import java.awt.event.ActionEvent;
import java.util.concurrent.TimeUnit;
import javax.swing.AbstractAction;
import javax.swing.Action;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

/**
 * Demo plugin - saving and restoring configuration.
 *
 * @author onoprien
 */
@Plugin
public class ConfigurationPlugin extends ConsolePlugin {
    
    /** Current configuration of this plugin. */
    private volatile int state;
    private PeriodicTask stateChanger;

    /**
     * Called by the framework when the console starts, to initialize this plugin..
     */
    @Override
    public void initialize() {
        
        getServices().addProperty("restoreOnStartup", false);
        getServices().addPreference(new String[] {"LSST", "Demo", "Config"}, null, "${restoreOnStartup} Restore on startup.");
        
        Action act = new AbstractAction("Keep changing state") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (stateChanger == null) {
                    stateChanger = getConsole().getScheduler().scheduleAtFixedRate(() -> System.out.println("State changed to "+ ++state), 5, 5, TimeUnit.SECONDS);
                } else {
                    stateChanger.cancel(true);
                    stateChanger = null;
                }
            }
        };
        act.putValue(Action.SELECTED_KEY, false);
        getServices().addMenu(act, "Demo Plugins", "Configuration");
    }

    /**
     * Called by the framework when the console starts, after all plugins have been initialized.
     */
    @Override
    public void start() {
        state = 0;
        if ((Boolean) getServices().getProperty("restoreOnStartup")) {
            restore(getServices().getDescriptor().getConfiguration(), true);
        }
    }

    /**
     * Called by the framework to restore this plugin configuration.
     * 
     * @param storageBean JavaBean that holds configuration information.
     * @param lastRound {@code True} if this is the last call to this method, regardless of the return value.
     * @return {@code False} if this method should be called again in the next round.
     */
    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        Config config = (Config) storageBean;
        state = config.getConfigParameter();
        System.out.println("Restored state to "+ state);
        return true;
    }

    /**
     * Called by the framework when this plugin configuration needs to be saved.
     * 
     * @return JavaBean that holds information on the current configuration of this plugin.
     */
    @Override
    public ComponentDescriptor save() {
        Config configBean = new Config(getServices().getDescriptor());
        configBean.setConfigParameter(state);
        return configBean;
    }
    
    /**
     * Bean that holds plugin configuration information.
     */
    static public class Config extends ComponentDescriptor {
        
        public Config() {
        }
        
        public Config(ComponentDescriptor seed) {
            super(seed);
        }

        private int configParameter;

        public int getConfigParameter() {
            return configParameter;
        }

        public void setConfigParameter(int configParameter) {
            this.configParameter = configParameter;
        }
        
    }
    
}
