package org.lsst.ccs.subsystem.daq;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Folder;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.daq.ims.Source;
import org.lsst.ccs.daq.ims.SourceMetaData;
import org.lsst.ccs.daq.ims.Store;
import org.lsst.ccs.daq.ims.Utils;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/* Pre-defined Command levels */
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;

/**
 *  Device code for managing files in Daq store
 *
 *  @author Al Eisner
 */
public class DaqStoreManageDevice extends Device {

   /**
    *  Enumeration of monitoring channels
    */

    public enum MonChan {
        DAQ_CAPACITY,            // File storage space in bytes
	DAQ_REMAINING,           // Unused file storage space in bytes
	DAQ_FREE_FRACTION;       // Fraction of capacity unused
    }

   /**
    *  Data fields.
    */

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    private static final Logger LOG = Logger.getLogger(DaqStoreManageDevice.class.getName());
    private static final Pattern PATH_PATTERN = Pattern.compile("([0-9a-zA-Z\\-\\_]*)/?([0-9a-zA-Z\\-\\_]*)");
    private final double DAQ_UNITS = 1.0E9;  // bytes unit for monitoring
    private String partition;
    private Store store;

   /**
    *  Private lookup maps for driver read monitor reads.  The default
    *  initial capacity of 16 is in excess of the expected number of
    *  channels.  The key is a numeric identifier of a monitor channel
    *  defined in the groovy file.
    *
    *  chanMap identifies the quantity to be read (i.e., the query name).
    *  It is taken from the type parameter in the groovy file.
    */
    private Map<Integer, String> chanMap = new HashMap<>();
    private Integer chanKey;

    /**
     *  Configuration Parameters 
     */

    @ConfigurationParameter(isFinal = true, category = "Store",
                            description = "top-level folder for DAQ store")
    private volatile String daqFolder;

    @ConfigurationParameter(category = "Store",
      description = "fraction of free disk space to trigger a store purge")
    private volatile Double purgeThreshold;

    @ConfigurationParameter(category = "Store",
      description = "goal of purge for minimum fraction of free disk space")
    private volatile Double purgeTarget;

    @ConfigurationParameter(category = "Store",
                            description="enable automatic purge of DAQ store")
    private volatile Boolean enableAutoPurge;

   /**
    *  Set partition name for DAQ connection (before init() method)
    *
    *  @param pname  String partition name
    */
    void setPartition(String pname) {
        partition = pname;
    }

   /**
    *  Performs basic initialization of Device code.
    *  Verify that all ConfigurationParameters are set.
    */
    @Override
    protected void initDevice() {

        chanKey = 0;

        if (daqFolder == null) {
            ErrorUtils.reportConfigError(LOG, name, "daqFolder", "is missing");
        }
        if (purgeThreshold == null) {
            ErrorUtils.reportConfigError(LOG, name, "purgeThreshold", "is missing");
        }
        if (purgeTarget == null) {
            ErrorUtils.reportConfigError(LOG, name, "purgeTarget", "is missing");
        }
        if (enableAutoPurge == null) {
            ErrorUtils.reportConfigError(LOG,name,"enableAutoPurge",
                                         "value is missing");
        }
    }

   /**
    *  Connect to partition and perform any needed initialization.
    */
    @Override
    protected void initialize() {

        fullName = name + ": (partition " + partition + ")";
        try {
            if (store != null) {
                store.close();
            }
            store = new Store(partition);
            setOnline(true);
            initSensors();
            LOG.info("\n Connected to " + fullName);
	}
        catch (DAQException ex) {
            if (!inited) {
                throw new RuntimeException("DAQException while connecting to or initializing " + fullName + ": " + ex.getMessage());
            }
            close();
        }
        inited = true;
    }
   /**
    *  Closes the connection.
    */
    @Override
    protected void close() {
        try {
            if (store != null) {
                store.close();
                store = null;
            }
        }
        catch (DAQException ex) {
            throw new RuntimeException("DAQException while closing " + fullName + ": " + ex.getMessage());
        }
    }

   /**
    *  Make sure Store is connected
    */

    private void checkStore() {
        if (store == null) {
            throw new RuntimeException("Please connect to store first");
        }
    }

   /**
    *  Set fraction of available disk space which triggers a file purge
    *
    *  @param   fraction
    *  @throws  DAQException
    */
    @ConfigurationParameterChanger(propertyName = "purgeThreshold")
    public void setPurgeThreshold(Double fraction) throws DAQException
    {
        if (isOnline()) {
            if (fraction != null) {
                log.info(name + "config. change request: set PURGE_THRESHOLD to " + Double.toString(fraction));
                purgeThreshold = fraction;
            } else {
                throw new IllegalArgumentException(name + ": improper value for purgeThreshold, not changed");
            }
        } else {                      // initialization call
            if (fraction != null) {purgeThreshold = fraction;}
        }
    }

   /**
    *  Set fraction of available disk space which is goal of a file purge
    *
    *  @param   fraction
    *  @throws  DAQException
    */
    @ConfigurationParameterChanger(propertyName = "purgeTarget")
    public void setPurgeTarget(Double fraction) throws DAQException
    {
        if (isOnline()) {
            if (fraction != null) {
                log.info(name + "config. change request: set PURGE_TARGET to " + Double.toString(fraction));
                purgeTarget = fraction;
            } else {
                throw new IllegalArgumentException(name + ": improper value for purgeTarget, not changed");
            }
        } else {                      // initialization call
            if (fraction != null) {purgeTarget = fraction;}
        }
    }

   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  chName   The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String chName, int hwChan, String type, 
                                 String subtype) throws Exception
    {
        try {
            MonChan.valueOf(type);
        }
        catch (IllegalArgumentException e) {
            ErrorUtils.reportChannelError(LOG, chName, "type", type);
        }
        chanKey++;
        chanMap.put(chanKey,type);
        return new int[]{chanKey, 0};
    }

   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);   //NaN
        String  chanName = chanMap.get(type);
        double capacity;
        double remaining;
        try {
            capacity = (double) store.getCapacity();
            remaining = (double) store.getRemaining();
        }
        catch (DAQException e) {
            throw new RuntimeException("DAQException while reading file storage space " + fullName + ": " + e.getMessage());
        }
        if (chanName.equals("DAQ_CAPACITY")) {
	    value = capacity/DAQ_UNITS;
            if (capacity <= 0.) {value = Double.NaN;}
        } else if (chanName.equals("DAQ_REMAINING")) {
            value = remaining/DAQ_UNITS;
        } else if (chanName.equals("DAQ_FREE_FRACTION")) {
            if (capacity <= 0.) {
                value = Double.NaN;
            } else {
                value = remaining/capacity;
            }
        }
        return value;
    }

   /**
    *  Utilities
    */

    private Image imageFromPath(Matcher matcher) throws DAQException {
        Folder folder = store.getCatalog().find(matcher.group(1));
        if (folder == null) {
            throw new RuntimeException("No such folder: " + matcher.group(1));
        }
        Image image = folder.find(matcher.group(2));
        if (image == null) {
            throw new RuntimeException("No such image: " + matcher.group(2));
        }
        return image;
    }

    // private String imageSize(Image image) throws DAQException {
    //     List<Source> sources = image.listSources();
    //     long totalSize = 0;
    //     int nBad = 0;
    //     for (Source source : sources) {
    //         try {
    //             totalSize += source.getMetaData().getLength();
    //         } catch (DAQException x) {
    //             nBad++;
    //         }
    //     }
    //     return String.format("%s (%d sources (%d bad))", Utils.humanReadableByteCount(totalSize), sources.size(), nBad);
    // }

    private double spaceUsage() throws DAQException {
        double capacity = (double) store.getCapacity();
        double remaining = (double) store.getRemaining();
        return 1. - remaining/capacity;
    }

   /**
    *  List information on contents of partition, a specific folder within
    *  partition, or images in a specific file within that folder.  Code 
    *  largely taken from example/CommandTool.java in DAQ driver package.
    *
    *  @param  String path  optional - if persent, denotes folder or
    *                       folder/file
    *  #return String       String with "\n" delimiters
    *
    *  @throws DAQException
    */
    @Command(type=Command.CommandType.QUERY, name="list", alias="ls", 
             level=NORMAL,
             description="List DAQ partition contents", timeout=30)
    public String list(@Argument(name="path",
     description="<default=\"\"|folder|folder/file>", 
     defaultValue = "") String path) throws DAQException {
        checkStore();
        return Utils.list(store, path).collect(Collectors.joining("\n"));
    }

   /**
    *  Purge older DAQ files in daqFolder until total fractional space 
    *  usage is below purgeTarget.
    *
    *  NOTE:  this method is not protected against store capacity (or
    *  remaining) being returned as 0, a rare problem which will hoptefully
    *  get fixed before long.  (05-Jun-2020)
    *
    *  @throws DAQException
    */

    synchronized void purge() throws DAQException {
        checkStore();
        Folder folder = store.getCatalog().find(daqFolder);
        if (folder == null) {
            throw new IllegalArgumentException("Invalid folder: " + folder);
        }
        double fracBefore = spaceUsage();
        LOG.info(name + " purge requested, utilized space fraction = "
                 + Double.toString(fracBefore));
        List<Image> images = folder.listImages();
        images.sort((Image i1, Image i2) -> i1.getMetaData().getTimestamp().compareTo(i2.getMetaData().getTimestamp()));
        for (Image image : images) {
            if ((spaceUsage()) > purgeTarget) {
                LOG.info("Deleting: " + image.getMetaData().getName());
                image.delete();
            } else {
                break;
            }
        }
        double fracAfter =  spaceUsage();
	double fracDiff = fracBefore - fracAfter;
        if (!Double.isFinite(fracDiff)) {fracDiff = Double.NaN;}
	KeyValueData purged = new KeyValueData("store/purgedSpaceFraction",
                                               fracDiff);
        agent.publishSubsystemDataOnStatusBus(purged);
        LOG.info(name + " after purge, utilized space fraction = " + 
                 Double.toString(fracAfter));
    }

   /**
    *  If fraction of oocupied DAQ store space is above purgeThreshold,
    *  carry out a file purge.
    *
     *  @throws DAQException
    */

    void autoPurge() throws DAQException {
        if (!enableAutoPurge) {return;}
        checkStore();
        double usage = spaceUsage();
        if (usage > purgeThreshold) {
            LOG.info(name + " autoPurge initiated");
            purge();
        } else {
	    LOG.info(name + "autoPurge requested,a but space usage " +
                     Double.toString(usage) + " is below purgeThreshold "
                     + purgeThreshold.toString());
        }
    }

}
