package org.lsst.ccs.subsystem.daq;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.DAQDriverStats;
import org.lsst.ccs.daq.ims.DAQFirmwareStats;
import org.lsst.ccs.daq.ims.DAQRdsStats;
import org.lsst.ccs.daq.ims.DAQRmsStats;
import org.lsst.ccs.daq.ims.Stats;
import org.lsst.ccs.daq.ims.Store;
import org.lsst.ccs.framework.AgentPeriodicTask;;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.daq.alerts.DaqSubsystemAlerts;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;

public class DaqStatsMonitor implements HasLifecycle, ConfigurationBulkChangeHandler {

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    DataProviderDictionaryService dataProviderDictionaryService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alert;

    private String partition;
    private String fullName;
    private boolean online;
    private boolean inited = false;
    private boolean firstRead = true;
    private boolean alertWarning = false;
    private Stats stats;
    private Store store;
    private static final Logger LOG = Logger.getLogger(DaqStatsMonitor.class.getName());
    private final LocationSet locSetAll = LocationSet.all();
    private LocationSet locSet = locSetAll;
    private final Duration publishInterval = Duration.ofSeconds(15); //initial
    private Map<String,Long> currentSums = new HashMap<>(40);
    private Map<String,Long> previousSums = new HashMap<>(40);

   /*  Configuration Parameters */

    @ConfigurationParameter(isBuild = true, 
                            description = "enable periodic publication")
    volatile boolean enablePeriodicPublication;

    @ConfigurationParameter(isFinal = true, category = "Statistics",
                            maxLength = 15,
                            description = "Driver Stats for sums over location")
    protected volatile List<String> sumDriverStats = new ArrayList<>();

    @ConfigurationParameter(isFinal = true, category = "Statistics",
                            maxLength = 30,
                            description = "Firmware Stats for sums over location")
    protected volatile List<String> sumFirmwareStats = new ArrayList<>();

    @ConfigurationParameter(isFinal = true, category = "Statistics",
                            maxLength = 15,
                            description = "Rds Stats for sums over location")
    protected volatile List<String> sumRdsStats = new ArrayList<>();

    @ConfigurationParameter(isFinal = true, category = "Statistics",
                            maxLength = 15,
                            description = "Rms Stats for sums over location")
    protected volatile List<String> sumRmsStats = new ArrayList<>();

    @ConfigurationParameter(isFinal = true, category = "Statistics",
                            maxLength = 15,
                            description = "Driver Stats for warnings on sums")
    protected volatile List<String> sumDriverChecks = new ArrayList<>();

    @ConfigurationParameter(isFinal = true, category = "Statistics",
                            maxLength = 30,
                            description = "Firmware Stats for warnings on sums")
    protected volatile List<String> sumFirmwareChecks = new ArrayList<>();

    @ConfigurationParameter(isFinal = true, category = "Statistics",
                            maxLength = 15,
                            description = "Rds Stats for warnings on sums")
    protected volatile List<String> sumRdsChecks = new ArrayList<>();

    @ConfigurationParameter(isFinal = true, category = "Statistics",
                            maxLength = 15,
                            description = "Rms Stats for warnings on sums")
    protected volatile List<String> sumRmsChecks = new ArrayList<>();

    @ConfigurationParameter(category = "Statistics",
                            description = "minimum difference between successive reads to put warning in log")
    protected volatile Integer minDiffLogWarning;

    @ConfigurationParameter(category = "Statistics",
                            description = "minimum difference between successive reads to raise Alert at warning level")
    protected volatile Integer minDiffAlertWarning;

   /**
    *  Constructor
    */
    public DaqStatsMonitor() {}

    /**
     *  Gets the online state.
     *
     *  @return  Whether or not the device is online
     */
    public boolean isOnline()
    {
        return online;
    }

   /**
    * build phase:  set up periodic task for publication
    */
    @Override
    public void build() {
        Runnable publishStats = new Runnable() {
            @Override
	    public void run() {
                try {
                    for (Location loc : locSet) {
                        agent.publishSubsystemDataOnStatusBus(getDataForLocation(loc));
                    }
                    agent.publishSubsystemDataOnStatusBus(getSummedData());
                    if (!firstRead) {
                        checkDifferences(currentSums, previousSums);
                    } else {
                        firstRead = false;
                    }
                    previousSums.putAll(currentSums);
                }
                catch (DAQException ex) {
                    LOG.warning("DAQ Exception while acquiring data for "
				+ "publication: " + ex);
                }
            }
	};

	if (enablePeriodicPublication) {
            AgentPeriodicTask periodicTask = new AgentPeriodicTask("publishStats", 
            publishStats).withPeriod(publishInterval);
            periodicTaskService.scheduleAgentPeriodicTask(periodicTask);
            LOG.info(name + "  publication of trending data is enabled");
        } else {
            LOG.info(name + "  publication of trending data is disabled");
	}
    }

   /**
    *  Set partition name for DAQ connection (before init() method)
    *
    *  @param pname  String partition name
    */
    void setPartition(String pname) {
        partition = pname;
    }

   /**
    *  init phase:  open connection to DAQ partition
    *
    *  Also, use first Daq location to verify ConfigurationParameter Lists
    * 
    *  @throws DAQException
    */
    @Override
    public void init() {

        fullName = name + ": (partition" + partition + ")";
        try {
            if (stats != null) {
                close();
            }
            stats = new Stats(partition);
            online = true;
            LOG.info("\n " + name + " Connected to DAQ partition " + 
                     partition);
            store = new Store(partition);
            locSet = store.getConfiguredSources();
            store.close();           // a precaution
            LOG.info(name + String.format(":  using %d DAQ configured locations", locSet.size()));

            // Get a location to use for validation chevcks
            if (!locSet.iterator().hasNext()) {
                throw new RuntimeException(name + ":  there are no configured locations");
            }
            Location loc = locSet.iterator().next();
            // Stats on sum Lists must be valid 
            Map<String,Long> map;
            map = stats.getDAQDriverStats(loc, Stats.Clear.NO).getMap();
            for (String par:sumDriverStats) {
                if (!map.containsKey(par)) {
                    ErrorUtils.reportConfigError(LOG,name,"sumDriverStats",
                                                 par + " is invalid entry");
                }
            }
            map = stats.getDAQFirmwareStats(loc, Stats.Clear.NO).getMap();
            for (String par:sumFirmwareStats) {
                if (!map.containsKey(par)) {
                    ErrorUtils.reportConfigError(LOG,name,"sumFirmwareStats",
                                                 par + " is invalid entry");
                }
            }
            map = stats.getDAQRdsStats(loc, Stats.Clear.NO).getMap();
            for (String par:sumRdsStats) {
                if (!map.containsKey(par)) {
                    ErrorUtils.reportConfigError(LOG,name,"sumRdsStats",
                                                 par + " is invalid entry");
                }
            }
            map = stats.getDAQRmsStats(loc, Stats.Clear.NO).getMap();
            for (String par:sumRmsStats) {
                if (!map.containsKey(par)) {
                    ErrorUtils.reportConfigError(LOG,name,"sumRmsStats",
                                                 par + " is invalid entry");
                }
            }
            // Stats on warning Lists must be on sum Lists
           for (String par:sumDriverChecks) {
                if (!sumDriverStats.contains(par)) {
                    ErrorUtils.reportConfigError(LOG,name,"sumDriverChecks",
                                                 par + "is invalid entry");
                }
            }
           for (String par:sumFirmwareChecks) {
                if (!sumFirmwareStats.contains(par)) {
                    ErrorUtils.reportConfigError(LOG,name,"sumFirmwareChecks",
                                                 par + "is invalid entry");
                }
            }
            for (String par:sumRdsChecks) {
                if (!sumRdsStats.contains(par)) {
                    ErrorUtils.reportConfigError(LOG,name,"sumRdsChecks",
                                                 par + "is invalid entry");
                }
            }
            for (String par:sumRmsChecks) {
                if (!sumRmsStats.contains(par)) {
                    ErrorUtils.reportConfigError(LOG,name,"sumRmsChecks",
                                                 par + "is invalid entry");
                }
            }
            // Check for validity of thresholds is now in validateBulkChange
        }
        catch (DAQException ex) {
            if (!inited) {
                throw new RuntimeException("DAQException while connecting to "
                 + fullName + ":  " + ex.getMessage());
            }
            try {
                close();
	    }
            catch (DAQException exx) {}
        }
        inited = true;

        /* Register keys for publication */

        if (enablePeriodicPublication) {
            try {
                for (Location loc : locSet)  {
                    registerDataForLocation(loc);
                }
                dataProviderDictionaryService.registerData(getSummedData());
                LOG.info(name + " Registration of Stats data complete");
	    }
            catch (DAQException ex) {
                throw new RuntimeException(name + " data registration failed: " + ex);
	    }
        }
    }

   /**
    *  Closes connection
    * 
    *  @throws  DAQException
    */
    void close() throws DAQException {
        if (stats != null) {
            stats.close();
            stats = null;
            online = false;
        }
    }

   /**
    *  Checks validity of a proposed set of ConfigurationParameters
    *
    *  @param params Map of ConfigurationParameters from name to value
    */
    @Override
    public void validateBulkChange(Map<String,Object> params) {
        Integer logWarningParam = (Integer) params.get("minDiffLogWarning");
        Integer alertWarningParam = (Integer) params.get("minDiffAlertWarning");
        if (logWarningParam == null || logWarningParam <= 0) {
            String msg = " is invalid";
             if (inited) {
                throw new RuntimeException(name + ": minDiffLogWarning" + msg);
            } else {
                ErrorUtils.reportConfigError(LOG, name, "minDiffLogWarning", msg);
            }
	}
        if (alertWarningParam == null || alertWarningParam <= 0) {
            String msg = " is invalid";
             if (inited) {
                throw new RuntimeException(name + ": minDiffAlertWarning" + msg);
            } else {
                ErrorUtils.reportConfigError(LOG, name, "minDiffAlertWarning", msg);
            }
	}
        if (alertWarningParam < logWarningParam) {
            String msg = " shou;d not be less than mindiffLogwarning";
             if (inited) {
                throw new RuntimeException(name + ": minDiffAlertWarning" + msg);
            } else {
                ErrorUtils.reportConfigError(LOG, name, "minDiffAlertWarning", msg);
            }
	}
    }

   /**
    * Method to prepare all data from a given location for publication
    * Data are in the form of a KeyValueList, also for registration.
    * 
    * @param  location  Readout Location for data
    * @return KeyValueDataList of the four blocks of data
    * @thros  DAQException
    */

    private KeyValueData getDataForLocation(Location location) throws DAQException  {
        KeyValueDataList list = new KeyValueDataList();
        String lname = location.toString();
	list.addData(lname+"/Rms", stats.getDAQRmsStats(location, Stats.Clear.NO));
	list.addData(lname+"/Rds", stats.getDAQRdsStats(location, Stats.Clear.NO));
	list.addData(lname+"/Driver", stats.getDAQDriverStats(location, Stats.Clear.NO));
	list.addData(lname+"/Firmware", stats.getDAQFirmwareStats(location, Stats.Clear.NO));
        return list;
    }

   /**
    * Method to register all data classes from a given location
    * 
    * @param  location  Readout Location for data
    */

    private void registerDataForLocation(Location location) {
        String lname = location.toString();
        dataProviderDictionaryService.registerClass(DAQRmsStats.class,
                                                    lname + "/Rms");
        dataProviderDictionaryService.registerClass(DAQRdsStats.class,
                                                    lname + "/Rds");
        dataProviderDictionaryService.registerClass(DAQDriverStats.class,
                                                    lname + "/Driver");
        dataProviderDictionaryService.registerClass(DAQFirmwareStats.class,
                                                    lname + "/Firmware");
    }

   /**
    * Method to prepare data from summed over locations for publication.
    * This is intended to be used for statistics whoch are error counts.
    * Data are in the form of a KeyValueList, also for registration.
    *
    * This method also prepares the Map used for comparing consecutive values
    * 
    * @return KeyValueDataList of the data summed over locations
    * @thros  DAQException
    */

    private KeyValueData getSummedData() throws DAQException  {
        KeyValueDataList list = new KeyValueDataList();
        long sum;
        for (String key : sumDriverStats) {
            sum = 0;
            for (Location loc : locSet) {
            	sum += stats.getDAQDriverStats(loc, Stats.Clear.NO).getStatistic(key);
	    }
            list.addData("Sum/Driver/" + key, sum);
	    if (sumDriverChecks.contains(key)) {
                currentSums.put("Sum/Driver/" + key, sum);
            }
        }
        for (String key : sumFirmwareStats) {
            sum = 0;
            for (Location loc : locSet) {
            	sum += stats.getDAQFirmwareStats(loc, Stats.Clear.NO).getStatistic(key);
	    }
            list.addData("Sum/Firmware/" + key, sum);
	    if (sumFirmwareChecks.contains(key)) {
                currentSums.put("Sum/Firmware/" + key, sum);
            }
        }
        for (String key : sumRdsStats) {
            sum = 0;
            for (Location loc : locSet) {
            	sum += stats.getDAQRdsStats(loc, Stats.Clear.NO).getStatistic(key);
	    }
            list.addData("Sum/Rds/" + key, sum);
	    if (sumRdsChecks.contains(key)) {
                currentSums.put("Sum/Rds/" + key, sum);
            }
        }
        for (String key : sumRmsStats) {
            sum = 0;
            for (Location loc : locSet) {
            	sum += stats.getDAQRmsStats(loc, Stats.Clear.NO).getStatistic(key);
	    }
            list.addData("Sum/Rms/" + key, sum);
	    if (sumRmsChecks.contains(key)) {
                currentSums.put("Sum/Rms/" + key, sum);
            }
        }
        return list;
    }

   /**
    * Compare selected summed data between two consecutive reads of DAQ 
    * statistic and raise warnings in log and/or Alert system if difference
    * for each statistic is above a threshold.
    *
    * @param currentSums  Map<String,Long> of most-recently-read data
    * @param previousSums Map<String,Long> previously-read data
    */

    private void checkDifferences(Map<String,Long> currentSums,
                                  Map<String,Long> previousSums) {
        if (!currentSums.equals(previousSums)) {
            Long diff;
            for (String key : currentSums.keySet()) {
                diff = currentSums.get(key) - previousSums.get(key);
                if (diff >= minDiffAlertWarning) {
                    alert.raiseAlert(DaqSubsystemAlerts.STATS_DIFF.newAlert(),
                                     AlertState.WARNING, key + 
                                     " increment = " +  Long.toString(diff));
		    alertWarning = true;
                } else if (diff >= minDiffLogWarning) {
                    LOG.warning(name + ": " + key + " increment = " + 
                                Long.toString(diff));
                }
            }
        }  else if (alertWarning) {
            // If warning on previous reading, restore to nominal
            alert.raiseAlert(DaqSubsystemAlerts.STATS_DIFF.newAlert(),
                             AlertState.NOMINAL, "");
	    alertWarning = false;
	}
    }

   /**
    * Method to make printable table for one set of statistics
    *
    * @param  sname     Name of statistics block
    * @param  statMap   Map of statistics data
    * @param  location  requested Location (raft/reb)
    * @return Table
    */

    private String makeStatsTable(String sname, Map<String,Long> statMap,
                                  String location) {
        Set<String> keys = statMap.keySet();
        String table = "DAQ " + sname + " Stats for " + location +"\n" + timestamp()+"\n";
        for (String key : keys) {
            table += String.format("\n  %-22s %d", key, statMap.get(key));
        }
        table += "\n";
        return table;
    }

   /**
    * Print all four sets of statisics for specified location
    *
    * @param   location  Location in raft/board format, "R<nn>/Reb<m>"
    * @return  String reporting all data read 
    * @throws  DAQException
    */
    @Command(type=Command.CommandType.QUERY, name="readAllStats", 
             description="Read all sets of DAQ stats")
    public String readAllStats(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location) throws DAQException {
        return ("\n" + readAllRms(location) +
                "\n" + readAllRds(location) + 
                "\n" + readAllDriver(location) + 
	        "\n" + readAllFirmware(location) + "\n");
    }

   /**
    * Print then clear all four sets of statisics for specified location
    *
    * @param   location  Location in raft/board format, "R<nn>/Reb<m>"
    * @return  String reporting all data read 
    * @throws  DAQException
    */
    @Command(type=Command.CommandType.ACTION, name="clearAllStats", 
             description="Read and clear all sets of DAQ stats")
    public String clearAllStats(@Argument(name="location", description = "clearout board in format R<nn>/Reb<m>") String location) throws DAQException {
        return ("\n" + clearAllRms(location) + 
                "\n" + clearAllRds(location) + 
                "\n" + clearAllDriver(location) + 
	        "\n" + clearAllFirmware(location) + "\n");
    }

   /**
    * Print all DAQ Rms statistics data for specified Location
    * Reads all DAQRmsStatistics values and return them in table format.
    * 
    * @param   location  Location in raft/board format, "R<nn>/Reb<m>"
    * @return  String reporting all data read 
    * @throws  DAQException
    */
    @Command(type=Command.CommandType.QUERY, name="readAllRms", 
             description="Read all DAQ Rms stats")
    public String readAllRms(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location) throws DAQException {
	DAQRmsStats rmsStats = stats.getDAQRmsStats(Location.of(location),
                                                    Stats.Clear.NO);
        Map<String,Long> statMap = rmsStats.getMap();
        return makeStatsTable("Rms",statMap,location);
    }

   /**
    * Print all DAQ Rds statistics data for specified Location
    * Reads all DAQRdsStatistics values and return them in table format.
    * 
    * @param   location  Location in raft/board format, "R<nn>/Reb<m>"
    * @return  String reporting all data read 
    * @throws  DAQException
    */
    @Command(type=Command.CommandType.QUERY, name="readAllRds", 
             description="Read all DAQ Rds stats")
    public String readAllRds(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location) throws DAQException {
	DAQRdsStats rdsStats = stats.getDAQRdsStats(Location.of(location),
                                                    Stats.Clear.NO);
        Map<String,Long> statMap = rdsStats.getMap();
        return makeStatsTable("Rds",statMap,location);
    }

   /**
    * Print all DAQ Driver statistics data for specified Location
    * Reads all DAQDriverStatistics values and return them in table format.
    * 
    * @param   location  Location in raft/board format, "R<nn>/Reb<m>"
    * @return  String reporting all data read 
    * @throws  DAQException
    */
    @Command(type=Command.CommandType.QUERY, name="readAllDriver", 
             description="Read all DAQ Driver stats")
    public String readAllDriver(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location) throws DAQException {
        DAQDriverStats driverStats = stats.getDAQDriverStats(Location.of(location), Stats.Clear.NO);
        Map<String,Long> statMap = driverStats.getMap();
        return makeStatsTable("Driver",statMap,location);
    }

   /**
    * Print all DAQ Firmware statistics data for specified Location
    * Reads all DAQFirmwareStatistics values and return them in table format.
    * 
    * @param   location  Location in raft/board format, "R<nn>/Reb<m>"
    * @return  String reporting all data read 
    * @throws  DAQException
    */
    @Command(type=Command.CommandType.QUERY, name="readAllFirmware", 
             description="Read all DAQ Firmware stats")
    public String readAllFirmware(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location) throws DAQException {
	DAQFirmwareStats firmwareStats = stats.getDAQFirmwareStats(Location.of(location), Stats.Clear.NO);
        Map<String,Long> statMap = firmwareStats.getMap();
        return makeStatsTable("Firmware",statMap,location);
    }

   /**
    * Print and Clear all DAQ Rms statistics data for specified Location
    * Like readAllRms, but clears data after reading
    * 
    * @param   location  Location in raft/board format, "R<nn>/Reb<m>"
    * @return  String reporting all data read 
    * @throws  DAQException
    */
    @Command(type=Command.CommandType.ACTION, name="clearAllRms", 
             description="Read and clear DAQ Rms stats")
    public String clearAllRms(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location) throws DAQException {
	DAQRmsStats rmsStats = stats.getDAQRmsStats(Location.of(location),
                                                    Stats.Clear.YES);
        Map<String,Long> statMap = rmsStats.getMap();
        return makeStatsTable("Rms",statMap,location)+"Cleared";
    }

   /**
    * Print and Clear all DAQ Rds statistics data for specified Location
    * Like readAllRds, but clears data after reading
    * 
    * @param   location  Location in raft/board format, "R<nn>/Reb<m>"
    * @return  String reporting all data read 
    * @throws  DAQException
    */
    @Command(type=Command.CommandType.ACTION, name="clearAllRds", 
             description="Read and clear DAQ Rds stats")
    public String clearAllRds(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location) throws DAQException {
	DAQRdsStats rdsStats = stats.getDAQRdsStats(Location.of(location),
                                                    Stats.Clear.YES);
        Map<String,Long> statMap = rdsStats.getMap();
        return makeStatsTable("Rds",statMap,location)+"Cleared";
    }

   /**
    * Print and Clear all DAQ Driver statistics data for specified Location
    * Like readAllDriver, but clears data after reading
    * 
    * @param   location  Location in raft/board format, "R<nn>/Reb<m>"
    * @return  String reporting all data read 
    * @throws  DAQException
    */
    @Command(type=Command.CommandType.ACTION, name="clearAllDriver", 
             description="Read and clear DAQ Driver stats")
    public String clearAllDriver(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location) throws DAQException {
        DAQDriverStats driverStats = stats.getDAQDriverStats(Location.of(location), Stats.Clear.YES);
        Map<String,Long> statMap = driverStats.getMap();
        return makeStatsTable("Driver",statMap,location)+"Cleared";
    }

   /**
    * Print and Clear all DAQ Firmware statistics data for specified Location
    * Like readAllFirmware, but clears data after reading
    * 
    * @param   location  Location in raft/board format, "R<nn>/Reb<m>"
    * @return  String reporting all data read 
    * @throws  DAQException
    */
    @Command(type=Command.CommandType.ACTION, name="clearAllFirmware", 
             description="Read and clear DAQ Firmware stats")
    public String clearAllFirmware(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location) throws DAQException {
	DAQFirmwareStats firmwareStats = stats.getDAQFirmwareStats(Location.of(location), Stats.Clear.YES);
        Map<String,Long> statMap = firmwareStats.getMap();
        return makeStatsTable("Firmware",statMap,location)+"Cleared";
    }

   /**
    * Read specified entry in DAQ Rms Stats for specified location
    *
    * @param  location  Location in raft/board format, "R<nn>/Reb<m>"
    * @param  name      Name of requested quantity
    * @return Value of requested quantity
    * @throws DAQException
    */
    @Command(type=Command.CommandType.QUERY, name="readRmsStat", 
             description="read specified DAQ Rms statistic for specified locaation")
	public String readRmsStat(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location, @Argument(name="statistic name") String quantity) throws DAQException {
	DAQRmsStats rmsStats = stats.getDAQRmsStats(Location.of(location), Stats.Clear.NO);
        return rmsStats.getStatistic(quantity).toString();
    }

   /**
    * Read specified entry in DAQ Rds Stats for specified location
    *
    * @param  location  Location in raft/board format, "R<nn>/Reb<m>"
    * @param  name      Name of requested quantity
    * @return Value of requested quantity
    * @throws DAQException
    */
    @Command(type=Command.CommandType.QUERY, name="readRdsStat", 
             description="read specified DAQ Rds statistic for specified locaation")
	public String readRdsStat(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location, @Argument(name="statistic name") String quantity) throws DAQException {
	DAQRdsStats rdsStats = stats.getDAQRdsStats(Location.of(location), Stats.Clear.NO);
        return rdsStats.getStatistic(quantity).toString();
    }

   /**
    * Read specified entry in DAQ Driver Stats for specified location
    *
    * @param  location  Location in raft/board format, "R<nn>/Reb<m>"
    * @param  name      Name of requested quantity
    * @return Value of requested quantity
    * @throws DAQException
    */
    @Command(type=Command.CommandType.QUERY, name="readDriverStat", 
             description="read specified DAQ Driver statistic for specified locaation")
	public String readDriverStat(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location, @Argument(name="statistic name") String quantity) throws DAQException {
	DAQDriverStats driverStats = stats.getDAQDriverStats(Location.of(location), Stats.Clear.NO);
        return driverStats.getStatistic(quantity).toString();
    }

   /**
    * Read specified entry in DAQ Firmware Stats for specified location
    *
    * @param  location  Location in raft/board format, "R<nn>/Reb<m>"
    * @param  name      Name of requested quantity
    * @return Value of requested quantity
    * @throws DAQException
    */
    @Command(type=Command.CommandType.QUERY, name="readFirmwareStat", 
             description="read specified DAQ Firmware statistic for specified locaation")
	public String readFirmwareStat(@Argument(name="location", description = "readout board in format R<nn>/Reb<m>") String location, @Argument(name="statistic name") String quantity) throws DAQException {
	DAQFirmwareStats firmwareStats = stats.getDAQFirmwareStats(Location.of(location), Stats.Clear.NO);
        return firmwareStats.getStatistic(quantity).toString();
    }

    /**
     * Determines current time
     *
     * @return current time as a String
     */
    @Command(name="timestamp", description="Prints current time")
    public String timestamp() {
	Date now = new Date();
        return now.toString();
    }

}