package org.lsst.ccs.subsystem.daq;

import java.time.Duration;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.ClearAlertHandler.ClearAlertCode;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.daq.alerts.DaqSubsystemAlerts;

/* Pre-defined Command levels */
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING2;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING3;

/**
 ******************************************************************************
 *
 *  Daq subsystem
 *
 *  @author  Al Eisner
 *
 * *****************************************************************************
 */
public class DaqSubsystem extends Subsystem implements HasLifecycle, ClearAlertHandler {

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private DaqStoreManageDevice store;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private DaqStatsMonitor stats;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alert;

    @LookupName
    private String name;

    /**
     *  Private fields
     */

    private static final Logger LOG = Logger.getLogger(DaqSubsystem.class.getName());
    private final Duration autoPurgeInterval = Duration.ofMinutes(15); 
    private int nPurgeFail = 0;

    /*  Configuration Parameters */

    @ConfigurationParameter(isFinal = true, category = "Store",
                            description = "partition containing DAQ store")
    private volatile String daqPartition;

    @ConfigurationParameter(category="Store",
      description = "number of successive autoPurge failures for alarm")
    private volatile Integer alarmPurgeFailures;

    /**
     *  Constructor
     */
    public DaqSubsystem() {
        super("daqsub", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }

   /**
    * build phase:  set up periodic task for automatic purging of DAQ store
    */
    @Override
    public void build() {
        Runnable purgeDAQ = new Runnable() {
            @Override
	    public void run() {
                try {
                    store.autoPurge();
                    nPurgeFail = 0;
                }
                catch (DAQException ex) {
                    LOG.severe("DAQExceptiom during attempted sutoPurge: " + ex);
                    nPurgeFail++;
                    if (nPurgeFail >= alarmPurgeFailures) {
			String msg = String.format("%d successive failures of AutoPurge", nPurgeFail);
                        alert.raiseAlert(DaqSubsystemAlerts.PURGE_FAIL.newAlert(),
                                         AlertState.ALARM, msg);
			nPurgeFail = 0;
                    }
                }
            }
	};
        LOG.info(name + " in build() phase");
        AgentPeriodicTask purgeTask = new AgentPeriodicTask("purgeDAQ", 
            purgeDAQ).withPeriod(autoPurgeInterval);
        periodicTaskService.scheduleAgentPeriodicTask(purgeTask);
    }

   /** 
    * init phase
    */
    @Override
    public void init() {

        LOG.info(name + " starting init() phase");

        /* Verify that ConfigurationParameters are valid  */

        if (daqPartition == null) {
            ErrorUtils.reportConfigError(LOG,name,"daqPartition",
                                         "value is missing");
        }
        store.setPartition(daqPartition);
        stats.setPartition(daqPartition);

        if (alarmPurgeFailures == null) {
            ErrorUtils.reportConfigError(LOG,name,"alarmPurgeFailures",
                                         "value is missing");
        }

    }

    /**
     * Post-initialization phase
     */
    @Override
    public void postInit() {

        /** 
         *  By setting ???_TYPE_AGENT_PROPERTY we signal to consoles 
         *  that this subsystem is compatible with the ??? subsystm GUI
         *
         *  Not initially implemented.
         */
    }

    /**
     * Start
     */
    @Override
    public void start() {

        if (!RunMode.isSimulation()) {
            /* If DAQ store or stats is not online, throw an Exception */
	    
            if (!store.isOnline()) {
	        throw new RuntimeException("DAQ store not connected");
            }
            if (!stats.isOnline()) {
	        throw new RuntimeException("DAQ stats not connected");
	    }
        }
    }

   /**
    * Tests whether alert can be cleared.
    *
    * @param Alert - the Alert being testsd
    * @param AlertState - state of Alert being tested
    *
    * @return ClearAlertCode
    */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
        String alertId = alert.getAlertId();
        if (alertId.equals(DaqSubsystemAlerts.PURGE_FAIL.getId()) ||
            alertId.equals(DaqSubsystemAlerts.STATS_DIFF.getId()) ) {
            return ClearAlertCode.CLEAR_ALERT;
        }
        return ClearAlertCode.UNKNOWN_ALERT;
    }

   /**
    *  Purge DAQ file storage (two-day store), removing enough older 
    *  to fall below specified fractional usage (ConfigurationParameter
    *  of DaqStoreManageDevice).
    *
    *  @throws DAQException
    */
    @Command(type=Command.CommandType.ACTION, level=ENGINEERING1,
             name="purge", description="Purge older files in DAQ two-day store")
    public void purge() throws DAQException {
        store.purge();
    }

}
