package org.lsst.ccs.subsystem.daq;

import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/* Pre-defined Command levels */
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING2;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING3;

/**
 ******************************************************************************
 *
 *  Daq subsystem
 *
 *  @author  Al Eisner
 *
 * *****************************************************************************
 */
public class DaqSubsystem extends Subsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private DaqStoreManageDevice store;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private DaqStatsMonitor stats;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alert;
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @LookupName
    private String name;

    /**
     *  Private fields
     */

    private static final Logger LOG = Logger.getLogger(DaqSubsystem.class.getName());

    /*  Configuration Parameters */

    @ConfigurationParameter(isFinal = true, category = "DAQ",
                            description = "partition containing DAQ store")
    private String daqPartition;

    /**
     *  Constructor
     */
    public DaqSubsystem() {
        super("daqsub", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }

   /** 
    * init phase
    */
    @Override
    public void init() {

        /* Verify that ConfigurationParameters are valid  */

        if (daqPartition == null) {
            ErrorUtils.reportConfigError(LOG,name,"daqPartition",
                                         "value is missing");
        }
        store.setPartition(daqPartition);
        stats.setPartition(daqPartition);
    }

    /**
     * Post-initialization phase
     */
    @Override
    public void postInit() {

        /** 
         *  By setting ???_TYPE_AGENT_PROPERTY we signal to consoles 
         *  that this subsystem is compatible with the ??? subsystm GUI
         *
         *  Not initially implemented.
         */
    }

    /**
     * Start
     */
    @Override
    public void start() {

        if (!RunMode.isSimulation()) {
            /* If DAQ store or stats is not online, throw an Exception */
	    
            if (!store.isOnline()) {
	        throw new RuntimeException("DAQ store not connected");
            }
            if (!stats.isOnline()) {
	        throw new RuntimeException("DAQ stats not connected");
	    }
        }
    }

    /*  Informational Commands */

   /**
    *  Purge DAQ file storage (two-day store), removing enough older 
    *  to fall below specified fractional usage (ConfigurationParameter
    *  of DaqStoreManageDevice).
    *
    *  @throws DAQException
    */
    @Command(type=Command.CommandType.ACTION, level=ENGINEERING1,
             name="purge", description="Purge older files in DAQ two-day store")
    public void purge() throws DAQException {
        store.purge();
    }

}
