package org.lsst.ccs.subsystem.shell;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.regex.Pattern;

import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.AgentCommandDictionaryService.AgentCommandDictionaryEvent;
import org.lsst.ccs.services.AgentCommandDictionaryService.AgentCommandDictionaryListener;

import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.command.annotations.Command.CommandCategory;
import org.lsst.ccs.command.annotations.Command.CommandType;

/**
 * Utility class to dump command dictionaries for the agents on the buses.
 * 
 */
public class DumpCommands extends Agent {


// -- Construction and initialization : ----------------------------------------
    DumpCommands() {
        super("dump-commands", AgentType.CONSOLE);
    }

    private String dumpDictionaryCommands() {
        AgentCommandDictionaryService dictService = getAgentService(AgentCommandDictionaryService.class);
        DictionaryListener listener = new DictionaryListener();
        dictService.addAgentCommandDictionaryListener(listener);
        Map<AgentInfo, Map<String, Dictionary>> dictionaries = listener.getDictionaries();

        StringBuilder sb = new StringBuilder();

        Map<AgentInfo, TreeMap<Integer, Set<CommandWrapper>>> subsystemSpecificActionCommands = new LinkedHashMap<>();
        Map<AgentInfo, TreeMap<Integer, Set<CommandWrapper>>> subsystemSpecificQueryCommands = new LinkedHashMap<>();
        TreeMap<Integer, Set<CommandWrapper>> commonActionCommands = new TreeMap();
        TreeMap<Integer, Set<CommandWrapper>> commonQueryCommands = new TreeMap();

        TreeMap<String, AgentInfo> sortedAgents = new TreeMap();
        for (AgentInfo ai : getMessagingAccess().getAgentPresenceManager().listConnectedAgents()) {
            sortedAgents.put(ai.getName(), ai);
        }

        Set processedDescriptions = new TreeSet();
        for (AgentInfo ai : sortedAgents.values()) {

            String agentDescription = ai.getAgentProperty("agentDescription");
            switch (agentDescription) {
                case "LocalDatabase", "ClusterMonitor", "Mmm", "Jas3Console", "LockManager" -> {
                    continue;
                }
            }

            if (processedDescriptions.add(agentDescription)) {

                Map<String, Dictionary> dicts = dictionaries.get(ai);
                if (dicts == null) {
                    System.out.println("No dictionary for " + ai.getName());
                    continue;
                }

                TreeMap<Integer, Set<CommandWrapper>> commandsSortedByLevel;

                for (Map.Entry<String, Dictionary> e : dicts.entrySet()) {
                    String agentName = ai.getName();
                    String target = e.getKey().replace(agentName + "/", "").replace(agentName, "");
                    for (DictionaryCommand cmd : e.getValue()) {

                        CommandCategory cat = cmd.getCategory();
                        CommandType type = cmd.getType();

                        //https://jira.slac.stanford.edu/browse/LSSTCCS-2957
                        if (cmd.getCommandName().equals("getRaisedAlertSummary")
                                || cmd.getCommandName().equals("printRegisteredAlerts")
                                || cmd.getCommandName().equals("printAllLatestData")) {
                            cat = CommandCategory.CORE;
                        }

                        if (cmd.getCommandName().equals("getChannelNames") && target.isEmpty()) {
                            cat = CommandCategory.CORE;
                        }

                        //https://jira.slac.stanford.edu/browse/LSSTCCSRAFTS-766
                        if (cmd.getCommandName().equals("disableBackBias")) {
                            cat = CommandCategory.USER;
                        }

                        CommandWrapper wrapper = new CommandWrapper(cmd);

                        if (cat == CommandCategory.USER) {
                            if (type == CommandType.QUERY) {
                                commandsSortedByLevel = subsystemSpecificQueryCommands.computeIfAbsent(ai, (a) -> new TreeMap<>());
                            } else {
                                commandsSortedByLevel = subsystemSpecificActionCommands.computeIfAbsent(ai, (a) -> new TreeMap<>());
                            }
                        } else {
                            if (type == CommandType.QUERY) {
                                commandsSortedByLevel = commonQueryCommands;
                            } else {
                                commandsSortedByLevel = commonActionCommands;
                            }
                        }

                        int level = cmd.getLevel();

                        Set<CommandWrapper> wrapperSet = commandsSortedByLevel.computeIfAbsent(level, (l) -> new TreeSet<>());
                        wrapperSet.add(wrapper);
                        for (CommandWrapper wrp : wrapperSet) {
                            if (wrp.equals(wrapper)) {
                                wrp.addTarget(target);
                                break;
                            }
                        }
                    }
                }
            }

        }

        sb.append("Subsysem specific ACTION commands").append("\n\n");
        sb.append("|Subsystem|Level|Command|Description|Targets|\n");
        for (Entry<AgentInfo, TreeMap<Integer, Set<CommandWrapper>>> e : subsystemSpecificActionCommands.entrySet()) {
            AgentInfo ai = e.getKey();
            sb.append("|").append(ai.getAgentProperty("agentDescription")).append("| | | | |\n");
            for (Entry<Integer, Set<CommandWrapper>> e1 : e.getValue().entrySet()) {
                Integer level = e1.getKey();
                sb.append("| |").append(level).append("| | | |\n");
                for (CommandWrapper w : e1.getValue()) {
                    sb.append("| | |").append(w.commandName).append("|").append(w.getCommandDescription()).append("|").append(w.getTargets()).append("|\n");
                }

            }
        }

        sb.append("\n\nCommon ACTION commands").append("\n\n");
        sb.append("| |Level|Command|Description|Targets|\n");
        for (Entry<Integer, Set<CommandWrapper>> e1 : commonActionCommands.entrySet()) {
            Integer level = e1.getKey();
            sb.append("| |").append(level).append("| | | |\n");
            for (CommandWrapper w : e1.getValue()) {
                sb.append("| | |").append(w.commandName).append("|").append(w.getCommandDescription()).append("|").append(w.getTargets()).append("|\n");
            }
        }

        sb.append("\n\nSubsysem specific QUERY commands").append("\n\n");
        sb.append("|Subsystem|Level|Command|Description|Targets|\n");
        for (Entry<AgentInfo, TreeMap<Integer, Set<CommandWrapper>>> e : subsystemSpecificQueryCommands.entrySet()) {
            AgentInfo ai = e.getKey();
            sb.append("|").append(ai.getAgentProperty("agentDescription")).append("| | | | |\n");
            for (Entry<Integer, Set<CommandWrapper>> e1 : e.getValue().entrySet()) {
                Integer level = e1.getKey();
                sb.append("| |").append(level).append("| | | |\n");
                for (CommandWrapper w : e1.getValue()) {
                    sb.append("| | |").append(w.commandName).append("|").append(w.getCommandDescription()).append("|").append(w.getTargets()).append("|\n");
                }

            }
        }

        sb.append("\n\nCommon QUERY commands").append("\n\n");
        sb.append("| |Level|Command|Description|Targets|\n");
        for (Entry<Integer, Set<CommandWrapper>> e1 : commonQueryCommands.entrySet()) {
            Integer level = e1.getKey();
            sb.append("| |").append(level).append("| | | |\n");
            for (CommandWrapper w : e1.getValue()) {
                sb.append("| | |").append(w.commandName).append("|").append(w.getCommandDescription()).append("|").append(w.getTargets()).append("|\n");
            }
        }

        return sb.toString();
    }

    private final static Pattern RebPattern = Pattern.compile("R(\\d\\d)/Reb(.)");
    private final static Pattern SensorPattern = Pattern.compile("R(\\d\\d)/Reb(.)/S(.\\d)");
    private final static Pattern BiasPattern = Pattern.compile("R(\\d\\d)/Reb(.)/Bias(\\d)");
    private final static Pattern BiasOrDACPattern = Pattern.compile("R(\\d\\d)/Reb(.)/(Bias(\\d)|DAC)");
    private final static Pattern AspicPattern = Pattern.compile("R(\\d\\d)/Reb(.)/S(.\\d)/ASPIC(\\d)");
    private final static Pattern PowerSupplyPattern = Pattern.compile("RebPS/P(\\d\\d)");
    private final static Pattern CryoPattern = Pattern.compile("Cryo(\\d)");
    private final static Pattern CryoMaq20Pattern = Pattern.compile("Cryo(\\d)/Maq20");
    private final static Pattern ColdPattern = Pattern.compile("Cold(\\d)");

    private static Map<String, Pattern> specialPatterns = new HashMap();

    static {
        specialPatterns.put("R*/Reb*", RebPattern);
        specialPatterns.put("R*/Reb*/S*", SensorPattern);
        specialPatterns.put("R*/Reb*/Bias*", BiasPattern);
        specialPatterns.put("R*/Reb*/Bias* and R*/Reb*/DAC", BiasOrDACPattern);
        specialPatterns.put("R*/Reb*/S*/ASPIC*", AspicPattern);
        specialPatterns.put("RebPs/P*", PowerSupplyPattern);
        specialPatterns.put("Cryo*", CryoPattern);
        specialPatterns.put("Cryo*/Maq20", CryoMaq20Pattern);
        specialPatterns.put("Cold*", ColdPattern);
    }

    private static class CommandWrapper implements Comparable {

        public final String commandName;
        public final String commandDescription;
        public final int level;
        private final CommandCategory category;
        private final CommandType type;
        public final Set<String> targets = new TreeSet<>();

        public CommandWrapper(DictionaryCommand cmd) {
            this.commandName = cmd.getCommandName();
            this.commandDescription = cmd.getDescription().replaceAll("\n", " ").replaceAll("\t", "").replace("|", "/");
            this.category = cmd.getCategory();
            this.type = cmd.getType();
            this.level = cmd.getLevel();
        }

        @Override
        public boolean equals(Object obj) {
            CommandWrapper cw = (CommandWrapper) obj;
            return cw.commandName.equals(this.commandName) && cw.commandDescription.equals(this.commandDescription) && cw.level == this.level;
        }

        @Override
        public int hashCode() {
            return 123 + commandName.hashCode() + 12 * commandDescription.hashCode() + 123 * (level + 3);
        }

        public void addTarget(String target) {
            targets.add(target);
        }

        public String getCommandDescription() {
            return commandDescription.isEmpty() ? " " : commandDescription;
        }

        public String getTargets() {
            if (type == CommandType.CONFIGURATION && category != CommandCategory.USER) {
                if (targets.size() > 1) {
                    return "Any Component with Configuration Parameters";
                }
            }
            if (commandName.equals("getValue") && category == CommandCategory.CORE) {
                return "All Channels";
            }
            if (commandName.equals("printComponentConfigurationParameters")) {
                return "All Components with Configuration Parameters";
            }
            if (targets.size() > 1) {
                for (Entry<String, Pattern> e : specialPatterns.entrySet()) {
                    Pattern p = e.getValue();
                    boolean match = true;
                    for (String target : targets) {
                        if (!p.matcher(target).matches()) {
                            match = false;
                            break;
                        }
                    }
                    if (match) {
                        return e.getKey();
                    }
                }
            }

            String ret = targets.toString().replace("[", "").replace("]", "");
            return ret.isEmpty() ? " " : ret;
        }

        @Override
        public int compareTo(Object t) {
            CommandWrapper cw = (CommandWrapper) t;
            return this.commandName.compareTo(cw.commandName);
        }

    }

    private class DictionaryListener implements AgentCommandDictionaryListener {

        private Map<AgentInfo, Map<String, Dictionary>> dictionaries = new HashMap<>();

        @Override
        public void commandDictionaryUpdate(AgentCommandDictionaryService.AgentCommandDictionaryEvent evt) {
            if (evt.getEventType() == AgentCommandDictionaryEvent.EventType.ADDED) {
                dictionaries.put(evt.getAgentInfo(), evt.getDictionary());
            }
        }

        Map<AgentInfo, Map<String, Dictionary>> getDictionaries() {
            return dictionaries;
        }

    }

    
    public static void main(String[] args) throws Exception {
        DumpCommands agent = new DumpCommands();
        agent.startAgent();

        System.out.println("Wait 15 seconds to collect all dictionaries.");
        try {
            Thread.sleep(15000L);
        } catch (InterruptedException e) {}
        
        String commands = agent.dumpDictionaryCommands();
        
        Path filePath = Paths.get("commandsAndLevels.csv");
        try {
            Files.writeString(filePath, commands, StandardOpenOption.CREATE);
            System.out.println("Commands written to file "+filePath);
        } catch (Exception e) {
            e.printStackTrace();
        }
        
        agent.shutdownAgent();
        System.exit(0);
    }
    
    
}
