package org.lsst.ccs.subsystem.shell;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.lsst.ccs.Agent;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.command.CommandSetBuilder;
import org.lsst.ccs.command.CompositeCommandSet;
import org.lsst.ccs.command.RouteSelectionCommandSet;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.scripting.jython.JythonScriptExecutorUtils;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.AgentCommandDictionaryService.AgentCommandDictionaryEvent;
import org.lsst.ccs.services.AgentCommandDictionaryService.AgentCommandDictionaryListener;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLoginService;
import org.lsst.ccs.shell.JLineShell;
import org.lsst.ccs.subsystems.console.jython.JythonConsoleSocketConnection;

import jline.console.ConsoleReader;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.command.DictionaryUtils;
import org.lsst.ccs.command.Options;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import org.lsst.ccs.command.annotations.Option;

/**
 * JLine-based command shell.
 * Re-factored version of ConsoleBusMaster - allows creation of a command console
 * without creating a subsystem.
 * <p>
 * Static methods of this class can be used to either run the shell standalone
 * or construct <tt>JLineShell</tt> for embedding into the graphical console.
 */
public class ConsoleCommandShell implements AgentCommandDictionaryListener {

    public enum ConsoleParameters {
        TIMEOUT,
        SHOW_JYTHON_OUTPUT
    };

    public enum Visibility {
        VISIBILITY
    };
    
// -- Private parts : ----------------------------------------------------------
    private final AgentMessagingLayer messagingAccess;
    
    private final RouteSelectionCommandSet rsc;
    private final ConcurrentMessagingUtils sci;
    private final BusMasterCommands busMasterCommands = new BusMasterCommands();
    
    private final CompositeCommandSet builtInsCommandSet;
    
    private static final int timeoutMillis = 10000;
    private JythonConsoleSocketConnection jythonConsoleSocketConnection = null;
    private String showJythonOutput = "false";
    
    private final Agent agent;
    
// -- Construction and initialization : ----------------------------------------
    ConsoleCommandShell(Agent agent) {
        this.messagingAccess = agent.getMessagingAccess();
        sci = new ConcurrentMessagingUtils(messagingAccess, Duration.ofMillis(timeoutMillis));
        
        rsc = new RouteSelectionCommandSet();
        CommandSetBuilder builder = new CommandSetBuilder();
        builtInsCommandSet = new CompositeCommandSet();
        builtInsCommandSet.add(builder.buildCommandSet(busMasterCommands));
        // adding the commands for lock and level
        builtInsCommandSet.add(builder.buildCommandSet(new AgentLockServiceCommands(agent.getAgentService(AgentLockService.class), agent.getAgentService(AgentLoginService.class),rsc)));
        

        rsc.add(builtInsCommandSet);        
        
        rsc.getCommandDictionary().setLevelForTypes(-1);        
        rsc.getCommandDictionary().setLevelForTypes(0,Command.CommandType.QUERY);
        
        builtInsCommandSet.getCommandDictionary().setLevelForTypes(Command.NOT_DEFINED);
        
        this.agent = agent;
        agent.getAgentService(AgentCommandDictionaryService.class).addAgentCommandDictionaryListener(this);        
    }
    

// -- Getters : ----------------------------------------------------------------
    public RouteSelectionCommandSet getConsoleCommandSet() {
        return rsc;
    }

    @Override
    public void commandDictionaryUpdate(AgentCommandDictionaryEvent evt) {
        if ( ! evt.getAgentInfo().getType().equals(AgentInfo.AgentType.LISTENER) ) {
            String agentName = evt.getAgentInfo().getName();            
            if ( evt.getEventType().equals(AgentCommandDictionaryEvent.EventType.ADDED) ) {
                if (!rsc.containsPath(agentName)) {                    
                    agent.getScheduler().execute(() -> {
                        for (String path : evt.getDictionary().keySet()) {
                            String route = path;

                            //Make sure the path and route starts with the Agent name
                            //This is necessary because we might remove the agent
                            //name from the dictionary targets
                            if (!route.startsWith(agentName)) {
                                route = agentName + (route.isEmpty() ? "" : "/") + route;
                            }
                            if (!path.startsWith(agentName)) {
                                path = agentName + (path.isEmpty() ? "" : "/") + path;
                            }
                            rsc.addRoutingCommandSet(route, path, new BusCommandSet(sci, path, evt.getDictionary().get(path)));
                        }
                    });
                } 
            } else if ( evt.getEventType().equals(AgentCommandDictionaryEvent.EventType.REMOVED) ) {
                rsc.removeRoute(agentName);                
            } 
        }
    }

    
    
    BusMasterCommands getBusMasterCommands() {
        return busMasterCommands;
    }
    
// -- Local classes : ----------------------------------------------------------  
    /** This must be public for the command invocation to work */
    public class BusMasterCommands {

        public List<String> listSubsystems() {
            return listSubsystems(new Options(),"Worker", true);
        }
        
        @Option(name = "host", description = "Show the host information")        
        @Command(name = "listSubsystems", description = "List the subsystems on the CCS buses for a given Agent type", alias = "ls", type = Command.CommandType.QUERY)
        public List<String> listSubsystems(Options options, 
                @Argument(name = "Agent type", defaultValue="Worker") String agentType, 
                @Argument(defaultValue="true") boolean allTypesAbove) {

            AgentType referenceType = AgentType.valueOf(agentType.toUpperCase());
            List<String> result = new ArrayList<>();
            for ( AgentInfo agent : messagingAccess.getAgentPresenceManager().listConnectedAgents() ) {
                if ( (allTypesAbove && agent.getType().ordinal() >= referenceType.ordinal() ) ||  agent.getType().ordinal() == referenceType.ordinal() ) {
                    result.add(getAgentInformation(agent, options));
                }
                
            }
            java.util.Collections.sort(result);
            return result;
        }
        
        private String getAgentInformation(AgentInfo agentInfo, Options options) {
            StringBuilder sb = new StringBuilder();
            sb.append(agentInfo.getName());
            if ( options.hasOption("host") ) {
                sb.append("[");
                String host = agentInfo.getAgentProperty("org.lsst.ccs.agent.hostname");
                if ( host != null && ! host.isEmpty() ) {
                    sb.append("host=").append(host);
                    String jmxport = agentInfo.getAgentProperty("org.lsst.ccs.agent.jmxport");
                    if ( jmxport != null && ! jmxport.isEmpty() ) {
                        sb.append(":").append(jmxport);
                    }                    
                }
                sb.append("]");
            }
            return sb.toString();
        }

        @Command(description = "Set Console parameters", type = Command.CommandType.QUERY)
        public void set(@Argument(name = "item") ConsoleParameters what, @Argument(name = "value") String value) {
            if ( what == ConsoleParameters.TIMEOUT ) {
                sci.setDefaultTimeout(Duration.ofMillis(Long.valueOf(value)));
            } else if ( what == ConsoleParameters.SHOW_JYTHON_OUTPUT ) {
                if ( jythonConsoleSocketConnection != null ) {
                    jythonConsoleSocketConnection.setPrintStream(value.equals("true") ? System.out : null);
                }
                showJythonOutput = value;
            }
        }

        @Command(description = "Set Dictionary Category Visibility", type = Command.CommandType.QUERY)
        public void set(@Argument(name = "item") Visibility what, @Argument(name = "category") Command.CommandCategory category, boolean visibility) {
            if ( what == Visibility.VISIBILITY ) {
                rsc.getCommandDictionary().setCategoryVisible(category, visibility);
            }
        }
        
        @Command(description = "Get Dictionary Category Visibility", type = Command.CommandType.QUERY)
        public boolean get(@Argument(name = "item") Visibility what, @Argument(name = "category") Command.CommandCategory category) {
            if ( what == Visibility.VISIBILITY ) {
                return rsc.getCommandDictionary().isCategoryVisible(category);
            }
            throw new UnsupportedOperationException("Unknown item "+what);
        }
        
        @Command(description = "Get a Console parameter", type = Command.CommandType.QUERY)
        public String get(@Argument(name = "item") ConsoleParameters what ) {
            if ( what == ConsoleParameters.TIMEOUT ) {
                return String.valueOf(sci.getDefaultTimeout().toMillis());
            } else if ( what == ConsoleParameters.SHOW_JYTHON_OUTPUT ) {
                return showJythonOutput;
            }
            return null;
        }

        @Command(description = "Execute a Jython script", type = Command.CommandType.QUERY)
        public void executeScript(@Argument(name = "scriptPath") String scriptPath, @Argument(name = "scriptArguments") String... args) throws IOException {
            JythonScriptExecutorUtils.executeScript(scriptPath,args);
        }

        
        @Command(description = "Submit a Jython script to a Jython Interpreter", type = Command.CommandType.QUERY)
        public void connectToJythonConsole(
                @Argument(name = "host", defaultValue = "localhost") String host,
                @Argument(name = "port", defaultValue = "4444") int port
        ) throws IOException {
            if ( jythonConsoleSocketConnection != null ) {
                throw new RuntimeException("There is already a connection established. There can be only one! Please \"closeConnectionWithJythonConsole\" before opening a new one.");
            }
            jythonConsoleSocketConnection = new JythonConsoleSocketConnection("CommandShellConnection_"+System.currentTimeMillis(),port,host); 
            jythonConsoleSocketConnection.setPrintStream(showJythonOutput.equals("true") ? System.out : null);
            
        }
        
        
        @Command(description = "Submit a Jython script to a Jython Interpreter", type = Command.CommandType.QUERY)
        public void submitScript(
                @Argument(name = "scriptPath") String scriptPath, @Argument(name = "scriptArgs") String... args
        ) throws IOException {
            if ( jythonConsoleSocketConnection == null ) {
                throw new RuntimeException("No connection was established with a JythonConsole. Please invoke \"connectToJythonConsole\" first");
            }
            jythonConsoleSocketConnection.asynchFileExecution(scriptPath,args);
        }
        
        @Command(description = "Submit a Jython script to a Jython Interpreter", type = Command.CommandType.QUERY)
        public void closeConnectionWithJythonConsole() throws IOException {
            if ( jythonConsoleSocketConnection == null ) {
                throw new RuntimeException("There is open connection to a JythonConsole.");
            }
            jythonConsoleSocketConnection.close();
            jythonConsoleSocketConnection = null;
        }
        
        @Command(type = Command.CommandType.QUERY, description = "Provide a list of normal mode commands")
        public String listNormalModeCommands() {

            StringBuilder sb = new StringBuilder();
            AgentCommandDictionaryService dictService = agent.getAgentService(AgentCommandDictionaryService.class);

            //First build a list of the core commands that are in common with all.        
            sb.append("NORMAL mode commands in common with all agents").append("\n");
            List<DictionaryCommand> commonCommands = new ArrayList<>();
            Map<String, Dictionary> thisDict = dictService.getAgentCommandDictionary();
            for (Map.Entry<String, Dictionary> e : thisDict.entrySet()) {
                Dictionary d = e.getValue();
                for (DictionaryCommand dc : d) {
                    if (dc.getType() != Command.CommandType.QUERY && dc.getLevel() == Command.NORMAL) {
                        sb.append(" ").append(dc.getCommandName()).append(" ").append(dc.getDescription()).append(" (type=").append(dc.getType()).append(")").append("\n");
                        commonCommands.add(dc);
                    }
                }

            }

            boolean hasProcessedWorkerOrService = false;
            boolean hasProcessedImageHanlder = false;

            DictionaryListener listener = new DictionaryListener();
            dictService.addAgentCommandDictionaryListener(listener);
            Map<AgentInfo, Map<String, Dictionary>> dictionaries = listener.getDictionaries();
            for (Map.Entry<AgentInfo, Map<String, Dictionary>> e : dictionaries.entrySet()) {
                AgentInfo ai = e.getKey();
                boolean extractCommonCommands = false;
                if (ai.isAgentWorkerOrService() && !hasProcessedWorkerOrService) {
                    hasProcessedWorkerOrService = true;
                    extractCommonCommands = true;
                }
                Map<String, Dictionary> dicts = e.getValue();
                Map<String, List<DictionaryCommand>> levelZeroCommands = new HashMap<>();
                for (Map.Entry<String, Dictionary> eDict : dicts.entrySet()) {
                    String target = eDict.getKey();
                    Dictionary d = eDict.getValue();

                    for (DictionaryCommand dc : d) {
                        if (dc.getType() != Command.CommandType.QUERY && dc.getLevel() == Command.NORMAL) {
                            if (extractCommonCommands) {
                                switch (dc.getCommandName()) {
                                    case "switchToEngineeringMode":
                                    case "switchToNormalMode":
                                        if (dc.getType().equals(Command.CommandType.ACTION)) {
                                            sb.append(" ").append(dc.getCommandName()).append(" ").append(dc.getDescription()).append(" (type=").append(dc.getType()).append(")").append("\n");
                                            commonCommands.add(dc);
                                        }
                                        break;
                                    case "abort":
                                    case "stop":
                                        if (dc.getType().equals(Command.CommandType.SIGNAL)) {
                                            sb.append(" ").append(dc.getCommandName()).append(" ").append(dc.getDescription()).append(" (type=").append(dc.getType()).append(")").append("\n");
                                            commonCommands.add(dc);
                                        }
                                        break;
                                }
                            }

                            boolean isCommon = false;
                            for (DictionaryCommand common : commonCommands) {
                                if (DictionaryUtils.areDictionaryCommandsEqual(common, dc)) {
                                    isCommon = true;
                                    break;
                                }
                            }
                            if (!isCommon) {
                                levelZeroCommands.computeIfAbsent(target, (t) -> {
                                    return new ArrayList<>();
                                }).add(dc);
                            }
                        }
                    }
                }
                if (!levelZeroCommands.isEmpty()) {
                    if (ai.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, "").equalsIgnoreCase(AgentCategory.IMAGE_HANDLER.name())) {
                        if (!hasProcessedImageHanlder) {
                            hasProcessedImageHanlder = true;
                            sb.append("Agent: ").append(AgentCategory.IMAGE_HANDLER.name()).append("\n");
                        }
                    } else {
                        sb.append("Agent: ").append(ai.getName()).append("\n");
                    }
                    for (Map.Entry<String, List<DictionaryCommand>> eCmd : levelZeroCommands.entrySet()) {
                        String target = eCmd.getKey();
                        if (!target.isEmpty() && !target.equals(ai.getName())) {
                            sb.append(" Target: ").append(target).append("\n");
                        }
                        List<DictionaryCommand> cmds = eCmd.getValue();
                        for (DictionaryCommand dc : cmds) {
                            sb.append("  ").append(dc.getCommandName()).append(" ").append(dc.getDescription());
                            if (dc.getType() != Command.CommandType.ACTION) {
                                sb.append(" (type=").append(dc.getType()).append(")");
                            }
                            sb.append("\n");
                        }
                    }
                } else if (ai.getType().compareTo(AgentInfo.AgentType.WORKER) >= 0) {
                    sb.append("Agent: ").append(ai.getName()).append(" has no additional NORMAL mode commands\n");
                }

            }
            return sb.toString();

        }
        
        
        /**
         * Print the dictionary commands and their levels.
         *
         * @param agentName
         * @param typesToDisplay
         * @param categoryToDisplay
         * @return 
         */
        @org.lsst.ccs.command.annotations.Command(description = "Print dictionary commands and levels for an agent", type = QUERY, category = Command.CommandCategory.USER)
        public String printDictionaryCommandLevels(@Argument(allowedValueProvider = "listSubsystems", defaultValue = "CurrentTarget") String agentName,
                @Argument(defaultValue = "caqs") String typesToDisplay,
                @Argument(defaultValue = "csu") String categoryToDisplay) {
            if ("CurrentTarget".equals(agentName)) {
                agentName = rsc.getActiveRoute();
                if (agentName.contains("/")) {
                    agentName = agentName.substring(0, agentName.indexOf("/"));
                }
            }
            
            typesToDisplay = typesToDisplay.toUpperCase();
            categoryToDisplay = categoryToDisplay.toUpperCase();
            
            AgentCommandDictionaryService dictService = agent.getAgentService(AgentCommandDictionaryService.class);
            DictionaryListener listener = new DictionaryListener();
            dictService.addAgentCommandDictionaryListener(listener);
            Map<AgentInfo, Map<String, Dictionary>> dictionaries = listener.getDictionaries();
            Map<String,Dictionary> dicts = null;
            for (Map.Entry<AgentInfo, Map<String, Dictionary>> e : dictionaries.entrySet()) {
                AgentInfo ai = e.getKey();
                if ( ai.getName().equals(agentName) ) {
                    dicts = e.getValue();
                    break;
                }
            }

            if ( dicts == null ) {
                return "No dictionary found for agent name "+agentName;                                                
            }

            
            TreeMap<Integer,List<String>> commandsSortedByLevel = new TreeMap(); 
            
            for ( Map.Entry<String,Dictionary> e : dicts.entrySet() ) {
                String target = e.getKey();
                for (DictionaryCommand cmd : e.getValue()) {

                    String cat = cmd.getCategory().name().substring(0,1);
                    String type = cmd.getType().name().substring(0,1);

                    if ( !categoryToDisplay.contains(cat) ) {
                        continue;
                    }

                    if ( !typesToDisplay.contains(type) ) {
                        continue;
                    }

                    int level = cmd.getLevel();
                    String fullTarget = cmd.getCommandName();
                    if ( target != null && ! target.isEmpty() ) {
                        fullTarget = target + "/" + fullTarget;
                    }
                    fullTarget = fullTarget.replace(agentName+"/", "");                    
                    fullTarget = "("+type+","+cat+") "+fullTarget;
                    commandsSortedByLevel.computeIfAbsent(level, (l) -> new ArrayList<>()).add(fullTarget);                    
                }
            }
            
            StringBuilder sb = new StringBuilder();
            sb.append("Commands for subsystem ").append(agentName).append(" sorted by level and preceded by (type,category).\n").append("\n");
            for ( Map.Entry<Integer,List<String>> e : commandsSortedByLevel.entrySet() ) {
                sb.append("Commands at level ").append(e.getKey()).append("\n");
                for ( String cmd : e.getValue() ) {
                    sb.append("     ").append(cmd).append("\n");
                }
            }            
            return sb.toString();
        }
        
        
        

        private class DictionaryListener implements AgentCommandDictionaryListener {

            private Map<AgentInfo, Map<String, Dictionary>> dictionaries = new HashMap<>();

            @Override
            public void commandDictionaryUpdate(AgentCommandDictionaryService.AgentCommandDictionaryEvent evt) {
                if (evt.getEventType() == AgentCommandDictionaryEvent.EventType.ADDED) {
                    dictionaries.put(evt.getAgentInfo(), evt.getDictionary());
                }
            }

            Map<AgentInfo, Map<String, Dictionary>> getDictionaries() {
                return dictionaries;
            }

        }
        
    }

// -- Running console as subsystem : -------------------------------------------
    /**
     * Creates a command shell for embedding into external GUI.
     *
     * @param agent The agent in which the Shell is to operate.
     * @param reader JLine console reader to be used for input and output.
     * @throws IOException if the shell fails to initialize.
     */
    static public JLineShell createJLineShell(Agent agent, ConsoleReader reader) throws IOException {
        ConsoleCommandShell comShell = new ConsoleCommandShell(agent);
        return new JLineShell(comShell.getConsoleCommandSet(), reader, "${target} ccs>");
    }
    
    /** Create a ConsoleCommandShell. */
    static public ConsoleCommandShell createConsoleCommandShell(Agent agent) {
        ConsoleCommandShell comShell = new ConsoleCommandShell(agent);
        return comShell;
    }

    /**
     * Runs shell command console in standalone mode as a CCS subsystem named "console".
     */
    static public void main(String[] argv) throws Exception {
        Subsystem busMaster = new Subsystem("ccs-shell",AgentType.CONSOLE);
        busMaster.startAgent();
        ConsoleCommandShell comShell = new ConsoleCommandShell(busMaster);
        JLineShell shell = new JLineShell(comShell.getConsoleCommandSet(), "${target} ccs>");
        shell.run();
        System.exit(0);
    }
    
}
