package org.lsst.ccs.subsystems.console.jython;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.net.util.SubnetUtils;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 *
 * @author turri
 */
class JythonConsoleNetworkUtilities {

    static boolean isInetAddressLocalhost(InetAddress address) throws UnknownHostException {
        if ( address.isAnyLocalAddress() || address.isLoopbackAddress() ) {
            return true;
        }
        InetAddress localhost = InetAddress.getLocalHost();
        return (localhost.getHostAddress().equals(address.getHostAddress()) );
    }
    
    /**
     * Checks if the provided InetAddress is allowed.
     * It will check if the address is in range for any of the SubnetUtils provided.
     *
     * @param address
     * @param subnetUtilsList
     * @return
     */
    static boolean isIpAddressAllowed(String address, List<SubnetUtils> subnetUtilsList) {
        for ( SubnetUtils utils : subnetUtilsList ) {
            if ( utils.getInfo().getAddress().equals(address) || utils.getInfo().isInRange(address) ) {
                return true;
            }
        }
        return false;
    }

    /**
     * Create a list of SubnetUtils by reading the network access list.
     * This is stored in a file called access.list.
     * Each line can be in the form:
     * - simply an allowed ip address 1.2.3.4
     * - a CIDR-notation string, e.g. "192.168.0.1/16"
     * - a dotted decimal address and a dotted decimal mask "192.168.0.1/255.255.0.0"
     * - a comment if it starts with #
     * - empty (ignored)
     *
     * @return the list of SubnetUtils derived from this file
     */
    static List<SubnetUtils> loadNetworkAccessInformation() {

        ArrayList<SubnetUtils> accessList = new ArrayList<>();
        InputStream accessInputStream = BootstrapResourceUtils.getBootstrapResource("access.list");
        if (accessInputStream != null) {
            BufferedReader reader = new BufferedReader(new InputStreamReader(accessInputStream));
            try {
                String line;
                while ((line = reader.readLine()) != null) {

                    if (!line.isEmpty() && !line.startsWith("#")) {
                        String address, mask;
                        int separator = line.indexOf("/");
                        if (separator > -1) {
                            address = line.substring(0, separator);
                            mask = line.substring(separator + 1);
                        } else {
                            address = line;
                            mask = "255.255.255.255";
                        }

                        SubnetUtils subnetUtils;
                        try {
                            if (mask.contains(".")) {
                                subnetUtils = new SubnetUtils(address, mask);
                            } else {
                                subnetUtils = new SubnetUtils(address + "/" + mask);
                            }
                            accessList.add(subnetUtils);
                        } catch (IllegalArgumentException iae) {
                            System.out.println("**** WARNING: could not process access list information: " + line);
                        }

                    }
                }
            } catch (IOException ioe) {
                throw new RuntimeException("Could not read file access.list", ioe);
            }
        }


        return accessList;
    }
}
