package org.lsst.ccs.subsystem.shell;

import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.command.RouteSelectionCommandSet;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLoginService;

/**
 * Commands for agent lock service.
 * @author emarin
 */
public class AgentLockServiceCommands {
    
    private final AgentLockService als;
    private final AgentLoginService loginService;
    private final RouteSelectionCommandSet rsc;
    
    AgentLockServiceCommands(AgentLockService als, AgentLoginService loginService, RouteSelectionCommandSet rsc) {
        this.als = als;
        this.loginService = loginService;
        this.rsc = rsc;
    }
    
    @Deprecated
    @Command(description = "Set this agent's authorized operational level for the given subsystem")
    public void setLevel(@Argument(allowedValueProvider = "getAllLockedOrLockableAgents") String agentName, int level) throws Exception {
        System.out.println("DEPRECATED: use \"set level AGENT_NAME LEVEL\" instead");
        als.setLevelForAgent(agentName, level);
    }
    
    @Deprecated
    @Command(description = "Get this agent's authorized operatinal level for the given subsystem")
    public int getLevel(@Argument(allowedValueProvider = "getAllLockedOrLockableAgents") String agentName) {
        System.out.println("DEPRECATED: use \"get level AGENT_NAME\" instead");
        return als.getLevelForAgent(agentName);
    }
    
    public enum SetGetLevelCommands {
        LEVEL
    };


    @Command(description = "Set the level of an agent", level = Command.NORMAL)
    public void set(@Argument(name = "item") SetGetLevelCommands what, @Argument(allowedValueProvider = "getAllLockedOrLockableAgents") String agentName, int level) throws Exception {
        switch (what) {
            case LEVEL:
                als.setLevelForAgent(agentName, level);
        }
    }

    @Command(description = "Set the level of an agent", level = Command.NORMAL)
    public void set(@Argument(name = "item") SetGetLevelCommands what, int level) throws Exception {
        String agentName = rsc.getActiveRoute();
        if ( agentName.contains("/") ) {
            agentName = agentName.substring(0,agentName.indexOf("/"));
        }
        set(what,agentName,level);
    }

    @Command(description = "Get the level of an agent", level = Command.NORMAL)
    public int get(@Argument(name = "item") SetGetLevelCommands what, @Argument(allowedValueProvider = "getAllLockedOrLockableAgents") String agentName) throws Exception {
        switch (what) {
            case LEVEL:
                return als.getLevelForAgent(agentName);
        }
        return -1;
    }
    
    @Command(description = "Get the level of an agent", level = Command.NORMAL)
    public int get(@Argument(name = "item") SetGetLevelCommands what) throws Exception {
        String agentName = rsc.getActiveRoute();
        if ( agentName.contains("/") ) {
            agentName = agentName.substring(0,agentName.indexOf("/"));
        }
        return get(what,agentName);
    }
        
    @Command(description = "Lock the given subsystem")
    public void lock(@Argument(allowedValueProvider = "getLockableAgents", defaultValue = "CurrentTarget") String agentName) throws Exception {
        if ( "CurrentTarget".equals(agentName) ) {
            agentName = rsc.getActiveRoute();
            if (agentName.contains("/")) {
                agentName = agentName.substring(0, agentName.indexOf("/"));
            }
        }
        als.lockAgent(agentName);
    }
    
    @Command(description = "Unlock the given subsystem")
    public void unlock(@Argument(allowedValueProvider = "getLockedAgents", defaultValue = "CurrentTarget") String agentName) throws Exception {
        if ( "CurrentTarget".equals(agentName) ) {
            agentName = rsc.getActiveRoute();
            if (agentName.contains("/")) {
                agentName = agentName.substring(0, agentName.indexOf("/"));
            }
        }
        als.unlockAgent(agentName);
    }

    @Command(description = "Attach the lock on the given subsystem to the current agent")
    public void attachLock(@Argument(allowedValueProvider = "getAttachableAgents", defaultValue = "CurrentTarget") String agentName) throws Exception {
        if ( "CurrentTarget".equals(agentName) ) {
            agentName = rsc.getActiveRoute();
            if (agentName.contains("/")) {
                agentName = agentName.substring(0, agentName.indexOf("/"));
            }
        }
        als.attachLock(agentName);
    }

    @Command(description = "Detach the lock on the given subsystem from the current agent")
    public void detachLock(@Argument(allowedValueProvider = "getDetachableAgents", defaultValue = "CurrentTarget") String agentName) throws Exception {
        if ( "CurrentTarget".equals(agentName) ) {
            agentName = rsc.getActiveRoute();
            if (agentName.contains("/")) {
                agentName = agentName.substring(0, agentName.indexOf("/"));
            }
        }
        als.detachLock(agentName);
    }

    @Command(description = "List subsystems locked by the current user")
    public List<String> getLockedAgents() {
        return als.getLockedAgents();
    }
    
    @Command(description = "List subsystems that are currently online and can be locked")
    public List<String> getLockableAgents() {
        return als.getLockableAgents();
    }

    @Command(description = "List locked subsystems that could be attached to the current agent")
    public List<String> getAttachableAgents() {
        return als.getAttachableAgents();

    }
    
    @Command(description = "List locked and attached subsystems that could be detached from the current agent")
    public List<String> getDetachableAgents() {
        return als.getDetachableAgents();
    }
    
    public List<String> getAllLockedOrLockableAgents() {
        List<String> res = als.getLockableAgents();
        res.addAll(als.getLockedAgents());
        return res;
    }

    @Command(description = "Show lock for the specified subsystem")
    public String showLock(@Argument(allowedValueProvider = "getAllLockedOrLockableAgents", defaultValue = "CurrentTarget") String agentName) {
        if ( "CurrentTarget".equals(agentName) ) {
            agentName = rsc.getActiveRoute();
            if (agentName.contains("/")) {
                agentName = agentName.substring(0, agentName.indexOf("/"));
            }
        }
        if (agentName.trim().isEmpty()) {
            return "Subsystem not specified";
        }
        AgentLock lock = als.getExistingLockForAgent(agentName);
        if (lock == null) {
            return "Unlocked";
        } else {
            return "Locked by "+ lock.getOwner();
        }
    }

    @Command(description = "List all locks owned by the specified user. If no user name is given, list locks owned by the current user.")
    public String showLocks(@Argument(defaultValue = "") String user) {
        if (user == null || user.trim().isEmpty()) {
            user = loginService.getUserId();
        }
        StringBuilder sb = new StringBuilder();
        sb.append("Locks owned by user ").append(user).append(":\n");
        for (String subsystem: als.getLockedAgents()) {
            AgentLock lock = als.getExistingLockForAgent(subsystem);
            if (lock != null) {
                sb.append(subsystem).append(" => Token = ").append(lock.getToken()).append("\n");
            }
        }
        return sb.toString();
    }

    @Command(description = "List attached locks")
    public String showLocalLocks() {
        StringBuilder sb = new StringBuilder();
        for (String subsystem: als.getDetachableAgents()) {
            AgentLock lock = als.getExistingLockForAgent(subsystem);
            if (lock != null) {
                sb.append(subsystem).append(" => Token = ").append(lock.getToken()).append("\n");
            }
        }
        return sb.toString();
    }

    @Command(description = "List all existing locks")
    public String showAllLocks() {
        Map<String,AgentLock> locks = new TreeMap<>(als.getLocks());
        StringBuilder sb = new StringBuilder();
        locks.forEach((agent, lock) -> {
            if (lock != null) {
                sb.append(agent).append(" : ").append(lock.getOwner()).append("\n");
            }
        });
        return sb.toString();
    }
    
    @Command(description = "Set a specific user ID for the current console")
    public void login(String userId) throws Exception {
        loginService.login(userId, null);
    }

    @Command(description = "Disconnect the user ID from the current console")
    public void logout() throws Exception {
        loginService.disconnect();
    }
}
