package org.lsst.ccs.subsystem.shell;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

import org.lsst.ccs.Agent;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.command.CommandSetBuilder;
import org.lsst.ccs.command.CompositeCommandSet;
import org.lsst.ccs.command.RouteSelectionCommandSet;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.scripting.jython.JythonScriptExecutorUtils;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.AgentCommandDictionaryService.AgentCommandDictionaryEvent;
import org.lsst.ccs.services.AgentCommandDictionaryService.AgentCommandDictionaryListener;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLoginService;
import org.lsst.ccs.shell.JLineShell;
import org.lsst.ccs.subsystems.console.jython.JythonConsoleSocketConnection;

import jline.console.ConsoleReader;
import org.lsst.ccs.command.Options;
import org.lsst.ccs.command.annotations.Option;

/**
 * JLine-based command shell.
 * Re-factored version of ConsoleBusMaster - allows creation of a command console
 * without creating a subsystem.
 * <p>
 * Static methods of this class can be used to either run the shell standalone
 * or construct <tt>JLineShell</tt> for embedding into the graphical console.
 */
public class ConsoleCommandShell implements AgentCommandDictionaryListener {

    public enum ConsoleParameters {
        TIMEOUT,
        SHOW_JYTHON_OUTPUT
    };

    public enum Visibility {
        VISIBILITY
    };
    
// -- Private parts : ----------------------------------------------------------
    private final AgentMessagingLayer messagingAccess;
    
    private final RouteSelectionCommandSet rsc;
    private final ConcurrentMessagingUtils sci;
    private final BusMasterCommands busMasterCommands = new BusMasterCommands();
    
    private final CompositeCommandSet builtInsCommandSet;
    
    private static final int timeoutMillis = 10000;
    private JythonConsoleSocketConnection jythonConsoleSocketConnection = null;
    private String showJythonOutput = "false";
    
    private final Agent agent;
    
// -- Construction and initialization : ----------------------------------------
    ConsoleCommandShell(Agent agent) {
        this.messagingAccess = agent.getMessagingAccess();
        sci = new ConcurrentMessagingUtils(messagingAccess, Duration.ofMillis(timeoutMillis));
        
        rsc = new RouteSelectionCommandSet();
        CommandSetBuilder builder = new CommandSetBuilder();
        builtInsCommandSet = new CompositeCommandSet();
        builtInsCommandSet.add(builder.buildCommandSet(busMasterCommands));
        // adding the commands for lock and level
        builtInsCommandSet.add(builder.buildCommandSet(new AgentLockServiceCommands(agent.getAgentService(AgentLockService.class), agent.getAgentService(AgentLoginService.class),rsc)));
        

        rsc.add(builtInsCommandSet);        
        
        rsc.getCommandDictionary().setLevelForTypes(-1);        
        rsc.getCommandDictionary().setLevelForTypes(0,Command.CommandType.QUERY);
        
        this.agent = agent;
        agent.getAgentService(AgentCommandDictionaryService.class).addAgentCommandDictionaryListener(this);        
    }
    

// -- Getters : ----------------------------------------------------------------
    public RouteSelectionCommandSet getConsoleCommandSet() {
        return rsc;
    }

    @Override
    public void commandDictionaryUpdate(AgentCommandDictionaryEvent evt) {
        if ( ! evt.getAgentInfo().getType().equals(AgentInfo.AgentType.LISTENER) ) {
            String agentName = evt.getAgentInfo().getName();            
            if ( evt.getEventType().equals(AgentCommandDictionaryEvent.EventType.ADDED) ) {
                if (!rsc.containsPath(agentName)) {                    
                    agent.getScheduler().execute(() -> {
                        long start = System.currentTimeMillis();
                        boolean useLongPaths = "true".equals(evt.getAgentInfo().getAgentProperty("org.lsst.ccs.use.full.paths", "false").toLowerCase());

                        for (String path : evt.getDictionary().keySet()) {
                            String route = path;
                            if (!useLongPaths) {
                                int first = path.indexOf("/");
                                int last = path.lastIndexOf("/");
                                if (first != last) {
                                    route = path.substring(0, first) + path.substring(last);
                                }
                            }

                            //Make sure the path and route starts with the Agent name
                            //This is necessary because we might remove the agent
                            //name from the dictionary targets
                            if (!route.startsWith(agentName)) {
                                route = agentName + (route.isEmpty() ? "" : "/") + route;
                            }
                            if (!path.startsWith(agentName)) {
                                path = agentName + (path.isEmpty() ? "" : "/") + path;
                            }
                            rsc.addRoutingCommandSet(route, path, new BusCommandSet(sci, path, evt.getDictionary().get(path)));
                        }
                    });
                } 
            } else if ( evt.getEventType().equals(AgentCommandDictionaryEvent.EventType.REMOVED) ) {
                rsc.removeRoute(agentName);                
            } 
        }
    }

    
    
    BusMasterCommands getBusMasterCommands() {
        return busMasterCommands;
    }
    
// -- Local classes : ----------------------------------------------------------  
    /** This must be public for the command invocation to work */
    public class BusMasterCommands {

        public List<String> listSubsystems() {
            return listSubsystems(new Options(),"Worker", true);
        }
        
        @Option(name = "host", description = "Show the host information")        
        @Command(name = "listSubsystems", description = "List the subsystems on the CCS buses for a given Agent type", alias = "ls")
        public List<String> listSubsystems(Options options, 
                @Argument(name = "Agent type", defaultValue="Worker") String agentType, 
                @Argument(defaultValue="true") boolean allTypesAbove) {

            AgentType referenceType = AgentType.valueOf(agentType.toUpperCase());
            List<String> result = new ArrayList<>();
            for ( AgentInfo agent : messagingAccess.getAgentPresenceManager().listConnectedAgents() ) {
                if ( (allTypesAbove && agent.getType().ordinal() >= referenceType.ordinal() ) ||  agent.getType().ordinal() == referenceType.ordinal() ) {
                    result.add(getAgentInformation(agent, options));
                }
                
            }
            java.util.Collections.sort(result);
            return result;
        }
        
        private String getAgentInformation(AgentInfo agentInfo, Options options) {
            StringBuilder sb = new StringBuilder();
            sb.append(agentInfo.getName());
            if ( options.hasOption("host") ) {
                sb.append("[");
                String host = agentInfo.getAgentProperty("org.lsst.ccs.agent.hostname");
                if ( host != null && ! host.isEmpty() ) {
                    sb.append("host=").append(host);
                    String jmxport = agentInfo.getAgentProperty("org.lsst.ccs.agent.jmxport");
                    if ( jmxport != null && ! jmxport.isEmpty() ) {
                        sb.append(":").append(jmxport);
                    }                    
                }
                sb.append("]");
            }
            return sb.toString();
        }

        @Command(description = "Set Console parameters")
        public void set(@Argument(name = "item") ConsoleParameters what, @Argument(name = "value") String value) {
            if ( what == ConsoleParameters.TIMEOUT ) {
                sci.setDefaultTimeout(Duration.ofMillis(Long.valueOf(value)));
            } else if ( what == ConsoleParameters.SHOW_JYTHON_OUTPUT ) {
                if ( jythonConsoleSocketConnection != null ) {
                    jythonConsoleSocketConnection.setPrintStream(value.equals("true") ? System.out : null);
                }
                showJythonOutput = value;
            }
        }

        @Command(description = "Set Dictionary Category Visibility")
        public void set(@Argument(name = "item") Visibility what, @Argument(name = "category") Command.CommandCategory category, boolean visibility) {
            if ( what == Visibility.VISIBILITY ) {
                rsc.getCommandDictionary().setCategoryVisible(category, visibility);
            }
        }
        
        @Command(description = "Get Dictionary Category Visibility")
        public boolean get(@Argument(name = "item") Visibility what, @Argument(name = "category") Command.CommandCategory category) {
            if ( what == Visibility.VISIBILITY ) {
                return rsc.getCommandDictionary().isCategoryVisible(category);
            }
            throw new UnsupportedOperationException("Unknown item "+what);
        }
        
        @Command(description = "Get a Console parameter")
        public String get(@Argument(name = "item") ConsoleParameters what ) {
            if ( what == ConsoleParameters.TIMEOUT ) {
                return String.valueOf(sci.getDefaultTimeout().toMillis());
            } else if ( what == ConsoleParameters.SHOW_JYTHON_OUTPUT ) {
                return showJythonOutput;
            }
            return null;
        }

        @Command(description = "Execute a Jython script")
        public void executeScript(@Argument(name = "scriptPath") String scriptPath, @Argument(name = "scriptArguments") String... args) throws IOException {
            JythonScriptExecutorUtils.executeScript(scriptPath,args);
        }

        
        @Command(description = "Submit a Jython script to a Jython Interpreter")
        public void connectToJythonConsole(
                @Argument(name = "host", defaultValue = "localhost") String host,
                @Argument(name = "port", defaultValue = "4444") int port
        ) throws IOException {
            if ( jythonConsoleSocketConnection != null ) {
                throw new RuntimeException("There is already a connection established. There can be only one! Please \"closeConnectionWithJythonConsole\" before opening a new one.");
            }
            jythonConsoleSocketConnection = new JythonConsoleSocketConnection("CommandShellConnection_"+System.currentTimeMillis(),port,host); 
            jythonConsoleSocketConnection.setPrintStream(showJythonOutput.equals("true") ? System.out : null);
            
        }
        
        
        @Command(description = "Submit a Jython script to a Jython Interpreter")
        public void submitScript(
                @Argument(name = "scriptPath") String scriptPath, @Argument(name = "scriptArgs") String... args
        ) throws IOException {
            if ( jythonConsoleSocketConnection == null ) {
                throw new RuntimeException("No connection was established with a JythonConsole. Please invoke \"connectToJythonConsole\" first");
            }
            jythonConsoleSocketConnection.asynchFileExecution(scriptPath,args);
        }
        
        @Command(description = "Submit a Jython script to a Jython Interpreter")
        public void closeConnectionWithJythonConsole() throws IOException {
            if ( jythonConsoleSocketConnection == null ) {
                throw new RuntimeException("There is open connection to a JythonConsole.");
            }
            jythonConsoleSocketConnection.close();
            jythonConsoleSocketConnection = null;
        }
        
        
    }

// -- Running console as subsystem : -------------------------------------------
    /**
     * Creates a command shell for embedding into external GUI.
     *
     * @param agent The agent in which the Shell is to operate.
     * @param reader JLine console reader to be used for input and output.
     * @throws IOException if the shell fails to initialize.
     */
    static public JLineShell createJLineShell(Agent agent, ConsoleReader reader) throws IOException {
        ConsoleCommandShell comShell = new ConsoleCommandShell(agent);
        return new JLineShell(comShell.getConsoleCommandSet(), reader, "${target} ccs>");
    }
    
    /** Create a ConsoleCommandShell. */
    static public ConsoleCommandShell createConsoleCommandShell(Agent agent) {
        ConsoleCommandShell comShell = new ConsoleCommandShell(agent);
        return comShell;
    }

    /**
     * Runs shell command console in standalone mode as a CCS subsystem named "console".
     */
    static public void main(String[] argv) throws Exception {
        Subsystem busMaster = new Subsystem("ccs-shell",AgentType.CONSOLE);
        busMaster.startAgent();
        ConsoleCommandShell comShell = new ConsoleCommandShell(busMaster);
        JLineShell shell = new JLineShell(comShell.getConsoleCommandSet(), "${target} ccs>");
        shell.run();
        System.exit(0);
    }
    
}
