package org.lsst.ccs.subsystem.shell;

import java.util.List;

import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.command.RouteSelectionCommandSet;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLoginService;

/**
 * Commands for agent lock service.
 * @author emarin
 */
public class AgentLockServiceCommands {
    
    private final AgentLockService als;
    private final AgentLoginService loginService;
    private final RouteSelectionCommandSet rsc;
    
    AgentLockServiceCommands(AgentLockService als, AgentLoginService loginService, RouteSelectionCommandSet rsc) {
        this.als = als;
        this.loginService = loginService;
        this.rsc = rsc;
    }
    
    @Command(description = "Set this agent's authorized operational level for the given subsystem")
    public void setLevel(@Argument(allowedValueProvider = "getAllWorkerAgents") String agentName, int level) throws Exception {
        System.out.println("DEPRECATED: use \"set level AGENT_NAME LEVEL\" instead");
        als.setLevelForAgent(agentName, level);
    }
    
    @Command(description = "Get this agent's authorized operatinal level for the given subsystem")
    public int getLevel(@Argument(allowedValueProvider = "getAllWorkerAgents") String agentName) {
        System.out.println("DEPRECATED: use \"get level AGENT_NAME\" instead");
        return als.getLevelForAgent(agentName);
    }
    
    public enum SetGetLevelCommands {
        LEVEL
    };


    @Command(description = "Set the level of an agent", level = Command.NORMAL)
    public void set(@Argument(name = "item") SetGetLevelCommands what, @Argument(allowedValueProvider = "getAllWorkerAgents") String agentName, int level) throws Exception {
        switch (what) {
            case LEVEL:
                als.setLevelForAgent(agentName, level);
        }
    }

    @Command(description = "Set the level of an agent", level = Command.NORMAL)
    public void set(@Argument(name = "item") SetGetLevelCommands what, int level) throws Exception {
        String agentName = rsc.getActiveRoute();
        if ( agentName.contains("/") ) {
            agentName = agentName.substring(0,agentName.indexOf("/"));
        }
        set(what,agentName,level);
    }

    @Command(description = "Get the level of an agent", level = Command.NORMAL)
    public int get(@Argument(name = "item") SetGetLevelCommands what, @Argument(allowedValueProvider = "getAllWorkerAgents") String agentName) throws Exception {
        switch (what) {
            case LEVEL:
                return als.getLevelForAgent(agentName);
        }
        return -1;
    }
    
    @Command(description = "Get the level of an agent", level = Command.NORMAL)
    public int get(@Argument(name = "item") SetGetLevelCommands what) throws Exception {
        String agentName = rsc.getActiveRoute();
        if ( agentName.contains("/") ) {
            agentName = agentName.substring(0,agentName.indexOf("/"));
        }
        return get(what,agentName);
    }
        
    @Command(description = "Lock the given subsystem")
    public void lock(@Argument(allowedValueProvider = "getLockableAgents", defaultValue = "CurrentTarget") String agentName) throws Exception {
        if ( "CurrentTarget".equals(agentName) ) {
            agentName = rsc.getActiveRoute();
            if (agentName.contains("/")) {
                agentName = agentName.substring(0, agentName.indexOf("/"));
            }
        }
        als.lockAgent(agentName);
    }
    
    @Command(description = "Unlock the given subsystem")
    public void unlock(@Argument(allowedValueProvider = "getLockedAgents", defaultValue = "CurrentTarget") String agentName) throws Exception {
        if ( "CurrentTarget".equals(agentName) ) {
            agentName = rsc.getActiveRoute();
            if (agentName.contains("/")) {
                agentName = agentName.substring(0, agentName.indexOf("/"));
            }
        }
        als.unlockAgent(agentName);
    }

    @Command(description = "Attach the lock on the given subsystem to the current agent")
    public void attachLock(@Argument(allowedValueProvider = "getAttachableAgents", defaultValue = "CurrentTarget") String agentName) throws Exception {
        if ( "CurrentTarget".equals(agentName) ) {
            agentName = rsc.getActiveRoute();
            if (agentName.contains("/")) {
                agentName = agentName.substring(0, agentName.indexOf("/"));
            }
        }
        als.attachLock(agentName);
    }

    @Command(description = "Detach the lock on the given subsystem from the current agent")
    public void detachLock(@Argument(allowedValueProvider = "getDetachableAgents", defaultValue = "CurrentTarget") String agentName) throws Exception {
        if ( "CurrentTarget".equals(agentName) ) {
            agentName = rsc.getActiveRoute();
            if (agentName.contains("/")) {
                agentName = agentName.substring(0, agentName.indexOf("/"));
            }
        }
        als.detachLock(agentName);
    }

    @Command(description = "Return locked agents")
    public List<String> getLockedAgents() {
        return als.getLockedAgents();
    }
    
    @Command(description = "Return agents that could be locked")
    public List<String> getLockableAgents() {
        return als.getLockableAgents();
    }

    @Command(description = "Return locked agents that could be attached to the current agent")
    public List<String> getAttachableAgents() {
        return als.getAttachableAgents();

    }
    @Command(description = "Return locked agents that could be detached from the current agent")
    public List<String> getDetachableAgents() {
        return als.getDetachableAgents();
    }
    
    public List<String> getAllWorkerAgents() {
        return als.getAllWorkerAgents();
    }

    @Command(description = "Show owned locks")
    public void showLocks() {
        for (String subsystem: als.getLockedAgents()) {
            AgentLock lock = als.getExistingLockForAgent(subsystem);
            System.out.printf("%-15s => Token = %s\n", subsystem, lock.getToken());
        }
    }

    @Command(description = "Show attached locks")
    public void showLocalLocks() {
        for (String subsystem: als.getDetachableAgents()) {
            AgentLock lock = als.getLockForAgent(subsystem);
            System.out.printf("%-15s => Token = %s\n", subsystem, lock.getToken());
        }
    }
    
    @Command(description = "Set a specific user ID for the current console")
    public void login(String userId) throws Exception {
        loginService.login(userId, null);
    }

    @Command(description = "Disconnect the user ID from the current console")
    public void logout() throws Exception {
        loginService.disconnect();
    }
}
