package org.lsst.ccs.subsystem.shell;

import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.command.BasicCommand;
import org.lsst.ccs.command.CommandArgumentMatchException;
import org.lsst.ccs.command.CommandInvocationException;
import org.lsst.ccs.command.CommandSet;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.command.SupportedOption;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;

/**
 * A command set which invokes commands by sending messages on the CCS command
 * bus.
 *
 * @author tonyj
 */
class BusCommandSet implements CommandSet {

    private final String dictionaryPath;
    private final Dictionary dict;
    private final ConcurrentMessagingUtils sci;

    BusCommandSet(ConcurrentMessagingUtils sci, String dictionaryPath, Dictionary dict) {
        this.sci = sci;
        this.dictionaryPath = dictionaryPath;
        this.dict = dict;
    }

    @Override
    public Dictionary getCommandDictionary() {
        return dict;
    }

    @Override
    public Object invoke(BasicCommand bc) throws CommandInvocationException, CommandArgumentMatchException {

        DictionaryCommand findCommand = dict.findCommand(bc);
        if (findCommand == null) {
            throw new CommandInvocationException("No handler found for command %s with %d arguments", bc.getCommand(), bc.getArgumentCount());
        }

        if (findCommand.hasOptions()) {
            for (String opt : bc.getOptions().getOptions()) {
                boolean isOptionSupported = false;
                boolean isShortOption = opt.length() == 1;
                for (SupportedOption so : findCommand.getSupportedOptions()) {
                    isOptionSupported = isShortOption ? so.getSingleLetterName().equals(opt) : so.getName().equals(opt);
                    if (isOptionSupported) {
                        break;
                    }
                }
                if (!isOptionSupported) {
                    throw new CommandInvocationException("Option " + opt + " is not one of the supported options: " + findCommand.getSupportedOptions());
                }
            }
        } else if (bc.getOptions().hasOptions()) {
            throw new CommandInvocationException("Options cannot be provided to a command that does not support them.");
        }
        
        CommandRequest basicCommandBusMessage = new CommandRequest(dictionaryPath, bc);

        try {
            return sci.sendSynchronousCommand(basicCommandBusMessage);
        } catch (Exception ex) {
            throw new CommandInvocationException("Error dispatching command: " + ex.getMessage(), ex);
        }
    }

}
