package org.lsst.ccs.subsystem.cluster.monitor;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.time.Duration;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.CommandMessageListener;
import org.lsst.ccs.messaging.LogMessageListener;
import org.lsst.ccs.messaging.StatusMessageListener;

/**
 * A Subsystem that tracks the bus traffic.
 *
 * @author The LSST CCS Team
 */
public class BusTrafficMonitor implements HasLifecycle {

    private static final Object counterLock = new Object();
    private static Counter counter = new Counter();
    private static Accumulate sizeAccumulator = new Accumulate();
    private static Accumulate timeAccumulator = new Accumulate();
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @Override
    public void build() {
        AgentPeriodicTask resetCounterAndPublish = new AgentPeriodicTask("busTrafficStatsUpdate",
                () -> {
                    resetCounterAndPublish();
                }).withPeriod(Duration.ofSeconds(60));

        agent.getAgentPeriodicTaskService().scheduleAgentPeriodicTask(resetCounterAndPublish);
    }

    @Override
    public void start() {
        agent.getMessagingAccess().addCommandMessageListener(new CommandBusCounterListener());
        agent.getMessagingAccess().addStatusMessageListener(new StatusBusCounterListener());
        agent.getMessagingAccess().addLogMessageListener(new LogBusCounterListener());
    }

    private void resetCounterAndPublish() {
        Counter oldCounter = counter;
        Accumulate oldSizeAccumulator = sizeAccumulator;
        Accumulate oldTimeAccumulator = timeAccumulator;
        synchronized (counterLock) {
            counter = new Counter();
            sizeAccumulator = new Accumulate();
            timeAccumulator = new Accumulate();
        }
        KeyValueData counterData = new KeyValueData("traffic", (Serializable) oldCounter.getCountersMap());
        agent.publishSubsystemDataOnStatusBus(counterData);
        KeyValueData sizeData = new KeyValueData("size", (Serializable) oldSizeAccumulator.getAverageMap());
        agent.publishSubsystemDataOnStatusBus(sizeData);
        KeyValueData timeData = new KeyValueData("time", (Serializable) oldTimeAccumulator.getAverageMap());
        agent.publishSubsystemDataOnStatusBus(timeData);
    }

    private static class CommandBusCounterListener implements CommandMessageListener {

        String busName = "Command";

        @Override
        public void onCommandMessage(CommandMessage msg) {
            long deltaT = System.currentTimeMillis() - msg.getTimeStamp();
            String busKey = msg.getOriginAgentInfo().getType().ordinal() >= AgentInfo.AgentType.WORKER.ordinal() ? msg.getOriginAgentInfo().getName() : "client";
            busKey += "/" + busName;
            String busMessageTypeKey = busName + "/" + msg.getClass().getSimpleName();
            String messageKey = busKey + "/" + msg.getClass().getSimpleName();
            double size = sizeOf(msg);
            synchronized (counterLock) {
                counter.increment(busName + "/all");
                counter.increment(busMessageTypeKey);
                counter.increment(busKey + "/all");
                counter.increment(messageKey);
                sizeAccumulator.accumulate(size, busName + "/all");
                sizeAccumulator.accumulate(size, busMessageTypeKey);
                sizeAccumulator.accumulate(size, busKey + "/all");
                sizeAccumulator.accumulate(size, messageKey);
                timeAccumulator.accumulate(deltaT, busName + "/all");
                timeAccumulator.accumulate(deltaT, busMessageTypeKey);
                timeAccumulator.accumulate(deltaT, busKey + "/all");
                timeAccumulator.accumulate(deltaT, messageKey);
            }
        }

    }

    private static class LogBusCounterListener implements LogMessageListener {

        String busName = "Log";

        @Override
        public void onLogMessage(LogMessage msg) {
            long deltaT = System.currentTimeMillis() - msg.getTimeStamp();
            String busKey = msg.getOriginAgentInfo().getType().ordinal() >= AgentInfo.AgentType.WORKER.ordinal() ? msg.getOriginAgentInfo().getName() : "client";
            busKey += "/" + busName;
            double size = sizeOf(msg);
            synchronized (counterLock) {
                counter.increment(busName + "/all");
                counter.increment(busKey);
                sizeAccumulator.accumulate(size, busName + "/all");
                sizeAccumulator.accumulate(size, busKey);
                timeAccumulator.accumulate(deltaT, busName + "/all");
                timeAccumulator.accumulate(deltaT, busKey);
            }
        }
    }

    private static class StatusBusCounterListener implements StatusMessageListener {

        String busName = "Status";

        @Override
        public void onStatusMessage(StatusMessage msg) {
            long deltaT = System.currentTimeMillis() - msg.getTimeStamp();
            String busKey = msg.getOriginAgentInfo().getType().ordinal() >= AgentInfo.AgentType.WORKER.ordinal() ? msg.getOriginAgentInfo().getName() : "client";
            busKey += "/" + busName;
            String messageKey = busKey + "/" + msg.getClass().getSimpleName();
            String busMessageTypeKey = busName + "/" + msg.getClass().getSimpleName();
            double size = sizeOf(msg);
            synchronized (counterLock) {
                counter.increment(busName + "/all");
                counter.increment(busKey + "/all");
                counter.increment(busMessageTypeKey);
                counter.increment(messageKey);
                sizeAccumulator.accumulate(size, busName + "/all");
                sizeAccumulator.accumulate(size, busMessageTypeKey);
                sizeAccumulator.accumulate(size, busKey + "/all");
                sizeAccumulator.accumulate(size, messageKey);
                timeAccumulator.accumulate(deltaT, busName + "/all");
                timeAccumulator.accumulate(deltaT, busMessageTypeKey);
                timeAccumulator.accumulate(deltaT, busKey + "/all");
                timeAccumulator.accumulate(deltaT, messageKey);
            }
        }

    }

    public static long sizeOf(Serializable object) {
        ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
        try (ObjectOutputStream objectOutputStream = new ObjectOutputStream(byteOutputStream)) {
            objectOutputStream.writeObject(object);
            objectOutputStream.flush();
        } catch (IOException ieo) {
            return 0;
        }

        return byteOutputStream.toByteArray().length;
    }
}
