package org.lsst.ccs.subsystem.shell;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import jline.console.ConsoleReader;
import org.lsst.ccs.Agent;
import org.lsst.ccs.BusMaster;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.command.CommandSetBuilder;
import org.lsst.ccs.command.CompositeCommandSet;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.RouteSelectionCommandSet;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.AgentPresenceManager;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.scripting.jython.JythonScriptExecutorUtils;
import org.lsst.ccs.shell.JLineShell;
import org.lsst.ccs.subsystems.console.jython.JythonConsoleSocketConnection;
import org.lsst.ccs.services.AgentLockService;

/**
 * JLine-based command shell.
 * Re-factored version of ConsoleBusMaster - allows creation of a command console
 * without creating a subsystem.
 * <p>
 * Static methods of this class can be used to either run the shell standalone
 * or construct <tt>JLineShell</tt> for embedding into the graphical console.
 */
public class ConsoleCommandShell implements AgentPresenceListener, AgentLockService.AgentLevelListener, AgentLockService.AgentLockUpdateListener {

    public enum ConsoleParameters {
        TIMEOUT,
        SHOW_JYTHON_OUTPUT
    };

// -- Private parts : ----------------------------------------------------------
    private final AgentMessagingLayer messagingAccess;
    
    private final RouteSelectionCommandSet rsc;
    private final ConcurrentMessagingUtils sci;
    private final BusMasterCommands busMasterCommands = new BusMasterCommands();
    
    private final CompositeCommandSet builtInsCommandSet;
    
    private static final int timeoutMillis = 10000;
    private JythonConsoleSocketConnection jythonConsoleSocketConnection = null;
    private String showJythonOutput = "false";
    private final Agent agent;

// -- Construction and initialization : ----------------------------------------
    ConsoleCommandShell(Agent agent) {
        this.messagingAccess = agent.getMessagingAccess();
        sci = new ConcurrentMessagingUtils(messagingAccess, Duration.ofMillis(timeoutMillis));
        
        rsc = new RouteSelectionCommandSet();
        CommandSetBuilder builder = new CommandSetBuilder();
        builtInsCommandSet = new CompositeCommandSet();
        builtInsCommandSet.add(builder.buildCommandSet(busMasterCommands));
        // adding the commands for lock and level
        builtInsCommandSet.add(builder.buildCommandSet(new AgentLockServiceCommands(agent.getAgentLockService(),rsc)));
        
        rsc.add(builtInsCommandSet);

        rsc.getCommandDictionary().setVisibilityForTypes(Command.CommandType.values());
        this.agent = agent;
        
    }
    
    void init() {
        agent.getAgentLockService().addAgentLevelListener(this);
        agent.getAgentLockService().addAgentLockUpdateListener(this);
        AgentPresenceManager apMan = messagingAccess.getAgentPresenceManager();
        apMan.addAgentPresenceListener(this);
    }

// -- Getters : ----------------------------------------------------------------
    public RouteSelectionCommandSet getConsoleCommandSet() {
        return rsc;
    }

// -- Implementing AgentPresenceListener : -------------------------------------
    @Override
    public void connecting(AgentInfo agent) {
        if (!(agent.getType().equals(AgentInfo.AgentType.LISTENER) /*|| agent.getType().equals(AgentInfo.AgentType.CONSOLE)*/)) {
            if (!rsc.containsPath(agent.getName())) {
                //This is invoked on a separate Thread to avoid JGroup internal exceptions.
                AddSubsystem add = new AddSubsystem(agent);
                Thread t = new Thread(add);
                t.start();
            }
        }
    }

    @Override
    public synchronized void disconnecting(AgentInfo agent) {
        String agentName = agent.getName();
        rsc.removeRoute(agentName);
    }
    
    @Override
    public void onAgentLevelChange(String agentName, int level) {
        if (rsc.containsPath(agentName)) {
            rsc.setRouteLevel(agentName, level);
        }
    }
    
    @Override
    public void onAgentLockUpdate(String agentName, AgentLock lock) {
        if (rsc.containsPath(agentName)) {
            String consoleName = messagingAccess.getAgentInfo().getName();
            if(lock != null && lock.getOwnerAgentNames().contains(consoleName)) {
                // This console owns the lock
                rsc.setRouteCommandTypeVisibility(agentName, Command.CommandType.values());                    
            } else {
                // The agent is either unlocked or locked by someone else
                rsc.setRouteCommandTypeVisibility(agentName, new Command.CommandType[]{Command.CommandType.QUERY});                    
            }
        }
    }
    
    BusMasterCommands getBusMasterCommands() {
        return busMasterCommands;
    }
    
// -- Local classes : ----------------------------------------------------------  
    /** This must be public for the command invocation to work */
    public class BusMasterCommands {

        public List<String> listSubsystems() {
            return listSubsystems("","Worker", true);
        }

        @Command(name = "listSubsystems", description = "List the subsystems on the CCS buses for a given Agent type", alias = "ls")
        public List<String> listSubsystems(@Argument(name = "Options", defaultValue="") String options, 
                @Argument(name = "Agent type", defaultValue="Worker") String agentType, 
                @Argument(defaultValue="true") boolean allTypesAbove) {

            AgentType referenceType = AgentType.valueOf(agentType.toUpperCase());
            List<String> result = new ArrayList<>();
            for ( AgentInfo agent : messagingAccess.getAgentPresenceManager().listConnectedAgents() ) {
                if ( (allTypesAbove && agent.getType().ordinal() >= referenceType.ordinal() ) ||  agent.getType().ordinal() == referenceType.ordinal() ) {
                    result.add(getAgentInformation(agent, options));
                }
                
            }
            return result;
        }
        
        private String getAgentInformation(AgentInfo agentInfo, String options) {
            StringBuilder sb = new StringBuilder();
            sb.append(agentInfo.getName());
            if ( options.contains("h") ) {
                sb.append("[");
                String host = agentInfo.getAgentProperty("org.lsst.ccs.agent.hostname");
                if ( host != null && ! host.isEmpty() ) {
                    sb.append("host=").append(host);
                    String jmxport = agentInfo.getAgentProperty("org.lsst.ccs.agent.jmxport");
                    if ( jmxport != null && ! jmxport.isEmpty() ) {
                        sb.append(":").append(jmxport);
                    }                    
                }
                sb.append("]");
            }
            return sb.toString();
        }

        @Command(description = "Set Console parameters")
        public void set(@Argument(name = "item") ConsoleParameters what, @Argument(name = "value") String value) {
            if ( what == ConsoleParameters.TIMEOUT ) {
                sci.setDefaultTimeout(Duration.ofMillis(Long.valueOf(value)));
            } else if ( what == ConsoleParameters.SHOW_JYTHON_OUTPUT ) {
                if ( jythonConsoleSocketConnection != null ) {
                    jythonConsoleSocketConnection.setPrintStream(value.equals("true") ? System.out : null);
                }
                showJythonOutput = value;
            }
        }
                
        @Command(description = "Get a Console parameter")
        public String get(@Argument(name = "item") ConsoleParameters what ) {
            if ( what == ConsoleParameters.TIMEOUT ) {
                return String.valueOf(sci.getDefaultTimeout().toMillis());
            } else if ( what == ConsoleParameters.SHOW_JYTHON_OUTPUT ) {
                return showJythonOutput;
            }
            return null;
        }

        @Command(description = "Execute a Jython script")
        public void executeScript(@Argument(name = "scriptPath") String scriptPath, @Argument(name = "scriptArguments") String... args) throws IOException {
            JythonScriptExecutorUtils.executeScript(scriptPath,args);
        }

        
        @Command(description = "Submit a Jython script to a Jython Interpreter")
        public void connectToJythonConsole(
                @Argument(name = "host", defaultValue = "localhost") String host,
                @Argument(name = "port", defaultValue = "4444") int port
        ) throws IOException {
            if ( jythonConsoleSocketConnection != null ) {
                throw new RuntimeException("There is already a connection established. There can be only one! Please \"closeConnectionWithJythonConsole\" before opening a new one.");
            }
            jythonConsoleSocketConnection = new JythonConsoleSocketConnection("CommandShellConnection_"+System.currentTimeMillis(),port,host); 
            jythonConsoleSocketConnection.setPrintStream(showJythonOutput.equals("true") ? System.out : null);
            
        }
        
        
        @Command(description = "Submit a Jython script to a Jython Interpreter")
        public void submitScript(
                @Argument(name = "scriptPath") String scriptPath, @Argument(name = "scriptArgs") String... args
        ) throws IOException {
            if ( jythonConsoleSocketConnection == null ) {
                throw new RuntimeException("No connection was established with a JythonConsole. Please invoke \"connectToJythonConsole\" first");
            }
            jythonConsoleSocketConnection.asynchFileExecution(scriptPath,args);
        }
        
        @Command(description = "Submit a Jython script to a Jython Interpreter")
        public void closeConnectionWithJythonConsole() throws IOException {
            if ( jythonConsoleSocketConnection == null ) {
                throw new RuntimeException("There is open connection to a JythonConsole.");
            }
            jythonConsoleSocketConnection.close();
            jythonConsoleSocketConnection = null;
        }
        
        
    }

    private class AddSubsystem implements Runnable {

        private final AgentInfo agent;
        AddSubsystem(AgentInfo agentInfo) {
            this.agent = agentInfo;
        }

        @Override
        public void run() {
            try {
                CommandRequest getDictionaryCommand = new CommandRequest(agent.getName(), "getDictionaries");
                HashMap<String, Dictionary> dictionaries = (HashMap<String, Dictionary>) sci.sendSynchronousCommand(getDictionaryCommand);
                synchronized (ConsoleCommandShell.this) {
                    for (String path : dictionaries.keySet() ) {
                        String useLongPathsProperty = agent.getAgentProperty("use.full.paths");
                        boolean useLongPaths = useLongPathsProperty != null && useLongPathsProperty.toLowerCase().equals("true");
                        String route = path;
                        if ( ! useLongPaths ) {
                            int first = path.indexOf("/");
                            int last = path.lastIndexOf("/");
                            if (first != last) {
                                route = path.substring(0, first) + path.substring(last);
                            }
                        }
                            rsc.addRoutingCommandSet(route, path, new BusCommandSet(sci, path, dictionaries.get(path)));
                    }
//                    if (agent.getType() == AgentType.WORKER) {
                        String agentName = agent.getName();
                        //Asking the agent lock service for the level for this agent
                        int dictionaryLevel = messagingAccess.getAgentLockService().getLevelForAgent(agentName);
                        AgentLock lock = messagingAccess.getAgentLockService().getLockForAgent(agentName);
                        rsc.setRouteLevel(agentName, dictionaryLevel);
                        rsc.setRouteCommandTypeVisibility(agentName, lock != null ? Command.CommandType.values() : new Command.CommandType[]{Command.CommandType.QUERY});
//                    }
                }
            } catch (Exception ex) {
                throw new RuntimeException(ex);
            }
        }
    }

// -- Running console as subsystem : -------------------------------------------
    /**
     * Creates a command shell for embedding into external GUI.
     *
     * @param agent The agent in which the Shell is to operate.
     * @param reader JLine console reader to be used for input and output.
     * @throws IOException if the shell fails to initialize.
     */
    static public JLineShell createJLineShell(Agent agent, ConsoleReader reader) throws IOException {
        ConsoleCommandShell comShell = new ConsoleCommandShell(agent);
        comShell.init();
        return new JLineShell(comShell.getConsoleCommandSet(), reader, "${target} ccs>");
    }
    
    /** Create a ConsoleCommandShell. */
    static public ConsoleCommandShell createConsoleCommandShell(Agent agent) {
        ConsoleCommandShell comShell = new ConsoleCommandShell(agent);
        comShell.init();
        return comShell;
    }

    /**
     * Runs shell command console in standalone mode as a CCS subsystem named "console".
     */
    static public void main(String[] argv) throws Exception {
        BusMaster busMaster = new BusMaster("ccs-shell");
        busMaster.startAgent();
        ConsoleCommandShell comShell = new ConsoleCommandShell(busMaster);
        JLineShell shell = new JLineShell(comShell.getConsoleCommandSet(), "${target} ccs>");
        comShell.init();
        shell.run();
        System.exit(0);
    }
    
}
