package org.lsst.ccs.subsystem.shell;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import jline.console.ConsoleReader;
import org.lsst.ccs.BusMaster;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.command.CommandSetBuilder;
import org.lsst.ccs.command.CompositeCommandSet;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.RouteSelectionCommandSet;
import org.lsst.ccs.command.RoutingCommandSet;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.AgentPresenceManager;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.scripting.jython.JythonScriptExecutorUtils;
import org.lsst.ccs.shell.JLineShell;
import org.lsst.ccs.subsystems.console.jython.JythonConsoleSocketConnection;

/**
 * JLine-based command shell.
 * Re-factored version of ConsoleBusMaster - allows creation of a command console
 * without creating a subsystem.
 * <p>
 * Static methods of this class can be used to either run the shell standalone
 * or construct <tt>JLineShell</tt> for embedding into the graphical console.
 */
public class ConsoleCommandShell implements AgentPresenceListener {

    public enum ConsoleParameters {
        TIMEOUT,
        SHOW_JYTHON_OUTPUT
    };

// -- Private parts : ----------------------------------------------------------
    private final AgentMessagingLayer messagingAccess;

    private final RouteSelectionCommandSet rsc;
    private final CompositeCommandSet ccs;
    private final ConcurrentMessagingUtils sci;
    private final BusMasterCommands busMasterCommands = new BusMasterCommands();
    private static final int timeoutMillis = 10000;
    private final Map<String, RoutingCommandSet> routes = new LinkedHashMap<>();
    private JythonConsoleSocketConnection jythonConsoleSocketConnection = null;
    private String showJythonOutput = "false";

// -- Construction and initialization : ----------------------------------------
    ConsoleCommandShell(AgentMessagingLayer messagingAccess) {
        this.messagingAccess = messagingAccess;
        sci = new ConcurrentMessagingUtils(messagingAccess, Duration.ofMillis(timeoutMillis));
        CommandSetBuilder builder = new CommandSetBuilder();
        ccs = new CompositeCommandSet();
        ccs.add(builder.buildCommandSet(busMasterCommands));
        rsc = new RouteSelectionCommandSet(ccs);
    }

    void init() {
        AgentPresenceManager apMan = messagingAccess.getAgentPresenceManager();
        apMan.addAgentPresenceListener(this);
    }

// -- Getters : ----------------------------------------------------------------
    public CompositeCommandSet getConsoleCommandSet() {
        return rsc;
    }

// -- Implementing AgentPresenceListener : -------------------------------------
    @Override
    public void connecting(AgentInfo agent) {
        if (!(agent.getType().equals(AgentInfo.AgentType.LISTENER) || agent.getType().equals(AgentInfo.AgentType.CONSOLE))) {
            if (!routes.containsKey(agent.getName())) {
                //This is invoked on a separate Thread to avoid JGroup internal exceptions.
                AddSubsystem add = new AddSubsystem(agent.getName());
                Thread t = new Thread(add);
                t.start();
            }
        }
    }

    @Override
    public void disconnecting(AgentInfo agent) {
        String agentName = agent.getName();
        synchronized (this) {
            Iterator<Map.Entry<String, RoutingCommandSet>> it = routes.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String, RoutingCommandSet> e = it.next();
                String routeName = e.getKey();
                if (routeName.equals(agentName) || routeName.startsWith(agentName + "/")) {
                    ccs.remove(e.getValue());
                    it.remove();
                }
            }
        }
    }
    
    BusMasterCommands getBusMasterCommands() {
        return busMasterCommands;
    }

// -- Local classes : ----------------------------------------------------------  
    /** This must be public for the command invocation to work */
    public class BusMasterCommands {

        public List<String> listSubsystems() {
            return listSubsystems("","Worker", true);
        }

        @Command(name = "listSubsystems", description = "List the subsystems on the CCS buses for a given Agent type", alias = "ls")
        public List<String> listSubsystems(@Argument(name = "Options", defaultValue="") String options, 
                @Argument(name = "Agent type", defaultValue="Worker") String agentType, 
                @Argument(defaultValue="true") boolean allTypesAbove) {

            AgentType referenceType = AgentType.valueOf(agentType.toUpperCase());
            List<String> result = new ArrayList<>();
            for ( AgentInfo agent : messagingAccess.getAgentPresenceManager().listConnectedAgents() ) {
                if ( (allTypesAbove && agent.getType().ordinal() >= referenceType.ordinal() ) ||  agent.getType().ordinal() == referenceType.ordinal() ) {
                    result.add(getAgentInformation(agent, options));
                }
                
            }
            return result;
        }
        
        private String getAgentInformation(AgentInfo agentInfo, String options) {
            StringBuilder sb = new StringBuilder();
            sb.append(agentInfo.getName());
            if ( options.contains("h") ) {
                sb.append("[");
                String host = agentInfo.getAgentProperty("org.lsst.ccs.agent.hostname");
                if ( host != null && ! host.isEmpty() ) {
                    sb.append("host=").append(host);
                    String jmxport = agentInfo.getAgentProperty("org.lsst.ccs.agent.jmxport");
                    if ( jmxport != null && ! jmxport.isEmpty() ) {
                        sb.append(":").append(jmxport);
                    }                    
                }
                sb.append("]");
            }
            return sb.toString();
        }

        @Command(description = "Set Console parameters")
        public void set(@Argument(name = "item") ConsoleParameters what, @Argument(name = "value") String value) {
            if ( what == ConsoleParameters.TIMEOUT ) {
                sci.setDefaultTimeout(Duration.ofMillis(Long.valueOf(value)));
            } else if ( what == ConsoleParameters.SHOW_JYTHON_OUTPUT ) {
                if ( jythonConsoleSocketConnection != null ) {
                    jythonConsoleSocketConnection.setPrintStream(value.equals("true") ? System.out : null);
                }
                showJythonOutput = value;
            }
        }
                
        @Command(description = "Get a Console parameter")
        public String get(@Argument(name = "item") ConsoleParameters what ) {
            if ( what == ConsoleParameters.TIMEOUT ) {
                return String.valueOf(sci.getDefaultTimeout().toMillis());
            } else if ( what == ConsoleParameters.SHOW_JYTHON_OUTPUT ) {
                return showJythonOutput;
            }
            return null;
        }

        @Command(description = "Execute a Jython script")
        public void executeScript(@Argument(name = "scriptPath") String scriptPath, @Argument(name = "scriptArguments") String... args) throws IOException {
            JythonScriptExecutorUtils.executeScript(scriptPath,args);
        }

        
        @Command(description = "Submit a Jython script to a Jython Interpreter")
        public void connectToJythonConsole(
                @Argument(name = "host", defaultValue = "localhost") String host,
                @Argument(name = "port", defaultValue = "4444") int port
        ) throws IOException {
            if ( jythonConsoleSocketConnection != null ) {
                throw new RuntimeException("There is already a connection established. There can be only one! Please \"closeConnectionWithJythonConsole\" before opening a new one.");
            }
            jythonConsoleSocketConnection = new JythonConsoleSocketConnection("CommandShellConnection_"+System.currentTimeMillis(),port,host); 
            jythonConsoleSocketConnection.setPrintStream(showJythonOutput.equals("true") ? System.out : null);
            
        }
        
        
        @Command(description = "Submit a Jython script to a Jython Interpreter")
        public void submitScript(
                @Argument(name = "scriptPath") String scriptPath, @Argument(name = "scriptArgs") String... args
        ) throws IOException {
            if ( jythonConsoleSocketConnection == null ) {
                throw new RuntimeException("No connection was established with a JythonConsole. Please invoke \"connectToJythonConsole\" first");
            }
            jythonConsoleSocketConnection.asynchFileExecution(scriptPath,args);
        }
        
        @Command(description = "Submit a Jython script to a Jython Interpreter")
        public void closeConnectionWithJythonConsole() throws IOException {
            if ( jythonConsoleSocketConnection == null ) {
                throw new RuntimeException("There is open connection to a JythonConsole.");
            }
            jythonConsoleSocketConnection.close();
            jythonConsoleSocketConnection = null;
        }
        
        
    }

    
    synchronized Map<String, RoutingCommandSet> getRoutes() {
        return routes;
    }
    
    private class AddSubsystem implements Runnable {

        private final String agentName;

        AddSubsystem(String name) {
            this.agentName = name;
        }

        @Override
        public void run() {
            try {
                CommandRequest getDictionaryCommand = new CommandRequest(agentName, "getDictionaries");
                HashMap<String, Dictionary> dictionaries = (HashMap<String, Dictionary>) sci.sendSynchronousCommand(getDictionaryCommand);
                synchronized (ConsoleCommandShell.this) {
                    for (String path : dictionaries.keySet() ) {
                        int first = path.indexOf("/");
                        int last = path.lastIndexOf("/");
                        String name = path;
                        if ( first != last ) {
                            name = path.substring(0,first)+path.substring(last);                            
                        }
                        addDictionary(name, path, dictionaries.get(path));
                    }
                }
            } catch (Exception e) {
                //TODO: Think about if throwing an exception here really makes sense
                throw new RuntimeException(e);
            }
        }

        private void addDictionary(String routeName, String name, Dictionary dictionary) {
            if (!routes.containsKey(routeName)) {
                RoutingCommandSet crcs = new RoutingCommandSet(routeName, new BusCommandSet(sci, name, dictionary));
                ccs.add(crcs);
                routes.put(routeName, crcs);
            } else {
                throw new RuntimeException("Error: Route " + routeName + " already exist");
            }
        }

    }

// -- Running console as subsystem : -------------------------------------------
    /**
     * Creates a command shell for embedding into external GUI.
     *
     * @param messagingAccess Busses access layer.
     * @param reader JLine console reader to be used for input and output.
     * @throws IOException if the shell fails to initialize.
     */
    static public JLineShell createJLineShell(AgentMessagingLayer messagingAccess, ConsoleReader reader) throws IOException {
        ConsoleCommandShell comShell = new ConsoleCommandShell(messagingAccess);
        comShell.init();
        return new JLineShell(comShell.getConsoleCommandSet(), reader, "${target} ccs>");
    }

    /**
     * Runs shell command console in standalone mode as a CCS subsystem named "console".
     */
    static public void main(String[] argv) throws Exception {
        BusMaster busMaster = new BusMaster("console");
        busMaster.startAgent();
        ConsoleCommandShell comShell = new ConsoleCommandShell(busMaster.getMessagingAccess());
        JLineShell shell = new JLineShell(comShell.getConsoleCommandSet(), "${target} ccs>");
        comShell.init();
        shell.run();
        System.exit(0);
    }

}
