package org.lsst.ccs.subsystem.monitor;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystem.monitor.data.MonitorChan;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.data.MonitorState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements monitoring.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Monitor {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static long
        DEFAULT_UPDATE_PERIOD = 1000,
        CHECK_PERIOD          = 10000;

    final static String
        CALC_DEVICE_NAME = "Calc";

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    int numChans;
    Map<String, Device> devcMap;
    Map<String, Channel> chanMap;
    Map<String, Alarm> alarmMap;
    Map<String, Line> lineMap;
    CalcDevice calcDevc = new CalcDevice();
    List<Device> devcList = new ArrayList<>();
    List<Alarm> alarmList = new ArrayList<>();
    Channel[] chanData;
    BitSet loLimChange = new BitSet(), hiLimChange = new BitSet(),
           goodChans = new BitSet(), onlineChans = new BitSet();
    Module module;
    AlarmHandler alarmH;
    Logger log;
    Subsystem subsys;


   /**
    ***************************************************************************
    **
    **  Interface for handling alarm events.
    **
    ***************************************************************************
    */
    public interface AlarmHandler {

        void processAlarm(int event, int parm);

    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to update the channel state
    **  in a timely manner.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            /*
            **  Read the sensor values
            */
            readSensors();

            /*
            **  Check the limits and set the channel state
            */
            checkLimits();
        }

    }

                
   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to check hardware devices state.
    **
    ***************************************************************************
    */
    private class CheckDevices extends TimerTask {

        @Override
        public void run()
        {
            for (Device devc : devcList) {
                devc.checkOnline();
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public Monitor(Module mod, AlarmHandler alarmHand, Logger logger)
    {
        devcMap = mod.getEnvironment().getChildren(Device.class);
        chanMap = mod.getEnvironment().getChildren(Channel.class);
        alarmMap = mod.getEnvironment().getChildren(Alarm.class);
        lineMap = mod.getEnvironment().getChildren(Line.class);
        module = mod;
        alarmH = alarmHand;
        log = logger;
        subsys = module.getSubsystem();
    }


   /**
    ***************************************************************************
    **
    **  Initializes the monitoring configuration.
    **
    ***************************************************************************
    */
    public void initConfiguration()
    {
        devcMap.put(CALC_DEVICE_NAME, calcDevc);
        for (Device devc : devcMap.values()) {
            devcList.add(devc);
            devc.configure(this);
        }

        for (Line line : lineMap.values()) {
            line.configure(this);
        }

        for (Alarm alarm : alarmMap.values()) {
            alarmList.add(alarm);
            alarm.configure(this);
        }

        chanData = new Channel[chanMap.size()];
        int id = 0;
        for (Channel ch : chanMap.values()) {
            ch.configure(this, id);
            chanData[id++] = ch;
        }
        calcDevc.configChans();
    }


   /**
    ***************************************************************************
    **
    **  Initializes the monitoring sensors.
    **
    ***************************************************************************
    */
    public void initSensors()
    {
        /*
        **  Perform full initialization of all devices
        */
        for (Device devc : devcList) {
            devc.initialize();
        }

        /*
        **  Initialize the alarms
        */
        for (Alarm alarm : alarmList) {
            alarm.initialize();
        }

        /*
        **  Start the timer task which checks the device state periodically
        */
        (new Timer(true)).schedule(new CheckDevices(), 0L, CHECK_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    public void readSensors()
    {
        for (Device devc : devcList) {
            devc.readSensors();
        }
    }


   /**
    ***************************************************************************
    **
    **  Starts monitoring.
    **
    ***************************************************************************
    */
    public void start()
    {
        start(DEFAULT_UPDATE_PERIOD);
    }
    
    public void start(long period)
    {
        (new Timer(true)).schedule(new UpdateState(), 0L, period);
    }


   /**
    ***************************************************************************
    **
    **  Gets the full monitoring state.
    **
    ***************************************************************************
    */
    public MonitorFullState getFullState()
    {
        MonitorFullState state = new MonitorFullState();
        for (Channel ch : chanData) {
            state.addChannel(new MonitorChan(ch.getName(), ch.description,
                                             ch.units, ch.limitLo,
                                             ch.limitHi, ch.value));
        }
        state.setMonitorState(new MonitorState(goodChans, onlineChans,
                                               loLimChange, hiLimChange));
        return state;
    }


   /**
    ***************************************************************************
    **
    **  Publishes the state of the monitoring module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    public void publishState()
    {
        MonitorState state = new MonitorState(goodChans, onlineChans,
                                              loLimChange, hiLimChange);
        module.publish(MonitorState.KEY, state);
    }    


   /**
    ***************************************************************************
    **
    **  Publishes all the data values of the monitoring module.
    **
    ***************************************************************************
    */
    public void publishData()
    {
        long timeStamp = System.currentTimeMillis();
        List<ValueNotification> values = new ArrayList<>();
        for (Channel ch : chanData) {
            values.add(new ValueNotification(ch.name, ch.value, timeStamp));
        }
        module.publishData(values);
    }


   /**
    ***************************************************************************
    **
    **  Publishes all the limit values of the monitoring module.
    **
    **  This is intended to be called at startup time.
    **
    ***************************************************************************
    */
    public void publishLimits()
    {
        for (Channel ch : chanData) {
            subsys.publishMetaData(ch.getName(), "alarmLow",
                                   String.valueOf(ch.limitLo));
            subsys.publishMetaData(ch.getName(), "alarmHigh",
                                   String.valueOf(ch.limitHi));
        }
    }    


   /**
    ***************************************************************************
    **
    **  Checks whether all current channel values are within limits.
    **
    ***************************************************************************
    */
    public void checkLimits()
    {
        /*
        **  Generate the current channel state
        */
        BitSet chanState = new BitSet(numChans);
        for (Channel ch : chanData) {
            ch.checkLimits(chanState);
        }
        for (Alarm alarm : alarmList) {
            alarm.setState();
        }
                
        /*
        **  If state changed, publish it and log channel data
        */
        chanState.xor(goodChans);
        goodChans.xor(chanState);
        if (!chanState.isEmpty()) {
            publishState();
            publishData();
        }
    }


   /**
    ***************************************************************************
    **
    **  Clears limits-changed indicators.
    **
    ***************************************************************************
    */
    public void clearLimitChanges()
    {
        loLimChange.clear();
        hiLimChange.clear();
    }


   /**
    ***************************************************************************
    **
    **  Gets the device with the given name.
    **
    ***************************************************************************
    */
    public Device getDevice(String name)
    {
        return devcMap.get(name);
    }


   /**
    ***************************************************************************
    **
    **  Gets the line with the given name.
    **
    ***************************************************************************
    */
    public Line getLine(String name)
    {
        return lineMap.get(name);
    }


   /**
    ***************************************************************************
    **
    **  Gets the alarm with the given name.
    **
    ***************************************************************************
    */
    public Alarm getAlarm(String name)
    {
        return alarmMap.get(name);
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel with the given name.
    **
    ***************************************************************************
    */
    public Channel getChannel(String name)
    {
        return chanMap.get(name);
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel with the given ID.
    **
    ***************************************************************************
    */
    public Channel getChannel(int id)
    {
        return chanData[id];
    }


   /**
    ***************************************************************************
    **
    **  Gets the ID of the channel with the given name.
    **
    ***************************************************************************
    */
    public int getChannelId(String name)
    {
        Channel chan = chanMap.get(name);
        return chan == null ? -1 : chan.id;
    }


   /**
    ***************************************************************************
    **
    **  Sets channels on- or off-line.
    **
    ***************************************************************************
    */
    public void setOnline(BitSet mask, boolean online)
    {
        BitSet prevOnline = (BitSet)onlineChans.clone();
        if (online) {
            onlineChans.or(mask);
        }
        else {
            onlineChans.andNot(mask);
        }
        if (!onlineChans.equals(prevOnline)) {
            publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of channels.
    **
    ***************************************************************************
    */
    public int getNumChans()
    {
        return numChans;
    }


   /**
    ***************************************************************************
    **
    **  Gets the associated module.
    **
    ***************************************************************************
    */
    public Module getModule()
    {
        return module;
    }


   /**
    ***************************************************************************
    **
    **  Reports parameter errors.
    **
    ***************************************************************************
    */
    public void reportError(String cName, String pName, Object pValue)
        throws Exception
    {
        log.error("Invalid " + pName + " (" + pValue + ") for " + cName);
        throw new Exception();
    }


   /**
    ***************************************************************************
    **
    **  Reports configuration parameter errors and exits.
    **
    ***************************************************************************
    */
    public void reportConfigError(String cName, String pName, String message)
    {
        log.error("Configuration parameter " + pName + " for " + cName + " "
                    + message);
//        System.exit(0);
        try {
            subsys.shutdown();
        }
        catch (HardwareException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

}
