package org.lsst.ccs.subsystem.monitor;

import java.util.BitSet;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Handles a monitoring channel.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Channel extends ConfigurableComponent {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    public final static int
        TYPE_UNKNOWN = -1,
        TYPE_TEMP    = 0,
        TYPE_VOLTS   = 1,
        TYPE_POWER   = 2,
        TYPE_SWITCH  = 3,
        TYPE_DIFF    = 4;

    final static int
        LIMIT_CHECK_NONE  = 0,   // Don't check limit
        LIMIT_CHECK_FLAG  = 1,   // Flag if limit exceeded
        LIMIT_CHECK_ALARM = 2;   // Alarm if limit exceeded

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private final static Map<String,Integer> checkMap = new HashMap<>();
    static {
        checkMap.put("NONE",  LIMIT_CHECK_NONE);
        checkMap.put("FLAG",  LIMIT_CHECK_FLAG);
        checkMap.put("ALARM", LIMIT_CHECK_ALARM);
    }

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    // Supplied immutable fields
    String     description;   // Channel description
    String     units;         // Units name
    String     devcName;      // The name of the associated device
    int        hwChan;        // Hardware channel
    String     typeS;         // Channel type string (TEMP, VOLTS, POWER...)
    String     subtypeS;      // Channel subtype string (extra config info)
    double     offset;        // Value offset
    double     scale;         // Value scale
    String     checkLoS;      // Low limit check option string
    String     alarmLoS;      // Low limit alarm name (null = none)
    String     checkHiS;      // High limit check option string
    String     alarmHiS;      // High limit alarm name (null = none)

    // Supplied configurable fields
    double     limitLo;       // Low limit value
    double     limitHi;       // High limit value

    // Derived immutable fields
    String     name;          // Channel name
    Device     devc;          // The associated device
    int        type;          // Encoded channel type
    int        subtype;       // Encoded channel subtype
    int        checkLo;       // Low limit check option
    double     resetLo;       // Low limit reset value
    Alarm      alarmLo;       // Low limit alarm (null = none)
    int        checkHi;       // High limit check option
    double     resetHi;       // High limit reset value
    Alarm      alarmHi;       // High limit alarm (null = none)
    int        id;            // External ID
    boolean    valid;         // True if configured correctly
    Monitor    mon;           // Associated monitor object
    Logger     log;           // The logger

    // Volatile fields
    double     value;         // Current value read from sensor


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public Channel(String desc, String unit, String devName, int chan,
                   String type, String subtype, double offs, double scal,
                   String checkLo, double limLo, double deadbandLo,
                   String alarmLo, 
                   String checkHi, double limHi, double deadbandHi,
                   String alarmHi)
    {
        description = desc;
        units       = unit;
        devcName    = devName;
        hwChan      = chan;
        typeS       = type;
        subtypeS    = subtype;
        offset      = offs;
        scale       = scal;
        checkLoS    = checkLo;
        limitLo     = limLo;
        resetLo     = deadbandLo <= 0 ? limitLo : limitLo + deadbandLo;
        alarmLoS    = alarmLo;
        checkHiS    = checkHi;
        limitHi     = limHi;
        resetHi     = deadbandHi <= 0 ? limitHi : limitHi - deadbandHi;
        alarmHiS    = alarmHi;
    }


   /**
    ***************************************************************************
    **
    **  Configures channel description.
    **
    ***************************************************************************
    */
    void configure(Monitor mon, int id)
    {
        name = getName();
        this.mon = mon;
        this.id = id;
        log = mon.log;

        checkLo = LIMIT_CHECK_FLAG;
        checkHi = LIMIT_CHECK_FLAG;
        type    = TYPE_UNKNOWN;
        subtype = -1;

        try {
            if (devcName == null) {
                devcName = Monitor.CALC_DEVICE_NAME;
            }
            devc = mon.getDevice(devcName);
            if (devc == null) {
                mon.reportError(name, "device name", devcName);
            }

            Integer iCheck = checkMap.get(checkLoS.toUpperCase());
            if (iCheck == null) {
                mon.reportError(name, "low check option", checkLoS);
            }
            checkLo = iCheck;
            iCheck = checkMap.get(checkHiS.toUpperCase());
            if (iCheck == null) {
                mon.reportError(name, "high check option", checkHiS);
            }
            checkHi = iCheck;

            if (alarmLoS != null) {
                alarmLo = mon.getAlarm(alarmLoS);
            }
            if (alarmLo == null
                  && (checkLo == LIMIT_CHECK_ALARM || alarmLoS != null)) {
                mon.reportError(name, "low alarm name", alarmLoS);
            }
            if (alarmHiS != null) {
                alarmHi = mon.getAlarm(alarmHiS);
            }
            if (alarmHi == null
                  && (checkHi == LIMIT_CHECK_ALARM || alarmHiS != null)) {
                    mon.reportError(name, "high alarm name", alarmHiS);
            }

            int[] types = devc.checkChannel(name, hwChan, typeS, subtypeS);
            type = types[0];
            subtype = types[1];

            valid = true;
            devc.addChannel(id);
        }
        catch (Exception e) {
        }
    }


   /**
    ***************************************************************************
    **
    **  Completes configuration of calculation channel description.
    **
    ***************************************************************************
    */
    void configCalc()
    {
        String[] names = subtypeS.split(":", -1);
        int id0 = 0, id1 = 0;
        try {
            if (names.length != 2
                  || (id0 = mon.getChannelId(names[0])) < 0
                  || (id1 = mon.getChannelId(names[1])) < 0) {
                mon.reportError(name, "subtype", subtypeS);
            }
            subtype = (id0 << 16) | id1;
        }
        catch (Exception e) {
            valid = false;
            devc.dropChannel(id);
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware sensor.
    **
    ***************************************************************************
    */
    void initSensor()
    {
        if (!valid) return;
        devc.initChannel(hwChan, type, subtype);
    }


   /**
    ***************************************************************************
    **
    **  Reads the hardware sensor.
    **
    ***************************************************************************
    */
    void readSensor()
    {
        if (!valid) return;
        if (devc != mon.calcDevc) {
            value = offset + scale * devc.readChannel(hwChan, type);
        }
        else {
            value = mon.getChannel(subtype >> 16).value
                      - mon.getChannel(subtype & 0xffff).value;
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks whether current value is within limits.
    **
    ***************************************************************************
    */
    void checkLimits(BitSet cState)
    {
        if (!valid) return;
        boolean ok = true;
        if (value < limitLo) {
            if (checkLo != LIMIT_CHECK_NONE) {
                ok = false;
                if (checkLo == LIMIT_CHECK_ALARM) {
                    alarmLo.updateState(Alarm.STATE_ERROR);
                }
            }
        }
        else if (value > limitHi) {
            if (checkHi != LIMIT_CHECK_NONE) {
                ok = false;
                if (checkHi == LIMIT_CHECK_ALARM) {
                    alarmHi.updateState(Alarm.STATE_ERROR);
                }
            }
        }
        else {
            if (checkLo == LIMIT_CHECK_ALARM) {
                alarmLo.updateState(value >= resetLo ? Alarm.STATE_GOOD
                                                     : Alarm.STATE_DEADBAND);
            }
            if (checkHi == LIMIT_CHECK_ALARM) {
                alarmHi.updateState(value <= resetHi ? Alarm.STATE_GOOD
                                                     : Alarm.STATE_DEADBAND);
            }
        }
        if (ok) {
            cState.set(id);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the low limit.
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setLimitLo(double limit)
    {
        if (!valid || limit == limitLo) return;
        limitLo = limit;
        mon.loLimChange.set(id);
        mon.publishState();
        mon.subsys.publishMetaData(name, "alarmLow", String.valueOf(limit));
    }


   /**
    ***************************************************************************
    **
    **  Sets the high limit.
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setLimitHi(double limit)
    {
        if (!valid || limit == limitHi) return;
        limitHi = limit;
        mon.hiLimChange.set(id);
        mon.publishState();
        mon.subsys.publishMetaData(name, "alarmHigh", String.valueOf(limit));
    }


   /**
    ***************************************************************************
    **
    **  Gets the current value.
    **
    ***************************************************************************
    */
    public double getValue()
    {
        return value;
    }

}
