package org.lsst.ccs.subsystem.monitor.ui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import org.lsst.ccs.bus.MetadataStatus;
import org.lsst.ccs.bus.ModuleInvokerCommand;
import org.lsst.ccs.bus.TrendingStatus;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.subsystem.monitor.data.MonitorChan;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.data.MonitorState;

/*
 * MonitorTrendingTable.java
 *
 * @author turri
 */
public class MonitorTrendingTable extends javax.swing.JPanel {

    final static String[] colNames
      = {"Description", "Value", "Units", "Low Limit", "High Limit", "Name"};
    final static Class[] colTypes = {String.class, TrendingValue.class,
                                     String.class, Double.class, Double.class,
                                     String.class};
    final static boolean[] colCanEdit = {false, false, false, true, true, false};
    final static Color colGood = new Color(160, 255, 160),
                       colError = new Color(255, 160, 160),
                       colOffln = new Color(160, 160, 255);
    final static Font myFont = new Font("Helvetica", Font.PLAIN, 12);
    final static Font changeFont = new Font("Helvetica", Font.BOLD, 12);

    final static int DESCRIPTION_IND = 0;
    final static int VALUE_IND = 1;
    final static int UNITS_IND = 2;
    final static int LOW_LIMIT_IND = 3;
    final static int HIGH_LIMIT_IND = 4;
    final static int NAME_IND = 5;
    CommandSender sender;

    /** Creates new form MonitorTrendingTable */
    public MonitorTrendingTable(CommandSender sender) {
        initComponents();
        JTableHeader hdr = jTable1.getTableHeader();
        hdr.setReorderingAllowed(false);
        hdr.setSize(hdr.getWidth(), hdr.getHeight() + 2);
        jTable1.setDefaultRenderer(TrendingValue.class,
                                   new TrendingTableCellRenderer());
        jTable1.setDefaultRenderer(Double.class, new LimitsCellRenderer());
        jTable1.setDefaultRenderer(String.class, new TextCellRenderer());
        jTable1.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
        jTable1.setRowSelectionAllowed(false);
        jTable1.setColumnSelectionAllowed(false);
        jTable1.setFont(myFont);
        jTable1.setRowHeight(jTable1.getRowHeight() + 2);

        this.sender = sender;
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jScrollPane1 = new javax.swing.JScrollPane();
        jTable1 = new javax.swing.JTable();

        setLayout(new javax.swing.BoxLayout(this, javax.swing.BoxLayout.LINE_AXIS));

        jTable1.setModel(new TrendingTableModel());
        jScrollPane1.setViewportView(jTable1);

        add(jScrollPane1);
    }// </editor-fold>//GEN-END:initComponents
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JTable jTable1;
    // End of variables declaration//GEN-END:variables

    public void updateTableModel(MonitorFullState status)
    {
        SwingUtilities.invokeLater(new UpdateTrendingTableModel(status));
    }

    public void updateTableValue(MonitorState status)
    {
        SwingUtilities.invokeLater(new UpdateTrendingTableState(status));
    }

    public void updateTableValue(TrendingStatus status)
    {
        SwingUtilities.invokeLater(new UpdateTrendingTable(status));
    }

    public void updateTableValue(MetadataStatus status)
    {
        SwingUtilities.invokeLater(new UpdateTrendingTableLimits(status));
    }

    private static String fmt(double val) {
        return String.format("%.2f  ", val);
    }

    class TrendingTableModel extends DefaultTableModel
                             implements TableModelListener {

        List<Double> lowLimits = new ArrayList<>(),
                     highLimits = new ArrayList<>();
        Map<String, Integer> chanMap = new HashMap<>();
        BitSet goodChans = new BitSet(), onlineChans = new BitSet(),
               lowLimitChange = new BitSet(), highLimitChange = new BitSet();

        public TrendingTableModel() {
            super(colNames, 0);
            addTableModelListener(this);
        }

        @Override
        public Class getColumnClass(int columnIndex) {
            return colTypes[columnIndex];
        }

        @Override
        public boolean isCellEditable(int rowIndex, int columnIndex) {
            return colCanEdit[columnIndex];
        }

        @Override
        public void tableChanged(TableModelEvent e)
        {
            int column = e.getColumn();
            if (column != LOW_LIMIT_IND && column != HIGH_LIMIT_IND) return;
            int row = e.getFirstRow();
            boolean isLow = column == LOW_LIMIT_IND;

            double newValue = (Double)getValueAt(row, column);
            double oldValue = isLow ? getLowLimit(row) : getHighLimit(row);

            if (newValue != oldValue) {
                setValueAt(oldValue, row, column);    // Put it back for now
                submitLimit((String)getValueAt(row, NAME_IND), newValue, isLow);
            }
        }

        void addTrendingRow(MonitorChan chan) {
            chanMap.put(chan.getName(), lowLimits.size());
            lowLimits.add(chan.getLowLimit());
            highLimits.add(chan.getHighLimit());
            addRow(new Object[]{chan.getDescription(), chan.getValue(),
                                chan.getUnits(), chan.getLowLimit(),
                                chan.getHighLimit(), chan.getName()});
        }

        void updateValues(TrendingStatus s) {
            List<ValueNotification> vnl = (List<ValueNotification>)s.getValue();
            for (int row = 0; row < getRowCount(); row++) {
                setValueAt(vnl.get(row).getData(), row, VALUE_IND);
            }
        }

        void updateState(MonitorState s) {
            goodChans = s.getGoodChans();
            onlineChans = s.getOnlineChans();
            int nRows = getRowCount();
            BitSet update = (BitSet)s.getLowLimitChange().clone();
            update.xor(lowLimitChange);
            lowLimitChange.xor(update);
            for (int row = update.nextSetBit(0); row >= 0;
                 row = update.nextSetBit(row + 1)) {
                setValueAt(getLowLimit(row), row, LOW_LIMIT_IND);
            }
            update = (BitSet)s.getHighLimitChange().clone();
            update.xor(highLimitChange);
            highLimitChange.xor(update);
            for (int row = update.nextSetBit(0); row >= 0;
                 row = update.nextSetBit(row + 1)) {
                setValueAt(getHighLimit(row), row, HIGH_LIMIT_IND);
            }
        }

        void updateLimit(MetadataStatus s) {
            boolean high = s.getMetadataName().equals("alarmHigh");
            double limit = Double.valueOf(s.getMetadataValue());
            int row = chanMap.get(s.getDataName());
            if (high) {
                setHighLimit(row, limit);
            }
            else {
                setLowLimit(row, limit);
            }
        }

        boolean isValueWithinLimits(int row) {
            return goodChans.get(row);
        }

        int getChannelState(int row) {
            return !onlineChans.get(row) ? -1 : !goodChans.get(row) ? 0 : 1;
        }

        void setLowLimit(int row, double value) {
            lowLimits.set(row, value);
            setValueAt(value, row, LOW_LIMIT_IND);
        }

        void setHighLimit(int row, double value) {
            highLimits.set(row, value);
            setValueAt(value, row, HIGH_LIMIT_IND);
        }

        double getLowLimit(int row) {
            return lowLimits.get(row);
        }

        double getHighLimit(int row) {
            return highLimits.get(row);
        }

        boolean hasLowLimitChanged(int row) {
            return lowLimitChange.get(row);
        }
        
        boolean hasHighLimitChanged(int row) {
            return highLimitChange.get(row);
        }

        void submitLimit(String name, double value, boolean isLowLimit) {
            String param = isLowLimit ? "limitLo" : "limitHi";
            sender.sendCommand(name, new ModuleInvokerCommand("change", param,
                                                              value));
        }
    }

    class TrendingValue {
    }

    class UpdateTrendingTable implements Runnable {

        TrendingStatus s;

        UpdateTrendingTable(TrendingStatus s) {
            this.s = s;
        }

        @Override
        public void run() {
            ((TrendingTableModel)jTable1.getModel()).updateValues(s);
        }
    }

    class UpdateTrendingTableState implements Runnable {

        MonitorState s;

        UpdateTrendingTableState(MonitorState s) {
            this.s = s;
        }

        @Override
        public void run() {
            ((TrendingTableModel)jTable1.getModel()).updateState(s);
        }

    }

    class UpdateTrendingTableLimits implements Runnable {

        MetadataStatus s;

        UpdateTrendingTableLimits(MetadataStatus s) {
            this.s = s;
        }

        @Override
        public void run() {
            ((TrendingTableModel)jTable1.getModel()).updateLimit(s);
        }

    }

    class UpdateTrendingTableModel implements Runnable {

        MonitorFullState s;

        UpdateTrendingTableModel(MonitorFullState s) {
            this.s = s;
        }

        @Override
        public void run() {

            List<MonitorChan> channels = s.getChannels();
            TrendingTableModel model = new TrendingTableModel();
            jTable1.setModel(model);

            model.updateState(s.getMonitorState());

            for (MonitorChan chan : channels) {
                model.addTrendingRow(chan);
            }

            for (int c = 0; c < jTable1.getColumnCount(); c++) {
                TableColumnModel colModel = jTable1.getColumnModel();
                TableColumn col = colModel.getColumn(c);
                TableCellRenderer rndr;
                Component comp;

                rndr = jTable1.getTableHeader().getDefaultRenderer();
                comp = rndr.getTableCellRendererComponent(jTable1,
                                                          col.getHeaderValue(),
                                                          false, false, 0, 0);
                int width = comp.getPreferredSize().width;

                rndr = jTable1.getCellRenderer(0, c);
                Class colClass = jTable1.getColumnClass(c);
                if (!colClass.equals(String.class)) {
                    comp = rndr.getTableCellRendererComponent(jTable1, -999.99,
                                                              false, false, 0, c);
                    width = Math.max(width, comp.getPreferredSize().width);
                }
                else {
                    for (int r = 0; r < jTable1.getRowCount(); r++) {
                        Object value = jTable1.getValueAt(r, c);
                        comp = rndr.getTableCellRendererComponent(jTable1, value,
                                                                  false, false,
                                                                  r, c);
                        width = Math.max(width, comp.getPreferredSize().width);
                    }
                }
                col.setPreferredWidth(width + 4);
            }
        }

    }

    class LimitsCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column)
        {
            Component c = super.getTableCellRendererComponent(table,
                                                              fmt((Double)value),
                                                              isSelected,
                                                              hasFocus,
                                                              row, column);
            TrendingTableModel model = (TrendingTableModel) table.getModel();
            if (column == LOW_LIMIT_IND && model.hasLowLimitChanged(row)
                  || column == HIGH_LIMIT_IND && model.hasHighLimitChanged(row)) {
                c.setFont(changeFont);
                c.setForeground(Color.blue);
            }
            else {
                c.setForeground(Color.black);
            }
            ((JLabel)c).setHorizontalAlignment(SwingConstants.RIGHT);

            return c;
        }

    }

    class TrendingTableCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column)
        {
            Component c = super.getTableCellRendererComponent(table,
                                                              fmt((Double)value),
                                                              isSelected,
                                                              hasFocus,
                                                              row, column);
            TrendingTableModel model = (TrendingTableModel)table.getModel();
            if (column == VALUE_IND) {
                int state = model.getChannelState(row);
                c.setBackground(state > 0 ? colGood : state == 0 ? colError
                                                                 : colOffln);
            }
            else {
                c.setBackground(Color.white);
            }
            ((JLabel)c).setHorizontalAlignment(SwingConstants.RIGHT);

            return c;
        }

    }

    class TextCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column)
        {
            return super.getTableCellRendererComponent(table,
                                                       "  " + (String)value,
                                                       isSelected, hasFocus,
                                                       row, column);
        }

    }

}
