package org.lsst.ccs.subsystem.monitor;

import org.lsst.ccs.framework.ConfigurableComponent;

/**
 ***************************************************************************
 **
 **  Implements alarms.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Alarm extends ConfigurableComponent {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    public final static int
        EVENT_TRIP  = 0,
        EVENT_LIMBO = 1,
        EVENT_RESET = 2;

    final static int
        STATE_GOOD     = 0,
        STATE_DEADBAND = 1,
        STATE_ERROR    = 2;

   /**
    ***********************************************************************
    **
    **  Data fields.
    **
    ***********************************************************************
    */
    // Supplied immutable fields
    String    lineName;    // Associated line name
    int       eventParm;   // Associated event parameter

    // Derived immutable fields
    Line      line;        // Associated line
    Monitor   mon;         // Associated monitor

    // Volatile fields
    int       state;       // Current state
    int       newState;    // New state


   /**
    ***********************************************************************
    **
    **  Constructor.
    **
    ***********************************************************************
    */
    public Alarm(String lineName, int eventParm)
    {
        this.lineName  = lineName;
        this.eventParm = eventParm;
    }


   /**
    ***********************************************************************
    **
    **  Configures the alarm.
    **
    ***********************************************************************
    */
    void configure(Monitor mon)
    {
        this.mon = mon;
        try {
            if (lineName != null) {
                line = mon.getLine(lineName);
                if (line == null) {
                    mon.reportError(getName(), "line", lineName);
                }
            }
        }
        catch(Exception e) {
        }
    }


   /**
    ***********************************************************************
    **
    **  Initializes the alarm.
    **
    ***********************************************************************
    */
    void initialize()
    {
        state = STATE_GOOD;
        if (line != null) {
            line.set(false);
        }
    }


   /**
    ***********************************************************************
    **
    **  Clears the (new) state.
    **
    ***********************************************************************
    */
    void clearState()
    {
        newState = STATE_GOOD;
    }


   /**
    ***********************************************************************
    **
    **  Updates the (new) state.
    **
    ***********************************************************************
    */
    void updateState(int value)
    {
        if (value > newState) {
            newState = value;
        }
    }


   /**
    ***********************************************************************
    **
    **  Sets the state (from the new state).
    **
    ***********************************************************************
    */
    void setState()
    {
        if (newState != state
            && (newState != STATE_DEADBAND || state != STATE_GOOD)) {

            Boolean set = null;
            Integer event = null;
            if (newState == STATE_ERROR) {
                set = true;
                event = EVENT_TRIP;
            }
            else if (state == STATE_ERROR) {
                set = false;
                event = newState == STATE_GOOD ? EVENT_RESET : EVENT_LIMBO;
            }
            else if (newState == STATE_GOOD) {
                event = EVENT_RESET;
            }
            if (line != null && set != null) {
                line.set(set);
            }
            if (eventParm >= 0 && event != null) {
                mon.alarmH.processAlarm(event, eventParm);
            }
            state = newState;
        }
        newState = STATE_GOOD;
    }

}