package org.lsst.ccs.subsystem.common;

import java.security.InvalidParameterException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorUpdateTask;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.data.MonitorTask;

/**
 *  Controls special monitoring tasks.
 * 
 *  @author saxton
 */
public class MonitorTaskControl implements HasLifecycle {
    
    /**
     *  Constants.
     */
    private static final String
        MON_UPDATE_TASK = "monitor-update",
        MON_CHECK_TASK = "monitor-check",
        MON_PUBLISH_TASK = "monitor-publish";

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService configurationService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Monitor monitor;

    private final Map<String, MonitorTask> specialTaskMap = new TreeMap<>();
    private final Map<String, MonitorTask> deviceTaskMap = new HashMap<>();
    private String publishTaskPath, updateTaskPath;
    private boolean periodChanged = false, fastActive = false;
    private final Set<MonitorUpdateTask> updateTasks = new HashSet<>();
    private long endTime = 0;
    
    private String tasksRoot;
   
    /**
     *  Create a new object and add it to the tree. 
     * 
     *  @param  subsys  The subsystem object
     *  @param  name  The name of this node
     *  @return  The created object
     */
    public static MonitorTaskControl createNode(Subsystem subsys, String name)
    {
        ComponentLookup lookupService = subsys.getComponentLookup();
        ComponentNode topNode = lookupService.getComponentNodeForObject(subsys);
        MonitorTaskControl control = new MonitorTaskControl();
        ComponentNode controlNode = new ComponentNode(name, control);
        lookupService.addComponentNodeToLookup(topNode, controlNode);
        return control;
    }


    /**
     *  Post-init phase.
     */
    @Override
    public void postInit()
    {
        List<String> allTasks = periodicTaskService.getAgentPeriodicTaskNames();

        // Get the names of the separate device check tasks, if any
        Set<String> checkTasks = new HashSet<>();
        for (String taskPath : allTasks) {
            if (taskPath.startsWith(MON_CHECK_TASK + "/")) {
                checkTasks.add(taskPath.substring(taskPath.indexOf('/') + 1));
            }
        }
        
        // Get the names of the special monitor update tasks
        int index = 0;
        for (String taskPath : allTasks) {
            if (taskPath.startsWith(MON_UPDATE_TASK + "/")) {
                String taskName = taskPath.substring(taskPath.indexOf('/') + 1);
                if (!checkTasks.contains(taskName)) {
                    specialTaskMap.put(taskName, new MonitorTask(taskName, index++, getTaskUpdatePeriod(taskName)));
                }
                else {
                    deviceTaskMap.put(taskName, new MonitorTask(taskName, index++, getTaskUpdatePeriod(taskName)));
                }
            }
        }

        // Set the publish & update task path names for retrieving the overall period
        publishTaskPath = MON_PUBLISH_TASK + (deviceTaskMap.isEmpty() ? "" : "/" + deviceTaskMap.keySet().iterator().next());
        updateTaskPath = MON_UPDATE_TASK + (deviceTaskMap.isEmpty() ? "" : "/" + deviceTaskMap.keySet().iterator().next());
        
        //The root of the task nodes
        tasksRoot = periodicTaskService.getAgentServiceName();

        // Get all the monitor update tasks
        for (String devcName : monitor.getDeviceNames()) {
            for (MonitorUpdateTask task : monitor.getMonitorUpdateTasksForDevice(monitor.getDevice(devcName))) {
                updateTasks.add(task);
            }
        } 
    }


    /**
     *  Command to set the period for which fast monitor publishing occurs.
     *
     *  @param  period  The period (ms) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the monitor fast publish period")
    public synchronized void setFastPeriod(@Argument(description="The fast publish period (ms)") int period)
    {
        long fastPeriod = Math.max(0, period);
        endTime = System.currentTimeMillis() + fastPeriod;
        for (MonitorUpdateTask task : updateTasks) {
            task.forceDataPublicationForDuration(Duration.ofMillis(fastPeriod));
        }
        fastActive = fastPeriod > 0;
    }


    /**
     *  Command to set the monitor publishing period.
     *
     *  @param  period  The period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the monitor publish period")
    public synchronized void setPublishPeriod(@Argument(description="The publish period (ms)") int period)
    {
        if (deviceTaskMap.isEmpty()) {
            configurationService.submitChange(tasksRoot+"/"+MON_PUBLISH_TASK, "taskPeriodMillis", Duration.ofMillis(period).toMillis());
        }
        else {
            for (MonitorTask task : deviceTaskMap.values()) {
                configurationService.submitChange(tasksRoot+"/"+MON_PUBLISH_TASK+"/"+task.getName(), "taskPeriodMillis", Duration.ofMillis(period).toMillis());
            }
        }
        for (MonitorTask task : specialTaskMap.values()) {
            synchronized (task) {
                if (!task.isActive()) {
                    configurationService.submitChange(tasksRoot+"/"+MON_PUBLISH_TASK+"/"+task.getName(), "taskPeriodMillis", Duration.ofMillis(period).toMillis());
                }
            }
        }
        configurationService.commitBulkChange();
        periodChanged = true;
    }


    /**
     *  Command to get the list of known special monitor task pair names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the special monitor task names")
    public List<String> getTaskNames()
    {
        List<String> monTasks = new ArrayList<>();
        monTasks.addAll(specialTaskMap.keySet());
        return monTasks;
    }


    /**
     *  Command to set the period for a special monitor task pair.
     * 
     *  @param  taskName  The name of the task
     *  @param  period    The period (ms)
     */
    @Command(type=Command.CommandType.ACTION, description="Set the period for a special monitor task")
    public void setTaskPeriod(@Argument(description="The task name") String taskName,
                              @Argument(description="The task period (ms)") int period)
    {
        MonitorTask task = getTask(taskName);
        synchronized (task) {
            task.setPeriod(period);
            if (task.isActive()) {
                setTaskUpdatePeriod(taskName, period);
                setTaskPublishPeriod(taskName, period);
            }
        }
    }


    /**
     *  Command to set the active state for a special monitor task pair.
     * 
     *  @param  taskName  The name of the task
     *  @param  active    Whether to activate or not
     */
    @Command(type=Command.CommandType.ACTION, description="Set the active state for a special monitor task")
    public void setTaskActive(@Argument(description="The task name") String taskName,
                              @Argument(description="Whether to make active") boolean active)
    {
        MonitorTask task = getTask(taskName);
        synchronized (task) {
            task.setActive(active);
            if (active) {
                int period = task.getPeriod();
                setTaskUpdatePeriod(taskName, period);
                setTaskPublishPeriod(taskName, period);
            }
            else {
                setTaskUpdatePeriod(taskName, getUpdatePeriod());
                setTaskPublishPeriod(taskName, getPublishPeriod());
            }
        }
    }


    /**
     *  Gets whether update period has changed since last query.
     * 
     *  @return  Whether changed
     */
    public synchronized boolean hasPeriodChanged()
    {
        boolean changed = periodChanged || fastActive;
        periodChanged = false;
        fastActive = getFastPeriod() > 0;
        return changed;
    }


    /**
     *  Gets the map of monitoring task pairs.
     * 
     *  @return  The map
     */
    public Map<String, MonitorTask> getMonitorTaskMap()
    {
        return specialTaskMap;
    }


    /**
     *  Gets the overall monitoring publishing period.
     *
     *  @return  The publishing period (ms)
     */
    public int getPublishPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod(publishTaskPath).toMillis();
    }


    /**
     *  Gets the overall monitoring update period.
     *
     *  @return  The update period (ms)
     */
    private int getUpdatePeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod(updateTaskPath).toMillis();
    }


    /**
     *  Gets the update period for a monitoring task pair.
     *
     *  @param  taskName  The task name
     *  @return  The update period (ms)
     */
    private int getTaskUpdatePeriod(String taskName)
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod(MON_UPDATE_TASK + "/" + taskName).toMillis();
    }
    

    /**
     *  Sets the update period for a monitoring task pair.
     *
     *  @param  taskName  The task name
     *  @param  period    The period (ms) to set.
     */
    private void setTaskUpdatePeriod(String taskName, int period)
    {
        periodicTaskService.setPeriodicTaskPeriod(MON_UPDATE_TASK + "/" + taskName, Duration.ofMillis(period));
    }
    

    /**
     *  Sets the publishing period for a monitoring task pair.
     *
     *  @param  taskName  The task name
     *  @param  period    The period (ms) to set.
     */
    private void setTaskPublishPeriod(String taskName, int period)
    {
        periodicTaskService.setPeriodicTaskPeriod(MON_PUBLISH_TASK + "/" + taskName, Duration.ofMillis(period));
    }
    

    /**
     *  Gets a named monitor task pair.
     * 
     *  @param  taskName    The task name
     */
    private MonitorTask getTask(String taskName)
    {
        MonitorTask task = specialTaskMap.get(taskName);
        if (task == null) {
            throw new InvalidParameterException("Unknown monitor task name: " + taskName);
        }
        return task;
    }


    /**
     *  Gets the remaining fast publish period.
     * 
     *  @return  The remaining period (ms)
     */
    public int getFastPeriod()
    {
        long remTime = 0;
        if (endTime > 0) {
            remTime = endTime - System.currentTimeMillis();
            if (remTime <= 0) {
                endTime = 0;
            }
        }
        return endTime == 0 ? 0 : (int)remTime;
    }

}
