package org.lsst.ccs.subsystem.common.devices.dataforth;

import java.util.Map;
import java.util.TreeMap;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.monitor.Channel;

/**
 *  Handles the simulated MAQ20 device.
 *
 *  @author Owen Saxton
 */
public class SimMaq20Device extends Maq20Device {

    private final Map<String, Channel> channelMap = new TreeMap<>();


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
        super.initDevice();
        Maq20.ModuleType[] modTypes = new Maq20.ModuleType[getModuleCount()];
        for (int j = 0; j < modTypes.length; j++) {
            modTypes[j] = getModuleData(j).modDef.type;
        }
        maq.initSimulation(modTypes, serials);
        fullName = path + " (Simulated Maq20 system)";
    }


    /**
     *  Sets initial channel values.
     */
    @Override
    public void postStart()
    {
        for (String chanPath : channelMap.keySet()) {
            try {
                setChannelValue(chanPath, 20.0 + Math.random());
            }
            catch (DriverException e) {}  // Just ignore it
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  ch  The Channel to check
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if there is a parameter error
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        int[] result = super.checkChannel(ch);
        channelMap.put(ch.getPath(), ch);
        return result;
    }


    @Command(type=Command.CommandType.ACTION, description="Set a channel value")
    public void setChannelValue(@Argument(description="Channel name") String chanName,
                                @Argument(description="Channel value") double value) throws DriverException
    {
        Channel chan = channelMap.get(chanName);
        if (chan == null) {
            throw new DriverException("Invalid channel name: " + chanName);
        }
        double offset = chan.convertRawValue(0.0);
        double scale = chan.convertRawValue(1.0) - offset;
        maq.setSimChannelValue(getModuleData(chan.getType() & 0xff).modId, chan.getHwChan(), (value - offset) / scale);
    }


    @Command(type=Command.CommandType.QUERY, description="Get the list of channel names")
    public String getChannelNames()
    {
        return channelMap.keySet().toString();
    }

}
