package org.lsst.ccs.subsystem.common.devices.dataforth;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.drivers.dataforth.Maq20Discrete;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Dataforth Maq20 system discrete output module.
 *
 *  @author Owen Saxton
 */
public class Maq20DiscControl implements HasLifecycle {

    /**
     *  Data fields.
     */
    @LookupName
    protected String name;
    @LookupPath
    protected String path;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    protected Maq20Device maqDevc;

    protected Integer modIndex;  // Module index

    private static final Logger LOG = Logger.getLogger(Maq20DiscControl.class.getName());
    private Maq20Device.ModuleData modData;


    /**
     *  Life-cycle initialization.
     * 
     *  Checks the configuration
     */
    @Override
    public void init()
    {
        if (modIndex == null) {
            ErrorUtils.reportConfigError(LOG, path, "modIndex", "not specified");
        }
        modData = maqDevc.getModuleData(modIndex);
        if (modData == null) {
            ErrorUtils.reportConfigError(LOG, path, "modIndex (" + modIndex + ")", "specifies non-existent module");
        }
        if (modData.modDef.opType != Maq20.OPER_DISCRETE && modData.modDef.opType != Maq20.OPER_DISCOUT) {
            ErrorUtils.reportConfigError(LOG, path, "modIndex (" + modIndex + ")", "specifies non-discrete output module");
        }
    }


    /**
     *  Sets the state of an output line.
     *
     *  @param  chan  The channel number
     *  @param  on    Whether to set on
     *  @throws  DriverException
     */
    public void setLineOn(int chan, boolean on) throws DriverException
    {
        getDisc().writeDisc(chan, on ? 1 : 0);
    }


    /**
     *  Gets the state of an output line.
     *
     *  @param  chan  The channel number
     *  @return  Whether line is on, or null if device offline
     */
    public Boolean isLineOn(int chan)
    {
        Boolean on = null;
        if (maqDevc.isOnline()) {
            try {
                on = getDisc().readDiscOut(chan) != 0;
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0} channel {1} state: {2}", new Object[]{path, chan, e});
                maqDevc.setOnline(false);
            }
        }
        return on;
    }

    /**
     *  Gets the state of an input line.
     *
     *  @param  chan  The channel number
     *  @return  Whether line is on, or null if device offline
     */
    public Boolean isInputOn(int chan)
    {
        Boolean on = null;
        if (maqDevc.isOnline()) {
            try {
                on = getDisc().readDiscIn(chan) != 0;
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0} channel {1} state: {2}", new Object[]{path, chan, e});
                maqDevc.setOnline(false);
            }
        }
        return on;
    }


    /**
     *  Gets the name.
     * 
     *  @return  The name
     */
    public String getName()
    {
        return name;
    }


    /**
     *  Gets the path.
     * 
     *  @return  The path
     */
    public String getPath()
    {
        return path;
    }


    /**
     *  Gets discrete I/O object.
     *
     *  @return  Discrete I/O object
     *  @throws  DriverException  if connection not open
     */
    private Maq20Discrete getDisc() throws DriverException
    {
        if (modData.maqDisc == null) {
            throw new DriverException("Connection not open");
        }
        return modData.maqDisc;
    }

    /**
     *  Gets the module index.
     *
     *  @return  The module index
     */
    public int getModIndex()
    {
        return modIndex == null ? -1 : modIndex;
    }


}
