package org.lsst.ccs.subsystem.common;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.DerivedChannel;

/**
 *  A derived Channel averaging a configuragle list of other Channels
 *  with configurable weights
 *
 *  @author eisner
 */
public class ConfiguredAverageChannel extends DerivedChannel {

    @LookupField(strategy = LookupField.Strategy.TREE)
    Map<String, Channel> channelMap = new HashMap<>();

    @LookupName
    private String name;

    private static final Logger LOG = Logger.getLogger(ConfiguredAverageChannel.class.getName());

    private final List<Channel> channels = new CopyOnWriteArrayList<>();
    private final List<Double> channelsW = new CopyOnWriteArrayList<>();
    private volatile String channelSelection;

    //Temporary Lists of channels and weights.
    //They are filled during the validation step and reused when configurations
    //are applied to save time
    private final List<Channel> tmpChannels = new CopyOnWriteArrayList<>();
    private final List<Double> tmpChannelsW = new CopyOnWriteArrayList<>();
    private volatile String tmpChannelSelection;

    private volatile double sumWeights;

    @ConfigurationParameter(name = "chanNames", category = "General", 
                            description = "Path regular expression to select the Channels to be averaged",
                            maxLength = 25, units = "unitless")
    protected volatile List<String> chanNames = new ArrayList<>();

    @ConfigurationParameter(name = "chanWeights", category = "General", 
                            description = "Weights of Channels to be averaged",
                            maxLength = 25, units = "unitless")
    protected volatile List<Double> chanWeights = new ArrayList<>();

    @ConfigurationParameter(name = "ignoreNaN", category = "General", 
                            isFinal = true, units = "unitless",
                            description = "if true ignore (omit) NaN channels")
    protected volatile boolean ignoreNaN;

   /**
    *  Checks validity of a proposed set of ConfigurationParameters
    *
    *  The Channel name and weight arrays must have same length,.
    *  Channels must be found on tree.
    *  Weights must be positive-definite.
    *
    *  @param Map of ConfigurationParameters from name to value
    */
    @Override
    public void validateBulkChange(Map<String,Object> params) {
        super.validateBulkChange(params);
        List<String> channelList = (List<String>) params.get("chanNames");
        List<Double> weightList = (List<Double>) params.get("chanWeights");
        int len = channelList.size();
        if (weightList.size() != len) {
            throw new RuntimeException(name + ": chanWeights and chanNames do not have same size");
        }

        StringBuilder sb = new StringBuilder();
        sb.append("Updated Channel/Weight lists for ConfiguredAverageChannel ").append(name).append("\n");

        //Clear the internal temporary Lists of channels and weights.
        tmpChannels.clear();
        tmpChannelsW.clear();

        for (int i = 0; i < len; i++) {
            try { 
                
                //Check that the weight is positive-definite
                Double chanWeight = weightList.get(i);
                if (!( chanWeight>= 0.)) {
                    throw new RuntimeException(name + "/chanWeights element " + i
                            + " is not positive-definite");
                }
                
                //Check that the provided channelNames element is a valid
                //regular expression and that it matches at least one Channel path.
                String chanName = channelList.get(i);
                Pattern p = Pattern.compile(chanName);
                sb.append("Regular expression ").append(chanName).append(" selected Channels with weight ").append(chanWeight).append(":\n");
                int channelCount = 0;
                boolean noMatch = true;
                for (Entry<String,Channel> e : channelMap.entrySet()) {
                    if ( p.matcher(e.getKey()).matches() ) {
                        noMatch = false;
                        Channel ch = e.getValue();
                        if (tmpChannels.contains(ch)) {
                            //This channel has already been selected by another regular expression
                            //This is a configuration error
                            throw new RuntimeException("Channel " + e.getKey() + " has already been selected by another entry in \"channelNames\" " + chanName);
                        } else {
                            tmpChannels.add(ch);
                            tmpChannelsW.add(chanWeight);
                            sb.append("-").append(channelCount++).append("- ").append(e.getKey()).append("\n");
                        }
                    }                    
                }
                if ( noMatch ) {
                    throw new RuntimeException(name + "/chanNames element " + 
                            channelList.get(i) + " didn't match any available channels.");                
                }
            } catch (PatternSyntaxException e) {
                throw new RuntimeException(name + "/chanNames element " + 
					   channelList.get(i) + " is not a valid regular expression.");                
            }
        }
        tmpChannelSelection = sb.toString();
    }    
        
   /**
    *  Set configuration parameters to new values.
    *  Checks input Map for those ConfigurationParameters which correspond
    *  to hardware settings; the others are left for framework to deal with.
    *
    *  @param  Map of parameters for which a change is requested,
    *          keyed by names
    */
    @Override
    public void setParameterBulk(Map<String,Object> params)  {
        super.setParameterBulk(params);
        
        //If either the channelNames or the channelWeights have been modified
        //we need to re-evaluate the internal lists of channels and weights.        
        boolean updatesRequired = false;
        Set<String> keys = params.keySet();
        if (keys.contains("chanWeights")) {
            chanWeights.clear();
            chanWeights.addAll((List<Double>) params.get("chanWeights"));
            updatesRequired = true;
        }
        if (keys.contains("chanNames")) {
	    chanNames.clear();
            chanNames.addAll((List<String>) params.get("chanNames"));
            updatesRequired = true;
        }
        if ( updatesRequired ) {
            
            //Clear the internal arrays of Channels and weights and fill
            //them with the content of the temporary arrays
            channels.clear();
            channels.addAll(tmpChannels);
            channelsW.clear();
            channelsW.addAll(tmpChannelsW);
            
            
            LOG.log(Level.INFO, tmpChannelSelection);
            //Re-evaluate the sum of weights.
            sumWeights = 0.;
            for ( Double w : channelsW ) {
                sumWeights += w;                
            }
        }
    }

   /**
    *  Evaluate weighted average of specified channel values.
    *
    *  @return  double average (NaN if any input value is NaN)
    */
    @Override
    public double evaluateDerivedValue() {
        double sum = 0.;
        double sumW = sumWeights;
        for (int i = 0; i < channels.size(); i++) {
            double val = channels.get(i).getValue();
            if (val != Double.NaN || !ignoreNaN) {
                sum += (channelsW.get(i) * val);
	    } else {
                sumW -= channelsW.get(i);
	    }
        }
        return (sumW > 0. ? sum/sumW : Double.NaN);
    }


    //For tests
    List<Channel> getChannels() {
        return channels;
    }
    List<Double> getWeights() {
        return channelsW;
    }
    double getSumOfWeights() {
        return sumWeights;
    }
}
